"use strict";

var _interopRequireWildcard = require("@babel/runtime/helpers/interopRequireWildcard");

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

exports.revokeToken = revokeToken;
exports.getToken = getToken;
exports.getWithoutPrompt = getWithoutPrompt;
exports.getWithPopup = getWithPopup;
exports.getWithRedirect = getWithRedirect;
exports.parseFromUrl = parseFromUrl;
exports.decodeToken = decodeToken;
exports.renewToken = renewToken;
exports.getUserInfo = getUserInfo;
exports.verifyToken = verifyToken;
exports.handleOAuthResponse = handleOAuthResponse;
exports.prepareTokenParams = prepareTokenParams;

var _http = _interopRequireDefault(require("./http"));

var _util = require("./util");

var _oauthUtil = require("./oauthUtil");

var sdkCrypto = _interopRequireWildcard(require("./crypto"));

var _AuthSdkError = _interopRequireDefault(require("./errors/AuthSdkError"));

var _OAuthError = _interopRequireDefault(require("./errors/OAuthError"));

var _constants = require("./constants");

var _browserStorage = _interopRequireDefault(require("./browser/browserStorage"));

var _pkce = _interopRequireDefault(require("./pkce"));

var _types = require("./types");

/*!
 * Copyright (c) 2015-present, Okta, Inc. and/or its affiliates. All rights reserved.
 * The Okta software accompanied by this notice is provided pursuant to the Apache License, Version 2.0 (the "License.")
 *
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0.
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 *
 */

/* global window, document, btoa */

/* eslint-disable complexity, max-statements */
const cookies = _browserStorage.default.storage; // Only the access token can be revoked in SPA applications

function revokeToken(sdk, token) {
  return Promise.resolve().then(function () {
    if (!token || !token.accessToken) {
      throw new _AuthSdkError.default('A valid access token object is required');
    }

    var clientId = sdk.options.clientId;

    if (!clientId) {
      throw new _AuthSdkError.default('A clientId must be specified in the OktaAuth constructor to revoke a token');
    }

    var revokeUrl = (0, _oauthUtil.getOAuthUrls)(sdk).revokeUrl;
    var accessToken = token.accessToken;
    var args = (0, _util.toQueryParams)({
      // eslint-disable-next-line camelcase
      token_type_hint: 'access_token',
      token: accessToken
    }).slice(1);
    var creds = btoa(clientId);
    return _http.default.post(sdk, revokeUrl, args, {
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
        'Authorization': 'Basic ' + creds
      }
    });
  });
}

function decodeToken(token) {
  var jwt = token.split('.');
  var decodedToken;

  try {
    decodedToken = {
      header: JSON.parse((0, _util.base64UrlToString)(jwt[0])),
      payload: JSON.parse((0, _util.base64UrlToString)(jwt[1])),
      signature: jwt[2]
    };
  } catch (e) {
    throw new _AuthSdkError.default('Malformed token');
  }

  return decodedToken;
} // Verify the id token


function verifyToken(sdk, token, validationParams) {
  return Promise.resolve().then(function () {
    if (!token || !token.idToken) {
      throw new _AuthSdkError.default('Only idTokens may be verified');
    }

    var jwt = decodeToken(token.idToken);
    var validationOptions = {
      clientId: sdk.options.clientId,
      issuer: sdk.options.issuer,
      ignoreSignature: sdk.options.ignoreSignature
    };
    Object.assign(validationOptions, validationParams); // Standard claim validation

    (0, _oauthUtil.validateClaims)(sdk, jwt.payload, validationOptions); // If the browser doesn't support native crypto or we choose not
    // to verify the signature, bail early

    if (validationOptions.ignoreSignature == true || !sdk.features.isTokenVerifySupported()) {
      return token;
    }

    return (0, _oauthUtil.getKey)(sdk, token.issuer, jwt.header.kid).then(function (key) {
      return sdkCrypto.verifyToken(token.idToken, key);
    }).then(function (valid) {
      if (!valid) {
        throw new _AuthSdkError.default('The token signature is not valid');
      }

      if (validationParams && validationParams.accessToken && token.claims.at_hash) {
        return sdkCrypto.getOidcHash(validationParams.accessToken).then(hash => {
          if (hash !== token.claims.at_hash) {
            throw new _AuthSdkError.default('Token hash verification failed');
          }
        });
      }
    }).then(() => {
      return token;
    });
  });
}

function addPostMessageListener(sdk, timeout, state) {
  var responseHandler;
  var timeoutId;
  var msgReceivedOrTimeout = new Promise(function (resolve, reject) {
    responseHandler = function responseHandler(e) {
      if (!e.data || e.data.state !== state) {
        // A message not meant for us
        return;
      } // Configuration mismatch between saved token and current app instance
      // This may happen if apps with different issuers are running on the same host url
      // If they share the same storage key, they may read and write tokens in the same location.
      // Common when developing against http://localhost


      if (e.origin !== sdk.getIssuerOrigin()) {
        return reject(new _AuthSdkError.default('The request does not match client configuration'));
      }

      resolve(e.data);
    };

    (0, _oauthUtil.addListener)(window, 'message', responseHandler);
    timeoutId = setTimeout(function () {
      reject(new _AuthSdkError.default('OAuth flow timed out'));
    }, timeout || 120000);
  });
  return msgReceivedOrTimeout.finally(function () {
    clearTimeout(timeoutId);
    (0, _oauthUtil.removeListener)(window, 'message', responseHandler);
  });
}

function exchangeCodeForToken(sdk, oauthParams, authorizationCode, urls) {
  // PKCE authorization_code flow
  // Retrieve saved values and build oauthParams for call to /token
  var meta = _pkce.default.loadMeta(sdk);

  var getTokenParams = {
    clientId: oauthParams.clientId,
    authorizationCode: authorizationCode,
    codeVerifier: meta.codeVerifier,
    redirectUri: meta.redirectUri
  };
  return _pkce.default.getToken(sdk, getTokenParams, urls).then(function (res) {
    validateResponse(res, getTokenParams);
    return res;
  }).finally(function () {
    _pkce.default.clearMeta(sdk);
  });
}

function validateResponse(res, oauthParams) {
  if (res['error'] || res['error_description']) {
    throw new _OAuthError.default(res['error'], res['error_description']);
  }

  if (res.state !== oauthParams.state) {
    throw new _AuthSdkError.default('OAuth flow response state doesn\'t match request state');
  }
} // eslint-disable-next-line max-len


function handleOAuthResponse(sdk, tokenParams, res, urls) {
  urls = urls || {};
  var responseType = tokenParams.responseType;

  if (!Array.isArray(responseType)) {
    responseType = [responseType];
  }

  var scopes = (0, _util.clone)(tokenParams.scopes);
  var clientId = tokenParams.clientId || sdk.options.clientId;
  var pkce = sdk.options.pkce !== false;
  return Promise.resolve().then(function () {
    validateResponse(res, tokenParams); // PKCE flow
    // We do not support "hybrid" scenarios where the response includes both a code and a token.
    // If the response contains a code it is used immediately to obtain new tokens.

    if (res.code && pkce) {
      // responseType is not sent to the token endpoint.
      // We populate this array to validate the response below
      responseType = ['token']; // an accessToken will always be returned

      if (scopes.indexOf('openid') !== -1) {
        responseType.push('id_token'); // an idToken will be returned if "openid" is in the scopes
      }

      return exchangeCodeForToken(sdk, tokenParams, res.code, urls);
    }

    return res;
  }).then(function (res) {
    var tokenDict = {};
    var expiresIn = res.expires_in;
    var tokenType = res.token_type;
    var accessToken = res.access_token;
    var idToken = res.id_token;

    if (accessToken) {
      tokenDict.accessToken = {
        value: accessToken,
        accessToken: accessToken,
        expiresAt: Number(expiresIn) + Math.floor(Date.now() / 1000),
        tokenType: tokenType,
        scopes: scopes,
        authorizeUrl: urls.authorizeUrl,
        userinfoUrl: urls.userinfoUrl
      };
    }

    if (idToken) {
      var jwt = sdk.token.decode(idToken);
      var idTokenObj = {
        value: idToken,
        idToken: idToken,
        claims: jwt.payload,
        expiresAt: jwt.payload.exp,
        scopes: scopes,
        authorizeUrl: urls.authorizeUrl,
        issuer: urls.issuer,
        clientId: clientId
      };
      var validationParams = {
        clientId: clientId,
        issuer: urls.issuer,
        nonce: tokenParams.nonce,
        accessToken: accessToken
      };

      if (tokenParams.ignoreSignature !== undefined) {
        validationParams.ignoreSignature = tokenParams.ignoreSignature;
      }

      return verifyToken(sdk, idTokenObj, validationParams).then(function () {
        tokenDict.idToken = idTokenObj;
        return tokenDict;
      });
    }

    return tokenDict;
  }).then(function (tokenDict) {
    // Validate received tokens against requested response types 
    if (responseType.indexOf('token') !== -1 && !tokenDict.accessToken) {
      // eslint-disable-next-line max-len
      throw new _AuthSdkError.default('Unable to parse OAuth flow response: response type "token" was requested but "access_token" was not returned.');
    }

    if (responseType.indexOf('id_token') !== -1 && !tokenDict.idToken) {
      // eslint-disable-next-line max-len
      throw new _AuthSdkError.default('Unable to parse OAuth flow response: response type "id_token" was requested but "id_token" was not returned.');
    }

    return {
      tokens: tokenDict,
      state: res.state,
      code: res.code
    };
  });
}

function getDefaultTokenParams(sdk) {
  return {
    pkce: sdk.options.pkce,
    clientId: sdk.options.clientId,
    redirectUri: sdk.options.redirectUri || window.location.href,
    responseType: ['token', 'id_token'],
    responseMode: sdk.options.responseMode,
    state: (0, _oauthUtil.generateState)(),
    nonce: (0, _oauthUtil.generateNonce)(),
    scopes: ['openid', 'email'],
    ignoreSignature: sdk.options.ignoreSignature
  };
}

function convertTokenParamsToOAuthParams(tokenParams) {
  // Quick validation
  if (!tokenParams.clientId) {
    throw new _AuthSdkError.default('A clientId must be specified in the OktaAuth constructor to get a token');
  }

  if ((0, _util.isString)(tokenParams.responseType) && tokenParams.responseType.indexOf(' ') !== -1) {
    throw new _AuthSdkError.default('Multiple OAuth responseTypes must be defined as an array');
  } // Convert our params to their actual OAuth equivalents


  var oauthParams = {
    'client_id': tokenParams.clientId,
    'code_challenge': tokenParams.codeChallenge,
    'code_challenge_method': tokenParams.codeChallengeMethod,
    'display': tokenParams.display,
    'idp': tokenParams.idp,
    'idp_scope': tokenParams.idpScope,
    'login_hint': tokenParams.loginHint,
    'max_age': tokenParams.maxAge,
    'nonce': tokenParams.nonce,
    'prompt': tokenParams.prompt,
    'redirect_uri': tokenParams.redirectUri,
    'response_mode': tokenParams.responseMode,
    'response_type': tokenParams.responseType,
    'sessionToken': tokenParams.sessionToken,
    'state': tokenParams.state
  };
  oauthParams = (0, _util.removeNils)(oauthParams);
  ['idp_scope', 'response_type'].forEach(function (mayBeArray) {
    if (Array.isArray(oauthParams[mayBeArray])) {
      oauthParams[mayBeArray] = oauthParams[mayBeArray].join(' ');
    }
  });

  if (tokenParams.responseType.indexOf('id_token') !== -1 && tokenParams.scopes.indexOf('openid') === -1) {
    throw new _AuthSdkError.default('openid scope must be specified in the scopes argument when requesting an id_token');
  } else {
    oauthParams.scope = tokenParams.scopes.join(' ');
  }

  return oauthParams;
}

function buildAuthorizeParams(tokenParams) {
  var oauthQueryParams = convertTokenParamsToOAuthParams(tokenParams);
  return (0, _util.toQueryParams)(oauthQueryParams);
}
/*
 * Retrieve an idToken from an Okta or a third party idp
 *
 * Two main flows:
 *
 *  1) Exchange a sessionToken for a token
 *
 *    Required:
 *      clientId: passed via the OktaAuth constructor or into getToken
 *      sessionToken: 'yourtoken'
 *
 *    Optional:
 *      redirectUri: defaults to window.location.href
 *      scopes: defaults to ['openid', 'email']
 *
 *    Forced:
 *      prompt: 'none'
 *      responseMode: 'okta_post_message'
 *      display: undefined
 *
 *  2) Get a token from an idp
 *
 *    Required:
 *      clientId: passed via the OktaAuth constructor or into getToken
 *
 *    Optional:
 *      redirectUri: defaults to window.location.href
 *      scopes: defaults to ['openid', 'email']
 *      idp: defaults to Okta as an idp
 *      prompt: no default. Pass 'none' to throw an error if user is not signed in
 *
 *    Forced:
 *      display: 'popup'
 *
 *  Only common optional params shown. Any OAuth parameters not explicitly forced are available to override
 *
 * @param {Object} oauthOptions
 * @param {String} [oauthOptions.clientId] ID of this client
 * @param {String} [oauthOptions.redirectUri] URI that the iframe or popup will go to once authenticated
 * @param {String[]} [oauthOptions.scopes] OAuth 2.0 scopes to request (openid must be specified)
 * @param {String} [oauthOptions.idp] ID of an external IdP to use for user authentication
 * @param {String} [oauthOptions.sessionToken] Bootstrap Session Token returned by the Okta Authentication API
 * @param {String} [oauthOptions.prompt] Determines whether the Okta login will be displayed on failure.
 *                                       Use 'none' to prevent this behavior
 *
 * @param {Object} options
 * @param {Integer} [options.timeout] Time in ms before the flow is automatically terminated. Defaults to 120000
 * @param {String} [options.popupTitle] Title dispayed in the popup.
 *                                      Defaults to 'External Identity Provider User Authentication'
 */


function getToken(sdk, options) {
  if (arguments.length > 2) {
    return Promise.reject(new _AuthSdkError.default('As of version 3.0, "getToken" takes only a single set of options'));
  }

  options = options || {};
  return prepareTokenParams(sdk, options).then(function (tokenParams) {
    // Start overriding any options that don't make sense
    var sessionTokenOverrides = {
      prompt: 'none',
      responseMode: 'okta_post_message',
      display: null
    };
    var idpOverrides = {
      display: 'popup'
    };

    if (options.sessionToken) {
      Object.assign(tokenParams, sessionTokenOverrides);
    } else if (options.idp) {
      Object.assign(tokenParams, idpOverrides);
    } // Use the query params to build the authorize url


    var requestUrl, endpoint, urls; // Get authorizeUrl and issuer

    urls = (0, _oauthUtil.getOAuthUrls)(sdk, tokenParams);
    endpoint = options.codeVerifier ? urls.tokenUrl : urls.authorizeUrl;
    requestUrl = endpoint + buildAuthorizeParams(tokenParams); // Determine the flow type

    var flowType;

    if (tokenParams.sessionToken || tokenParams.display === null) {
      flowType = 'IFRAME';
    } else if (tokenParams.display === 'popup') {
      flowType = 'POPUP';
    } else {
      flowType = 'IMPLICIT';
    } // Execute the flow type


    switch (flowType) {
      case 'IFRAME':
        var iframePromise = addPostMessageListener(sdk, options.timeout, tokenParams.state);
        var iframeEl = (0, _oauthUtil.loadFrame)(requestUrl);
        return iframePromise.then(function (res) {
          return handleOAuthResponse(sdk, tokenParams, res, urls);
        }).finally(function () {
          if (document.body.contains(iframeEl)) {
            iframeEl.parentElement.removeChild(iframeEl);
          }
        });

      case 'POPUP':
        var oauthPromise; // resolves with OAuth response
        // Add listener on postMessage before window creation, so
        // postMessage isn't triggered before we're listening

        if (tokenParams.responseMode === 'okta_post_message') {
          if (!sdk.features.isPopupPostMessageSupported()) {
            throw new _AuthSdkError.default('This browser doesn\'t have full postMessage support');
          }

          oauthPromise = addPostMessageListener(sdk, options.timeout, tokenParams.state);
        } // Create the window


        var windowOptions = {
          popupTitle: options.popupTitle
        };
        var windowEl = (0, _oauthUtil.loadPopup)(requestUrl, windowOptions); // The popup may be closed without receiving an OAuth response. Setup a poller to monitor the window.

        var popupPromise = new Promise(function (resolve, reject) {
          var closePoller = setInterval(function () {
            if (!windowEl || windowEl.closed) {
              clearInterval(closePoller);
              reject(new _AuthSdkError.default('Unable to parse OAuth flow response'));
            }
          }, 100); // Proxy the OAuth promise results

          oauthPromise.then(function (res) {
            clearInterval(closePoller);
            resolve(res);
          }).catch(function (err) {
            clearInterval(closePoller);
            reject(err);
          });
        });
        return popupPromise.then(function (res) {
          return handleOAuthResponse(sdk, tokenParams, res, urls);
        }).finally(function () {
          if (windowEl && !windowEl.closed) {
            windowEl.close();
          }
        });

      default:
        throw new _AuthSdkError.default('The full page redirect flow is not supported');
    }
  }).catch(e => {
    if (sdk.options.pkce) {
      _pkce.default.clearMeta(sdk);
    }

    throw e;
  });
}

function getWithoutPrompt(sdk, options) {
  if (arguments.length > 2) {
    return Promise.reject(new _AuthSdkError.default('As of version 3.0, "getWithoutPrompt" takes only a single set of options'));
  }

  options = (0, _util.clone)(options) || {};
  Object.assign(options, {
    prompt: 'none',
    responseMode: 'okta_post_message',
    display: null
  });
  return getToken(sdk, options);
}

function getWithPopup(sdk, options) {
  if (arguments.length > 2) {
    return Promise.reject(new _AuthSdkError.default('As of version 3.0, "getWithPopup" takes only a single set of options'));
  }

  options = (0, _util.clone)(options) || {};
  Object.assign(options, {
    display: 'popup',
    responseMode: 'okta_post_message'
  });
  return getToken(sdk, options);
}

function prepareTokenParams(sdk, options) {
  if ((0, _oauthUtil.isLoginRedirect)(sdk)) {
    return Promise.reject(new _AuthSdkError.default('The app should not attempt to call getToken on callback. ' + 'Authorize flow is already in process. Use parseFromUrl() to receive tokens.'));
  } // clone and prepare options


  options = (0, _util.clone)(options) || {}; // build params using defaults + options

  var tokenParams = getDefaultTokenParams(sdk);
  Object.assign(tokenParams, options);

  if (tokenParams.pkce === false) {
    return Promise.resolve(tokenParams);
  } // PKCE flow


  if (!sdk.features.isPKCESupported()) {
    var errorMessage = 'PKCE requires a modern browser with encryption support running in a secure context.';

    if (!sdk.features.isHTTPS()) {
      // eslint-disable-next-line max-len
      errorMessage += '\nThe current page is not being served with HTTPS protocol. PKCE requires secure HTTPS protocol.';
    }

    if (!sdk.features.hasTextEncoder()) {
      // eslint-disable-next-line max-len
      errorMessage += '\n"TextEncoder" is not defined. To use PKCE, you may need to include a polyfill/shim for this browser.';
    }

    return Promise.reject(new _AuthSdkError.default(errorMessage));
  } // set default code challenge method, if none provided


  if (!tokenParams.codeChallengeMethod) {
    tokenParams.codeChallengeMethod = _pkce.default.DEFAULT_CODE_CHALLENGE_METHOD;
  } // responseType is forced


  tokenParams.responseType = 'code';
  return (0, _oauthUtil.getWellKnown)(sdk, null).then(function (res) {
    var methods = res['code_challenge_methods_supported'] || [];

    if (methods.indexOf(tokenParams.codeChallengeMethod) === -1) {
      throw new _AuthSdkError.default('Invalid code_challenge_method');
    }
  }).then(function () {
    // PKCE authorization_code flow
    var codeVerifier = _pkce.default.generateVerifier(tokenParams.codeVerifier); // We will need these values after redirect when we call /token


    var meta = {
      codeVerifier: codeVerifier,
      redirectUri: tokenParams.redirectUri
    };

    _pkce.default.saveMeta(sdk, meta);

    return _pkce.default.computeChallenge(codeVerifier);
  }).then(function (codeChallenge) {
    // Clone/copy the params. Set codeChallenge
    var clonedParams = (0, _util.clone)(tokenParams) || {};
    Object.assign(clonedParams, tokenParams, {
      codeChallenge: codeChallenge
    });
    return clonedParams;
  });
}

function getWithRedirect(sdk, options) {
  if (arguments.length > 2) {
    return Promise.reject(new _AuthSdkError.default('As of version 3.0, "getWithRedirect" takes only a single set of options'));
  }

  options = (0, _util.clone)(options) || {};
  return prepareTokenParams(sdk, options).then(function (tokenParams) {
    var urls = (0, _oauthUtil.getOAuthUrls)(sdk, options);
    var requestUrl = urls.authorizeUrl + buildAuthorizeParams(tokenParams); // Set session cookie to store the oauthParams

    const {
      responseType,
      state,
      nonce,
      scopes,
      clientId,
      ignoreSignature
    } = tokenParams;
    cookies.set(_constants.REDIRECT_OAUTH_PARAMS_COOKIE_NAME, JSON.stringify({
      responseType,
      state,
      nonce,
      scopes,
      clientId,
      urls: urls,
      ignoreSignature
    }), null, sdk.options.cookies); // Set nonce cookie for servers to validate nonce in id_token

    cookies.set(_constants.REDIRECT_NONCE_COOKIE_NAME, tokenParams.nonce, null, sdk.options.cookies); // Set state cookie for servers to validate state

    cookies.set(_constants.REDIRECT_STATE_COOKIE_NAME, tokenParams.state, null, sdk.options.cookies);

    sdk.token.getWithRedirect._setLocation(requestUrl);
  });
}

function renewToken(sdk, token) {
  if (!(0, _types.isToken)(token)) {
    return Promise.reject(new _AuthSdkError.default('Renew must be passed a token with ' + 'an array of scopes and an accessToken or idToken'));
  }

  var responseType;

  if (sdk.options.pkce) {
    responseType = 'code';
  } else if ((0, _types.isAccessToken)(token)) {
    responseType = 'token';
  } else {
    responseType = 'id_token';
  }

  const {
    scopes,
    authorizeUrl,
    userinfoUrl,
    issuer
  } = token;
  return getWithoutPrompt(sdk, {
    responseType,
    scopes,
    authorizeUrl,
    userinfoUrl,
    issuer
  }).then(function (res) {
    // Multiple tokens may have come back. Return only the token which was requested.
    var tokens = res.tokens;
    return (0, _types.isIDToken)(token) ? tokens.idToken : tokens.accessToken;
  });
}

function removeHash(sdk) {
  var nativeHistory = sdk.token.parseFromUrl._getHistory();

  var nativeDoc = sdk.token.parseFromUrl._getDocument();

  var nativeLoc = sdk.token.parseFromUrl._getLocation();

  if (nativeHistory && nativeHistory.replaceState) {
    nativeHistory.replaceState(null, nativeDoc.title, nativeLoc.pathname + nativeLoc.search);
  } else {
    nativeLoc.hash = '';
  }
}

function removeSearch(sdk) {
  var nativeHistory = sdk.token.parseFromUrl._getHistory();

  var nativeDoc = sdk.token.parseFromUrl._getDocument();

  var nativeLoc = sdk.token.parseFromUrl._getLocation();

  if (nativeHistory && nativeHistory.replaceState) {
    nativeHistory.replaceState(null, nativeDoc.title, nativeLoc.pathname + nativeLoc.hash);
  } else {
    nativeLoc.search = '';
  }
}

function parseFromUrl(sdk, options) {
  options = options || {};

  if ((0, _util.isString)(options)) {
    options = {
      url: options
    };
  } else {
    options = options;
  } // https://openid.net/specs/openid-connect-core-1_0.html#Authentication


  var defaultResponseMode = sdk.options.pkce ? 'query' : 'fragment';
  var url = options.url;
  var responseMode = options.responseMode || sdk.options.responseMode || defaultResponseMode;

  var nativeLoc = sdk.token.parseFromUrl._getLocation();

  var paramStr;

  if (responseMode === 'query') {
    paramStr = url ? url.substring(url.indexOf('?')) : nativeLoc.search;
  } else {
    paramStr = url ? url.substring(url.indexOf('#')) : nativeLoc.hash;
  }

  if (!paramStr) {
    return Promise.reject(new _AuthSdkError.default('Unable to parse a token from the url'));
  }

  var oauthParamsCookie = cookies.get(_constants.REDIRECT_OAUTH_PARAMS_COOKIE_NAME);

  if (!oauthParamsCookie) {
    return Promise.reject(new _AuthSdkError.default('Unable to retrieve OAuth redirect params cookie'));
  }

  try {
    var oauthParams = JSON.parse(oauthParamsCookie);
    var urls = oauthParams.urls;
    delete oauthParams.urls;
    cookies.delete(_constants.REDIRECT_OAUTH_PARAMS_COOKIE_NAME);
  } catch (e) {
    return Promise.reject(new _AuthSdkError.default('Unable to parse the ' + _constants.REDIRECT_OAUTH_PARAMS_COOKIE_NAME + ' cookie: ' + e.message));
  }

  return Promise.resolve((0, _oauthUtil.urlParamsToObject)(paramStr)).then(function (res) {
    if (!url) {
      // Clean hash or search from the url
      responseMode === 'query' ? removeSearch(sdk) : removeHash(sdk);
    }

    return handleOAuthResponse(sdk, oauthParams, res, urls);
  });
}

async function getUserInfo(sdk, accessTokenObject, idTokenObject) {
  // If token objects were not passed, attempt to read from the TokenManager
  if (!accessTokenObject) {
    accessTokenObject = await sdk.tokenManager.get(_constants.ACCESS_TOKEN_STORAGE_KEY);
  }

  if (!idTokenObject) {
    idTokenObject = await sdk.tokenManager.get(_constants.ID_TOKEN_STORAGE_KEY);
  }

  if (!accessTokenObject || !(0, _types.isToken)(accessTokenObject) && !accessTokenObject.accessToken && !accessTokenObject.userinfoUrl) {
    return Promise.reject(new _AuthSdkError.default('getUserInfo requires an access token object'));
  }

  if (!idTokenObject || !(0, _types.isToken)(idTokenObject) && !idTokenObject.idToken) {
    return Promise.reject(new _AuthSdkError.default('getUserInfo requires an ID token object'));
  }

  return _http.default.httpRequest(sdk, {
    url: accessTokenObject.userinfoUrl,
    method: 'GET',
    accessToken: accessTokenObject.accessToken
  }).then(userInfo => {
    // Only return the userinfo response if subjects match to mitigate token substitution attacks
    if (userInfo.sub === idTokenObject.claims.sub) {
      return userInfo;
    }

    return Promise.reject(new _AuthSdkError.default('getUserInfo request was rejected due to token mismatch'));
  }).catch(function (err) {
    if (err.xhr && (err.xhr.status === 401 || err.xhr.status === 403)) {
      var authenticateHeader;

      if (err.xhr.headers && (0, _util.isFunction)(err.xhr.headers.get) && err.xhr.headers.get('WWW-Authenticate')) {
        authenticateHeader = err.xhr.headers.get('WWW-Authenticate');
      } else if ((0, _util.isFunction)(err.xhr.getResponseHeader)) {
        authenticateHeader = err.xhr.getResponseHeader('WWW-Authenticate');
      }

      if (authenticateHeader) {
        var errorMatches = authenticateHeader.match(/error="(.*?)"/) || [];
        var errorDescriptionMatches = authenticateHeader.match(/error_description="(.*?)"/) || [];
        var error = errorMatches[1];
        var errorDescription = errorDescriptionMatches[1];

        if (error && errorDescription) {
          err = new _OAuthError.default(error, errorDescription);
        }
      }
    }

    throw err;
  });
}