//! Provides access to the [CODATA recommended 2018 values for physical constants][codata].
//!
//! # Examples
//!
//! ```
//! use physical_constants;
//!
//! let epsilon_0 = physical_constants::VACUUM_ELECTRIC_PERMITTIVITY;
//! let mu_0 = physical_constants::VACUUM_MAG_PERMEABILITY;
//!
//! println!("speed of massless particles: {} m/s", 1f64/(epsilon_0*mu_0).sqrt());
//! println!("impedance of free space: {} Ω", (mu_0/epsilon_0).sqrt());
//! ```
//!
//! [codata]: http://physics.nist.gov/cuu/Constants/
//! [codata 2018]: http://physics.nist.gov/cuu/Constants/Table/allascii.txt

// Include the physical constants, which are generated by the build script.
include!(concat!(env!("OUT_DIR"), "/constants.rs"));

#[cfg(test)]
mod tests {
    use super::*;

    // Testing for exact equality of floating-point numbers is actually appropriate for once.

    #[test]
    fn test_exact_quantity() {
        // An exact quantity whose decimal expansion ends with "..." in the table
        assert_eq!(SPEED_OF_LIGHT_IN_VACUUM, 299792458f64);
    }

    #[test]
    fn test_no_exponent() {
        // A value that has no exponent in the listing
        assert_eq!(ATOMIC_MASS_CONSTANT_ENERGY_EQUIVALENT_IN_MEV, 931.49410242);
    }

    #[test]
    fn test_positive_exponent() {
        assert_eq!(BOLTZMANN_CONSTANT_IN_HZ_PER_K, 2.083661912e10);
    }

    #[test]
    fn test_negative_exponent() {
        assert_eq!(CLASSICAL_ELECTRON_RADIUS, 2.8179403262e-15);
    }

    #[test]
    fn test_negative_value() {
        assert_eq!(ELECTRON_CHARGE_TO_MASS_QUOTIENT, -1.75882001076e11);
    }

    #[test]
    fn test_dimensionless_value() {
        assert_eq!(PROTON_ELECTRON_MASS_RATIO, 1836.15267343);
    }

    #[test]
    fn test_first_quantity() {
        // The first quantity listed in the table
        assert_eq!(ALPHA_PARTICLE_ELECTRON_MASS_RATIO, 7294.29954142);
    }

    #[test]
    fn test_last_quantity() {
        // The last quantity listed in the table
        assert_eq!(W_TO_Z_MASS_RATIO, 0.88153);
    }
}
