#![allow(unused_doc_comments)]

use const_array_attrs::sorted;
use const_frac::Frac;
use crate:: {
    DynDim,
    unit::Conv,
    predefined::dim::*,
};

const ZERO: Frac = Frac::from_int(0);
const ONE: Frac = Frac::from_int(1);
const RAD2BASE: Conv = Conv(Frac::from_ratio(1, 2, 0).div(Frac::from_exp_pi(1)), ZERO);
//const RAD2BASE: Conv = Conv(Frac::new(false, 1, 0xEC58_DFA7_4641_AF52_AD0D_16E7_7D57_6623, 37, 38), ZERO);
//const SR2BASE: Conv = Conv(Frac::new(false, 1, 1, 2, 0, 1), ZERO);
const SR2BASE: Conv = Conv(Frac::from_ratio(1, 4, 0).div(Frac::from_exp_pi(1)), ZERO);
const INCH2METER: Conv = Conv(Frac::from_ratio(254, 1, -4), ZERO);
const FOOT2INCH: Conv = Conv(Frac::from_int(12), ZERO);
const YARD2FOOT: Conv = Conv(Frac::from_int(3), ZERO);
const POUND2GRAM: Conv = Conv(Frac::from_ratio(453_59237, 1, -5), ZERO);
const MINUTE2SECOND: Conv = Conv(Frac::from_int(60), ZERO);
const HOUR2MINUTE: Conv = Conv(Frac::from_int(60), ZERO);
const FIVE_PAR_NINE: Frac = Frac::from_ratio(5, 9, 0);
const RANKINE2KELVIN: Conv = Conv(FIVE_PAR_NINE, ZERO);
const FAHRENHEIT2CELSIUS: Conv = Conv(FIVE_PAR_NINE, Frac::from_ratio(32 * 5, 9, 0));
const CELSIUS2KELVIN: Conv = Conv(ONE, Frac::from_ratio(273_15, 1, -2));
const NEWTON2BASE: Conv = Conv(Frac::from_exp10(3), ZERO);
const GRAM_FORCE2NEWTON: Conv = Conv(Frac::from_ratio(9_80665, 1, -8), ZERO);
const JOULE2BASE: Conv = Conv(Frac::from_exp10(3), ZERO);
/// International steam table calorie
const CALORIE2JOULE: Conv = Conv(Frac::from_ratio(4_1868, 1, -4), ZERO);
const BTU2CALORIE: Conv = POUND2GRAM.then(RANKINE2KELVIN);
const WATT2BASE: Conv = Conv(Frac::from_exp10(3), ZERO);
/// Area
const ACRE2FT2: Conv = Conv(Frac::from_ratio(4 * 40 * 165 * 165, 100, 0), ZERO);
/// pressure
const AT2BASE: Conv = GRAM_FORCE2NEWTON.mul(NEWTON2BASE).mul(Conv(Frac::from_exp10(7), ZERO));
const PSI2BASE: Conv = POUND2GRAM.mul(GRAM_FORCE2NEWTON).mul(NEWTON2BASE).div(INCH2METER.mul(INCH2METER));

#[sorted]
pub const DEFAULT_UNIT_DEF: [(&str, (Conv, DynDim)); 216] = [
    // dimensionless

    /// ‐ hyphen
    ("\u{2010}", (Conv(ONE, ZERO), DynDim::new(Dimensionless::CODE))),
    /// ‒ figure dash
    ("\u{2012}", (Conv(ONE, ZERO), DynDim::new(Dimensionless::CODE))),
    /// – en dash
    ("\u{2013}", (Conv(ONE, ZERO), DynDim::new(Dimensionless::CODE))),
    /// — EM DASH
    ("\u{2014}", (Conv(ONE, ZERO), DynDim::new(Dimensionless::CODE))),
    /// ― HORIZONTAL BAR
    ("\u{2015}", (Conv(ONE, ZERO), DynDim::new(Dimensionless::CODE))),
    /// ¹
    ("\u{00B8}", (Conv(ONE, ZERO), DynDim::new(Dimensionless::CODE))),
    ("%", (Conv(Frac::from_exp10(-2), ZERO), DynDim::new(Dimensionless::CODE))),
    /// ㌫
    ("\u{332B}", (Conv(Frac::from_exp10(-2), ZERO), DynDim::new(Dimensionless::CODE))),
    /// ‰
    ("\u{2030}", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(Dimensionless::CODE))),
    /// ‱
    ("\u{2031}", (Conv(Frac::from_exp10(-4), ZERO), DynDim::new(Dimensionless::CODE))),
    ("ppm", (Conv(Frac::from_exp10(-6), ZERO), DynDim::new(Dimensionless::CODE))),
    /// ㏙
    ("\u{33D9}", (Conv(Frac::from_exp10(-6), ZERO), DynDim::new(Dimensionless::CODE))),
    ("ppb", (Conv(Frac::from_exp10(-9), ZERO), DynDim::new(Dimensionless::CODE))),
    ("ppt", (Conv(Frac::from_exp10(-12), ZERO), DynDim::new(Dimensionless::CODE))),
    ("ppq", (Conv(Frac::from_exp10(-15), ZERO), DynDim::new(Dimensionless::CODE))),
    /// °
    ("\u{00B0}", (
        Conv(Frac::from_ratio(1, 360, 0), ZERO),
        DynDim::new(Dimensionless::CODE))
    ),
    /// ˚
    ("\u{02DA}", (
        Conv(Frac::from_ratio(1, 360, 0), ZERO),
        DynDim::new(Dimensionless::CODE))
    ),
    ("rad", (RAD2BASE, DynDim::new(Dimensionless::CODE))),
    /// ㎭
    ("\u{33AD}", (RAD2BASE, DynDim::new(Dimensionless::CODE))),
    ("sr", (SR2BASE, DynDim::new(Dimensionless::CODE))),
    /// ㏛
    ("\u{33DB}", (SR2BASE, DynDim::new(Dimensionless::CODE))),
    // L
    ("m", (Conv(ONE, ZERO), DynDim::new(Length::CODE))),
    /// ㍍
    ("\u{334D}", (Conv(ONE, ZERO), DynDim::new(Length::CODE))),
    /// ㌖
    ("\u{3316}", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(Length::CODE))),
    /// ㍷
    ("\u{3377}", (Conv(Frac::from_exp10(-1), ZERO), DynDim::new(Length::CODE))),
    /// ㎙
    ("\u{3399}", (Conv(Frac::from_exp10(-12), ZERO), DynDim::new(Length::CODE))),
    /// ㎚
    ("\u{339A}", (Conv(Frac::from_exp10(-9), ZERO), DynDim::new(Length::CODE))),
    /// ㎛
    ("\u{339B}", (Conv(Frac::from_exp10(-6), ZERO), DynDim::new(Length::CODE))),
    /// ㎜
    ("\u{339C}", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(Length::CODE))),
    /// ㎝
    ("\u{339D}", (Conv(Frac::from_exp10(-2), ZERO), DynDim::new(Length::CODE))),
    /// ㎞
    ("\u{339E}", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(Length::CODE))),
    ("in.", (INCH2METER, DynDim::new(Length::CODE))),
    /// ㌅
    ("\u{3305}", (INCH2METER, DynDim::new(Length::CODE))),
    /// ㏌
    ("\u{33CC}", (INCH2METER, DynDim::new(Length::CODE))),
    ("ft.", (FOOT2INCH.then(INCH2METER), DynDim::new(Length::CODE))),
    /// ㌳
    ("\u{3333}", (FOOT2INCH.then(INCH2METER), DynDim::new(Length::CODE))),
    ("yd.", (YARD2FOOT.then(FOOT2INCH).then(INCH2METER), DynDim::new(Length::CODE))),
    /// ㍎
    ("\u{334E}", (YARD2FOOT.then(FOOT2INCH).then(INCH2METER), DynDim::new(Length::CODE))),
    /// ㍏
    ("\u{334F}", (YARD2FOOT.then(FOOT2INCH).then(INCH2METER), DynDim::new(Length::CODE))),
    /// Å
    ("\u{00C5}", (Conv(Frac::from_exp10(-10), ZERO), DynDim::new(Length::CODE))),
    /// Å
    ("\u{212B}", (Conv(Frac::from_exp10(-10), ZERO), DynDim::new(Length::CODE))),
    /// ㌋
    ("\u{330B}", (Conv(Frac::from_int(1852), ZERO), DynDim::new(Length::CODE))),
    ("mile", (
        Conv(Frac::from_int(1760), ZERO).then(YARD2FOOT).then(FOOT2INCH).then(INCH2METER),
        DynDim::new(Length::CODE)
    )),
    /// ㍄
    ("\u{3344}", (
        Conv(Frac::from_int(1760), ZERO).then(YARD2FOOT).then(FOOT2INCH).then(INCH2METER),
        DynDim::new(Length::CODE)
    )),
    /// ㏕
    ("\u{33D5}", (
        Conv(Frac::from_int(1760), ZERO).then(YARD2FOOT).then(FOOT2INCH).then(INCH2METER),
        DynDim::new(Length::CODE)
    )),
    ("au", (Conv(Frac::from_int(149_597_870_700), ZERO), DynDim::new(Length::CODE))),
    /// ㍳
    ("\u{3373}", (Conv(Frac::from_int(149_597_870_700), ZERO), DynDim::new(Length::CODE))),
    ("pc", (
        Conv(Frac::from_ratio(3_085_677_581, 1, 7), ZERO),
        DynDim::new(Length::CODE)
    )),
    /// ㍶
    ("\u{3376}", (
        Conv(Frac::from_ratio(3_085_677_581, 1, 7), ZERO),
        DynDim::new(Length::CODE)
    )),
    // M
    ("g", (Conv(ONE, ZERO), DynDim::new(Mass::CODE))),
    /// ㌘
    ("\u{3318}", (Conv(ONE, ZERO), DynDim::new(Mass::CODE))),
    /// ㌕
    ("\u{3315}", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(Mass::CODE))),
    /// ㎍
    ("\u{338D}", (Conv(Frac::from_exp10(-6), ZERO), DynDim::new(Mass::CODE))),
    /// ㎎
    ("\u{338E}", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(Mass::CODE))),
    /// ㎏
    ("\u{338F}", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(Mass::CODE))),
    ("t", (Conv(Frac::from_exp10(6), ZERO), DynDim::new(Mass::CODE))),
    /// ㌙
    ("\u{3319}", (Conv(Frac::from_exp10(6), ZERO), DynDim::new(Mass::CODE))),
    /// ㌧
    ("\u{3327}", (Conv(Frac::from_exp10(6), ZERO), DynDim::new(Mass::CODE))),
    /// ㍌
    ("\u{334C}", (Conv(Frac::from_exp10(9), ZERO), DynDim::new(Mass::CODE))),
    ("Da", (
        Conv(Frac::from_ratio(1_660_539_066_60, 1, -35), ZERO),
        DynDim::new(Mass::CODE)
    )),
    ("lb", (POUND2GRAM, DynDim::new(Mass::CODE))),
    /// ㍀
    ("\u{3340}", (POUND2GRAM, DynDim::new(Mass::CODE))),
    ("oz", (
        Conv(Frac::from_ratio(1, 16, 0), ZERO).mul(POUND2GRAM),
        DynDim::new(Mass::CODE))
    ),
    /// ㌉
    ("\u{3309}", (
        Conv(Frac::from_ratio(1, 16, 0), ZERO).mul(POUND2GRAM),
        DynDim::new(Mass::CODE))
    ),
    /// ㌌
    ("\u{330C}", (Conv(Frac::from_ratio(1, 5, 0), ZERO), DynDim::new(Mass::CODE))),
    // T
    ("s", (Conv(ONE, ZERO), DynDim::new(Time::CODE))),
    /// ㎰
    ("\u{33B0}", (Conv(Frac::from_exp10(-12), ZERO), DynDim::new(Time::CODE))),
    /// ㎱
    ("\u{33B1}", (Conv(Frac::from_exp10(-9), ZERO), DynDim::new(Time::CODE))),
    /// ㎲
    ("\u{33B2}", (Conv(Frac::from_exp10(-6), ZERO), DynDim::new(Time::CODE))),
    /// ㎳
    ("\u{33B3}", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(Time::CODE))),
    // θ
    ("K", (Conv(ONE, ZERO), DynDim::new(Temperature::CODE))),
    /// K
    ("\u{212A}", (Conv(ONE, ZERO), DynDim::new(Temperature::CODE))),
    /// °R
    ("\u{00B0}R", (RANKINE2KELVIN, DynDim::new(Temperature::CODE))),
    /// °F
    ("\u{00B0}F", (FAHRENHEIT2CELSIUS.then(CELSIUS2KELVIN), DynDim::new(Temperature::CODE))),
    /// ℉
    ("\u{2109}", (FAHRENHEIT2CELSIUS.then(CELSIUS2KELVIN), DynDim::new(Temperature::CODE))),
    /// °C
    ("\u{00B0}C", (CELSIUS2KELVIN, DynDim::new(Temperature::CODE))),
    /// ℃
    ("\u{2103}", (CELSIUS2KELVIN, DynDim::new(Temperature::CODE))),
    // N
    ("mol", (Conv(ONE, ZERO), DynDim::new(Amount::CODE))),
    /// ㏖
    ("\u{33D6}", (Conv(ONE, ZERO), DynDim::new(Amount::CODE))),
    // I
    ("A", (Conv(ONE, ZERO), DynDim::new(Current::CODE))),
    /// ㌂
    ("\u{3302}", (Conv(ONE, ZERO), DynDim::new(Current::CODE))),
    /// ㎀
    ("\u{3380}", (Conv(Frac::from_exp10(-12), ZERO), DynDim::new(Current::CODE))),
    /// ㎁
    ("\u{3381}", (Conv(Frac::from_exp10(-9), ZERO), DynDim::new(Current::CODE))),
    /// ㎂
    ("\u{3382}", (Conv(Frac::from_exp10(-6), ZERO), DynDim::new(Current::CODE))),
    /// ㎃
    ("\u{3383}", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(Current::CODE))),
    /// ㎄
    ("\u{3384}", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(Current::CODE))),
    /// ㏟
    ("\u{33DF}", (Conv(ONE, ZERO), DynDim::new(MagneticFieldStrength::CODE))),
    // J
    ("cd", (Conv(ONE, ZERO), DynDim::new(Luminous::CODE))),
    /// ㏅
    ("\u{33C5}", (Conv(ONE, ZERO), DynDim::new(Luminous::CODE))),
    ("lm", (SR2BASE, DynDim::new(Luminous::CODE))),
    /// ㏐
    ("\u{33D0}", (SR2BASE, DynDim::new(Luminous::CODE))),
    ("lx", (SR2BASE, DynDim::new(Illuminance::CODE))),
    /// ㏓
    ("\u{33D3}", (SR2BASE, DynDim::new(Illuminance::CODE))),
    // Frequency
    ("Hz", (Conv(ONE, ZERO), DynDim::new(Frequency::CODE))),
    /// ㌹
    ("\u{3339}", (Conv(ONE, ZERO), DynDim::new(Frequency::CODE))),
    /// ㎐
    ("\u{3339}", (Conv(ONE, ZERO), DynDim::new(Frequency::CODE))),
    /// ㎑
    ("\u{3339}", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(Frequency::CODE))),
    /// ㎒
    ("\u{3339}", (Conv(Frac::from_exp10(6), ZERO), DynDim::new(Frequency::CODE))),
    /// ㎓
    ("\u{3339}", (Conv(Frac::from_exp10(9), ZERO), DynDim::new(Frequency::CODE))),
    /// ㎔
    ("\u{3339}", (Conv(Frac::from_exp10(12), ZERO), DynDim::new(Frequency::CODE))),
    ("Bq", (Conv(ONE, ZERO), DynDim::new(Frequency::CODE))),
    /// ㏃
    ("\u{33C3}", (Conv(ONE, ZERO), DynDim::new(Frequency::CODE))),
    ("Ci", (Conv(Frac::from_ratio(37, 1, 9), ZERO), DynDim::new(Frequency::CODE))),
    /// ㌒
    ("\u{3312}", (
        Conv(Frac::from_ratio(37, 1, 9), ZERO),
        DynDim::new(Frequency::CODE)
    )),
    /// ㌟
    ("\u{331F}", (Conv(ONE, ZERO), DynDim::new(Frequency::CODE))),
    /// ㎮
    ("\u{33AE}", (RAD2BASE, DynDim::new(Frequency::CODE))),
    /// ㎯
    ("\u{33AF}", (RAD2BASE, DynDim::new(FrequencyAcceleration::CODE))),
    // Area
    /// ㍸
    ("\u{3378}", (Conv(Frac::from_exp10(-2), ZERO), DynDim::new(Area::CODE))),
    /// ㎟
    ("\u{339F}", (Conv(Frac::from_exp10(-6), ZERO), DynDim::new(Area::CODE))),
    /// ㎠
    ("\u{33A0}", (Conv(Frac::from_exp10(-4), ZERO), DynDim::new(Area::CODE))),
    /// ㎡
    ("\u{33A1}", (Conv(ONE, ZERO), DynDim::new(Area::CODE))),
    /// ㎢
    ("\u{33A2}", (Conv(Frac::from_exp10(6), ZERO), DynDim::new(Area::CODE))),
    /// are
    ("a.", (Conv(Frac::from_exp10(2), ZERO), DynDim::new(Area::CODE))),
    /// are (а:CYRILLIC SMALL LETTER A)
    ("\u{0430}", (Conv(Frac::from_exp10(2), ZERO), DynDim::new(Area::CODE))),
    /// ㌃
    ("\u{3303}", (Conv(Frac::from_exp10(2), ZERO), DynDim::new(Area::CODE))),
    /// ㌶
    ("\u{3336}", (Conv(Frac::from_exp10(4), ZERO), DynDim::new(Area::CODE))),
    /// ㏊
    ("\u{33CA}", (Conv(Frac::from_exp10(4), ZERO), DynDim::new(Area::CODE))),
    ("acre", (
        ACRE2FT2.mul(FOOT2INCH).mul(FOOT2INCH).mul(INCH2METER).mul(INCH2METER),
        DynDim::new(Area::CODE)
    )),
    /// ㌈
    ("\u{3308}",
        (ACRE2FT2.mul(FOOT2INCH).mul(FOOT2INCH).mul(INCH2METER).mul(INCH2METER),
        DynDim::new(Area::CODE))
    ),
    // Volume
    ("l", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(Volume::CODE))),
    ("L", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(Volume::CODE))),
    /// ℓ
    ("\u{2113}", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(Volume::CODE))),
    /// ㎕
    ("\u{3395}", (Conv(Frac::from_exp10(-9), ZERO), DynDim::new(Volume::CODE))),
    /// ㎖
    ("\u{3396}", (Conv(Frac::from_exp10(-6), ZERO), DynDim::new(Volume::CODE))),
    /// ㎗
    ("\u{3397}", (Conv(Frac::from_exp10(-4), ZERO), DynDim::new(Volume::CODE))),
    /// ㎘
    ("\u{3398}", (Conv(ONE, ZERO), DynDim::new(Volume::CODE))),
    /// ㍹
    ("\u{3379}", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(Volume::CODE))),
    /// ㎣
    ("\u{33A3}", (Conv(Frac::from_exp10(-9), ZERO), DynDim::new(Volume::CODE))),
    /// ㎤
    ("\u{33A4}", (Conv(Frac::from_exp10(-6), ZERO), DynDim::new(Volume::CODE))),
    /// ㏄
    ("\u{33C4}", (Conv(Frac::from_exp10(-6), ZERO), DynDim::new(Volume::CODE))),
    /// ㎥
    ("\u{33A5}", (Conv(ONE, ZERO), DynDim::new(Volume::CODE))),
    /// ㎦
    ("\u{33A6}", (Conv(Frac::from_exp10(9), ZERO), DynDim::new(Volume::CODE))),
    ("gal(en)", (
        Conv(Frac::from_ratio(4_546_09, 1, -8), ZERO),
        DynDim::new(Volume::CODE)
    )),
    ("gal(us)", (
        Conv(Frac::from_ratio(3_785_412, 1, -9), ZERO),
        DynDim::new(Volume::CODE)
    )),
    /// ㌎
    ("\u{330E}", (
        Conv(Frac::from_ratio(3_785_412, 1, -9), ZERO),
        DynDim::new(Volume::CODE)
    )),
    ("bbl", (
        Conv(Frac::from_ratio(158_987_294_928, 1, -12), ZERO),
        DynDim::new(Volume::CODE)
    )),
    /// ㌭
    ("\u{332D}", (
        Conv(Frac::from_ratio(158_987_294_928, 1, -12), ZERO),
        DynDim::new(Volume::CODE)
    )),
    // Velocity
    ("c", (Conv(Frac::from_int(299_792_458), ZERO), DynDim::new(Velocity::CODE))),
    /// ㎧
    ("\u{33A7}", (Conv(ONE, ZERO), DynDim::new(Velocity::CODE))),
    ("kn", (Conv(Frac::from_ratio(1852, 3600, 0), ZERO), DynDim::new(Velocity::CODE))),
    /// ㌩
    ("\u{3329}", (
        Conv(Frac::from_ratio(1852, 3600, 0), ZERO),
        DynDim::new(Velocity::CODE)
    )),
    // Acceleration
    /// ㎨
    ("\u{33A8}", (Conv(ONE, ZERO), DynDim::new(Velocity::CODE))),
    ("Gal", (Conv(Frac::from_exp10(-2), ZERO), DynDim::new(Velocity::CODE))),
    /// ㏿
    ("\u{33FF}", (Conv(Frac::from_exp10(-2), ZERO), DynDim::new(Velocity::CODE))),
    // Force
    ("N", (NEWTON2BASE, DynDim::new(Force::CODE))),
    ("gf", (GRAM_FORCE2NEWTON.then(NEWTON2BASE), DynDim::new(Force::CODE))),
    ("lbf", (POUND2GRAM.then(GRAM_FORCE2NEWTON).then(NEWTON2BASE), DynDim::new(Force::CODE))),
    // Pressure
    ("Pa", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(Pressure::CODE))),
    /// ㎩
    ("\u{33A9}", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(Pressure::CODE))),
    /// ㍱
    ("\u{3371}", (Conv(Frac::from_exp10(5), ZERO), DynDim::new(Pressure::CODE))),
    /// ㎪
    ("\u{33AA}", (Conv(Frac::from_exp10(6), ZERO), DynDim::new(Pressure::CODE))),
    /// ㎫
    ("\u{33AB}", (Conv(Frac::from_exp10(9), ZERO), DynDim::new(Pressure::CODE))),
    /// ㎬
    ("\u{33AC}", (Conv(Frac::from_exp10(12), ZERO), DynDim::new(Pressure::CODE))),
    ("bar", (Conv(Frac::from_exp10(8), ZERO), DynDim::new(Pressure::CODE))),
    /// ㍴
    ("\u{3374}", (Conv(Frac::from_exp10(8), ZERO), DynDim::new(Pressure::CODE))),
    /// ㍊
    ("\u{334A}", (Conv(Frac::from_exp10(5), ZERO), DynDim::new(Pressure::CODE))),
    /// ㏔
    ("\u{33D4}", (Conv(Frac::from_exp10(5), ZERO), DynDim::new(Pressure::CODE))),
    ("at.", (AT2BASE, DynDim::new(Pressure::CODE))),
    ("ata", (AT2BASE, DynDim::new(Pressure::CODE))),
    ("atg", (Conv(AT2BASE.0, Frac::from_int(101325000)), DynDim::new(Pressure::CODE))),
    ("psi", (PSI2BASE, DynDim::new(Pressure::CODE))),
    ("psia", (PSI2BASE,DynDim::new(Pressure::CODE))),
    ("psig", (Conv(PSI2BASE.0, Frac::from_int(101325000)), DynDim::new(Pressure::CODE))),
    // Pressure based liquid head. lacked length dimension.
    ("Hg", (Conv(Frac::from_ratio(101325, 760, 3), ZERO), DynDim::new(Head::CODE))),
    /// ㋌
    ("\u{32CC}", (
        Conv(Frac::from_ratio(101325, 760, 3), ZERO),
        DynDim::new(Head::CODE)
    )),
    ("Aq", (Conv(Frac::from_int(9806650), ZERO), DynDim::new(Head::CODE))),
    // Energy
    ("J", (JOULE2BASE, DynDim::new(Energy::CODE))),
    ("cal", (CALORIE2JOULE.then(JOULE2BASE), DynDim::new(Energy::CODE))),
    /// ㌍
    ("\u{330D}", (CALORIE2JOULE.then(JOULE2BASE), DynDim::new(Energy::CODE))),
    /// ㎈
    ("\u{3388}", (CALORIE2JOULE.then(JOULE2BASE), DynDim::new(Energy::CODE))),
    /// ㎉
    ("\u{3389}", (
        Conv(Frac::from_exp10(3), ZERO).then(CALORIE2JOULE).then(JOULE2BASE),
        DynDim::new(Energy::CODE)
    )),
    ("E_h", (
        Conv(Frac::from_ratio(4_359_744_722_207_1, 1, -28), ZERO),
        DynDim::new(Energy::CODE)
    )),
    ("Btu", (BTU2CALORIE.then(CALORIE2JOULE).then(JOULE2BASE), DynDim::new(Energy::CODE))),
    ("Wh", (WATT2BASE.mul(HOUR2MINUTE.then(MINUTE2SECOND)), DynDim::new(Energy::CODE))),
    ("erg", (Conv(Frac::from_exp10(-4), ZERO), DynDim::new(Energy::CODE))),
    /// ㋍
    ("\u{32CD}", (Conv(Frac::from_exp10(-4), ZERO), DynDim::new(Energy::CODE))),
    ("eV", (
        Conv(Frac::from_ratio(1_602_176_634, 1, -25), ZERO),
        DynDim::new(Energy::CODE)
    )),
    /// ㋎
    ("\u{32CE}", (
        Conv(Frac::from_ratio(1_602_176_634, 1, -25), ZERO),
        DynDim::new(Energy::CODE)
    )),
    // Radiation
    ("Gy", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(Enthalpy::CODE))),
    /// ㏉
    ("\u{33C9}", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(Enthalpy::CODE))),
    ("Sv", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(Enthalpy::CODE))),
    /// ㏜
    ("\u{33DC}", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(Enthalpy::CODE))),
    /// ㍕
    ("\u{3355}", (Conv(Frac::from_exp10(1), ZERO), DynDim::new(Enthalpy::CODE))),
    /// ㍖
    ("\u{3356}", (
        Conv(Frac::from_ratio(258, 1, -5), ZERO),
        DynDim::new(RadiationExposure::CODE)
    )),
    // Power
    ("W", (WATT2BASE, DynDim::new(Power::CODE))),
    /// ㍗
    ("\u{3357}", (WATT2BASE, DynDim::new(Power::CODE))),
    /// ㌗
    ("\u{3317}", (Conv(Frac::from_exp10(6), ZERO), DynDim::new(Power::CODE))),
    /// ㎺
    ("\u{33BA}", (Conv(Frac::from_exp10(-15), ZERO), DynDim::new(Power::CODE))),
    /// ㎻
    ("\u{33BB}", (Conv(Frac::from_exp10(-12), ZERO), DynDim::new(Power::CODE))),
    /// ㎼
    ("\u{33BC}", (Conv(Frac::from_exp10(-9), ZERO), DynDim::new(Power::CODE))),
    /// ㎽
    ("\u{33BD}", (Conv(Frac::from_exp10(-6), ZERO), DynDim::new(Power::CODE))),
    /// ㎾
    ("\u{33BE}", (Conv(Frac::from_exp10(6), ZERO), DynDim::new(Power::CODE))),
    /// ㎿
    ("\u{33BF}", (Conv(Frac::from_exp10(9), ZERO), DynDim::new(Power::CODE))),
    /// ㏋
    ("\u{33CB}", (
        Conv(Frac::from_int(550), ZERO)
            .mul(POUND2GRAM).mul(GRAM_FORCE2NEWTON).mul(NEWTON2BASE).mul(FOOT2INCH).mul(INCH2METER),
        DynDim::new(Power::CODE)
    )),
    // Electrical voltage
    ("V", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(Voltage::CODE))),
    /// ㏞
    ("\u{33DE}", (Conv(ONE, ZERO), DynDim::new(ElectricFieldStrength::CODE))),
    /// ㌾
    ("\u{333E}", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(Voltage::CODE))),
    /// ㎴
    ("\u{33B4}", (Conv(Frac::from_exp10(-9), ZERO), DynDim::new(Voltage::CODE))),
    /// ㎵
    ("\u{33B5}", (Conv(Frac::from_exp10(-6), ZERO), DynDim::new(Voltage::CODE))),
    /// ㎶
    ("\u{33B6}", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(Voltage::CODE))),
    /// ㎷
    ("\u{33B7}", (Conv(ONE, ZERO), DynDim::new(Voltage::CODE))),
    ///㎸
    ("\u{33B8}", (Conv(Frac::from_exp10(6), ZERO), DynDim::new(Voltage::CODE))),
    /// ㎹
    ("\u{33B9}", (Conv(Frac::from_int(1000_000_000), ZERO), DynDim::new(Voltage::CODE))),
    // Electrical registance
    /// Ω
    ("\u{03A9}", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(ElectricResistance::CODE))),
    /// Ω
    ("\u{2126}", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(ElectricResistance::CODE))),
    /// ㌊
    ("\u{330A}", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(ElectricResistance::CODE))),
    /// ㏀
    ("\u{33C0}", (Conv(Frac::from_exp10(6), ZERO), DynDim::new(ElectricResistance::CODE))),
    /// ㏁
    ("\u{33C1}", (Conv(Frac::from_int(1000_000_000), ZERO), DynDim::new(ElectricResistance::CODE))),
    ("S", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(ElectricConductance::CODE))),
    /// ℧
    ("\u{2127}", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(ElectricConductance::CODE))),
    ("H", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(ElectricalInductance::CODE))),

    // Electrical Capacitance
    ("F", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(Capacitance::CODE))),
    /// ㌲
    ("\u{3332}", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(Capacitance::CODE))),
    /// ㎊
    ("\u{338A}", (Conv(Frac::from_exp10(-15), ZERO), DynDim::new(Capacitance::CODE))),
    /// ㎋
    ("\u{338B}", (Conv(Frac::from_exp10(-12), ZERO), DynDim::new(Capacitance::CODE))),
    /// ㎌
    ("\u{338C}", (Conv(Frac::from_exp10(-9), ZERO), DynDim::new(Capacitance::CODE))),

    ("C", (Conv(ONE, ZERO), DynDim::new(ElectricCharge::CODE))),
    /// ㏆
    ("\u{33C6}", (Conv(Frac::from_exp10(-3), ZERO), DynDim::new(RadiationExposure::CODE))),
    ("Wb", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(MagneticFlux::CODE))),
    /// ㏝
    ("\u{33DD}", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(MagneticFlux::CODE))),
    ("T", (Conv(Frac::from_exp10(3), ZERO), DynDim::new(MagneticFluxDensity::CODE))),
];

#[cfg(test)]
mod tests {
    extern crate std;

    use super::*;
    use std::eprintln;

    #[test]
    #[ignore]
    fn dump_def_table() {
        for (key, (cnv, dim)) in DEFAULT_UNIT_DEF.iter() {
            eprintln!("{} = {:?}, {}", key, cnv, dim);
        }
        panic!();
    }

    #[test]
    fn test_ata() {
        let cnv = GRAM_FORCE2NEWTON;
        let cnv = cnv.mul(NEWTON2BASE);
        let a = Conv(Frac::from_exp10(7), ZERO);
        let cnv = cnv.mul(a);

        assert_eq!(cnv.0.to_f64(), 98066500.0);
    }
}