/*
Portions Copyright 2019-2021 ZomboDB, LLC.
Portions Copyright 2021-2022 Technology Concepts & Design, Inc. <support@tcdi.com>

All rights reserved.

Use of this source code is governed by the MIT license that can be found in the LICENSE file.
*/

use crate::datum::time::USECS_PER_SEC;
use crate::{direct_function_call_as_datum, pg_sys, FromDatum, IntoDatum, TimestampWithTimeZone};
use std::ops::{Deref, DerefMut};
use time::{format_description::FormatItem, PrimitiveDateTime};

#[derive(Debug, Copy, Clone)]
pub struct Timestamp(time::PrimitiveDateTime);

impl From<pg_sys::Timestamp> for Timestamp {
    fn from(item: pg_sys::Timestamp) -> Self {
        unsafe { Timestamp::from_datum(item as usize, false, pg_sys::TIMESTAMPOID).unwrap() }
    }
}

impl FromDatum for Timestamp {
    #[inline]
    unsafe fn from_datum(datum: pg_sys::Datum, is_null: bool, typoid: u32) -> Option<Timestamp> {
        let ts: Option<TimestampWithTimeZone> =
            TimestampWithTimeZone::from_datum(datum, is_null, typoid);
        match ts {
            None => None,
            Some(ts) => {
                let date = ts.date();
                let time = ts.time();

                Some(Timestamp(PrimitiveDateTime::new(date, time)))
            }
        }
    }
}
impl IntoDatum for Timestamp {
    #[inline]
    fn into_datum(self) -> Option<pg_sys::Datum> {
        let year = self.year();
        let month = self.month() as i32;
        let mday = self.day() as i32;
        let hour = self.hour() as i32;
        let minute = self.minute() as i32;
        let second = self.second() as f64 + (self.microsecond() as f64 / USECS_PER_SEC as f64);

        unsafe {
            direct_function_call_as_datum(
                pg_sys::make_timestamp,
                vec![
                    year.into_datum(),
                    month.into_datum(),
                    mday.into_datum(),
                    hour.into_datum(),
                    minute.into_datum(),
                    second.into_datum(),
                ],
            )
        }
    }

    fn type_oid() -> u32 {
        pg_sys::TIMESTAMPOID
    }
}
impl Timestamp {
    pub fn new(timestamp: time::PrimitiveDateTime) -> Self {
        Timestamp(timestamp)
    }
}

impl Deref for Timestamp {
    type Target = time::PrimitiveDateTime;

    fn deref(&self) -> &Self::Target {
        &self.0
    }
}
impl DerefMut for Timestamp {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.0
    }
}

impl serde::Serialize for Timestamp {
    fn serialize<S>(
        &self,
        serializer: S,
    ) -> std::result::Result<<S as serde::Serializer>::Ok, <S as serde::Serializer>::Error>
    where
        S: serde::Serializer,
    {
        if self.millisecond() > 0 {
            serializer.serialize_str(
                &self
                    .format(
                        &time::format_description::parse(&format!(
                            "[year]-[month]-[day]T[hour]:[minute]:[second].{}-00",
                            self.millisecond()
                        ))
                        .map_err(|e| {
                            serde::ser::Error::custom(format!(
                                "Timestamp invalid format problem: {:?}",
                                e
                            ))
                        })?,
                    )
                    .map_err(|e| {
                        serde::ser::Error::custom(format!("Timestamp formatting problem: {:?}", e))
                    })?,
            )
        } else {
            serializer.serialize_str(&self.format(&DEFAULT_TIMESTAMP_FORMAT).map_err(|e| {
                serde::ser::Error::custom(format!("Timestamp formatting problem: {:?}", e))
            })?)
        }
    }
}

static DEFAULT_TIMESTAMP_FORMAT: &[FormatItem<'static>] =
    time::macros::format_description!("[year]-[month]-[day]T[hour]:[minute]:[second]-00");
