use lazy_static::lazy_static;
use regex::Regex;
use std::collections::HashSet;

pub trait SqlString {
    fn sql_identifier(&self) -> Self;
    fn sql_value(&self) -> Self;
}

impl SqlString for String {
    fn sql_value(&self) -> Self {
        format!(r#"'{}'"#, self)
    }

    fn sql_identifier(&self) -> Self {
        if needs_quoting(self) {
            return format!(r#""{}""#, self);
        }
        self.to_owned()
    }
}

fn needs_quoting(string: &str) -> bool {
    let lower = string.to_ascii_lowercase();

    // If it doesn't match the lower case equivalent, assume the column name is actually mixed cased and that we need to quote it.
    if !string.eq(&lower) {
        return true;
    }

    if NOT_ALPHANUMERIC.is_match(string) {
        return true;
    }

    if POSTGRESQL_RESERVED_WORDS.contains(&lower) {
        return true;
    }

    false
}

#[cfg(test)]
mod tests {
    use crate::sql_string::*;

    #[test]
    fn it_does_not_quote_when_optional() {
        assert_eq!("hello".to_string().sql_identifier(), "hello")
    }

    #[test]
    fn it_quotes_mixed_case() {
        assert_eq!("fullName".to_string().sql_identifier(), "\"fullName\"")
    }

    #[test]
    fn it_quotes_reserved_words() {
        assert_eq!("action".to_string().sql_identifier(), "\"action\"")
    }

    #[test]
    fn it_quotes_terrible_names() {
        assert_eq!("my data".to_string().sql_identifier(), "\"my data\"")
    }
}

lazy_static! {
    static ref NOT_ALPHANUMERIC: Regex = Regex::new("[^a-zA-Z0-9]").unwrap();

    /// Reserved words according to https://www.postgresql.org/docs/14/sql-keywords-appendix.html
    static ref POSTGRESQL_RESERVED_WORDS: HashSet<String> = {
        let mut set = HashSet::new();
        set.insert("a".to_string());
        set.insert("abort".to_string());
        set.insert("abs".to_string());
        set.insert("absent".to_string());
        set.insert("absolute".to_string());
        set.insert("access".to_string());
        set.insert("according".to_string());
        set.insert("acos".to_string());
        set.insert("action".to_string());
        set.insert("ada".to_string());
        set.insert("add".to_string());
        set.insert("admin".to_string());
        set.insert("after".to_string());
        set.insert("aggregate".to_string());
        set.insert("all".to_string());
        set.insert("allocate".to_string());
        set.insert("also".to_string());
        set.insert("alter".to_string());
        set.insert("always".to_string());
        set.insert("analyse".to_string());
        set.insert("analyze".to_string());
        set.insert("and".to_string());
        set.insert("any".to_string());
        set.insert("are".to_string());
        set.insert("array".to_string());
        set.insert("array_agg".to_string());
        set.insert("array_max_cardinality".to_string());
        set.insert("as".to_string());
        set.insert("asc".to_string());
        set.insert("asensitive".to_string());
        set.insert("asin".to_string());
        set.insert("assertion".to_string());
        set.insert("assignment".to_string());
        set.insert("asymmetric".to_string());
        set.insert("at".to_string());
        set.insert("atan".to_string());
        set.insert("atomic".to_string());
        set.insert("attach".to_string());
        set.insert("attribute".to_string());
        set.insert("attributes".to_string());
        set.insert("authorization".to_string());
        set.insert("avg".to_string());
        set.insert("backward".to_string());
        set.insert("base64".to_string());
        set.insert("before".to_string());
        set.insert("begin".to_string());
        set.insert("begin_frame".to_string());
        set.insert("begin_partition".to_string());
        set.insert("bernoulli".to_string());
        set.insert("between".to_string());
        set.insert("bigint".to_string());
        set.insert("binary".to_string());
        set.insert("bit".to_string());
        set.insert("bit_length".to_string());
        set.insert("blob".to_string());
        set.insert("blocked".to_string());
        set.insert("bom".to_string());
        set.insert("boolean".to_string());
        set.insert("both".to_string());
        set.insert("breadth".to_string());
        set.insert("by".to_string());
        set.insert("c".to_string());
        set.insert("cache".to_string());
        set.insert("call".to_string());
        set.insert("called".to_string());
        set.insert("cardinality".to_string());
        set.insert("cascade".to_string());
        set.insert("cascaded".to_string());
        set.insert("case".to_string());
        set.insert("cast".to_string());
        set.insert("catalog".to_string());
        set.insert("catalog_name".to_string());
        set.insert("ceil".to_string());
        set.insert("ceiling".to_string());
        set.insert("chain".to_string());
        set.insert("chaining".to_string());
        set.insert("char".to_string());
        set.insert("character".to_string());
        set.insert("characteristics".to_string());
        set.insert("characters".to_string());
        set.insert("character_length".to_string());
        set.insert("character_set_catalog".to_string());
        set.insert("character_set_name".to_string());
        set.insert("character_set_schema".to_string());
        set.insert("char_length".to_string());
        set.insert("check".to_string());
        set.insert("checkpoint".to_string());
        set.insert("class".to_string());
        set.insert("classifier".to_string());
        set.insert("class_origin".to_string());
        set.insert("clob".to_string());
        set.insert("close".to_string());
        set.insert("cluster".to_string());
        set.insert("coalesce".to_string());
        set.insert("cobol".to_string());
        set.insert("collate".to_string());
        set.insert("collation".to_string());
        set.insert("collation_catalog".to_string());
        set.insert("collation_name".to_string());
        set.insert("collation_schema".to_string());
        set.insert("collect".to_string());
        set.insert("column".to_string());
        set.insert("columns".to_string());
        set.insert("column_name".to_string());
        set.insert("command_function".to_string());
        set.insert("command_function_code".to_string());
        set.insert("comment".to_string());
        set.insert("comments".to_string());
        set.insert("commit".to_string());
        set.insert("committed".to_string());
        set.insert("compression".to_string());
        set.insert("concurrently".to_string());
        set.insert("condition".to_string());
        set.insert("conditional".to_string());
        set.insert("condition_number".to_string());
        set.insert("configuration".to_string());
        set.insert("conflict".to_string());
        set.insert("connect".to_string());
        set.insert("connection".to_string());
        set.insert("connection_name".to_string());
        set.insert("constraint".to_string());
        set.insert("constraints".to_string());
        set.insert("constraint_catalog".to_string());
        set.insert("constraint_name".to_string());
        set.insert("constraint_schema".to_string());
        set.insert("constructor".to_string());
        set.insert("contains".to_string());
        set.insert("content".to_string());
        set.insert("continue".to_string());
        set.insert("control".to_string());
        set.insert("conversion".to_string());
        set.insert("convert".to_string());
        set.insert("copy".to_string());
        set.insert("corr".to_string());
        set.insert("corresponding".to_string());
        set.insert("cos".to_string());
        set.insert("cosh".to_string());
        set.insert("cost".to_string());
        set.insert("count".to_string());
        set.insert("covar_pop".to_string());
        set.insert("covar_samp".to_string());
        set.insert("create".to_string());
        set.insert("cross".to_string());
        set.insert("csv".to_string());
        set.insert("cube".to_string());
        set.insert("cume_dist".to_string());
        set.insert("current".to_string());
        set.insert("current_catalog".to_string());
        set.insert("current_date".to_string());
        set.insert("current_default_transform_group".to_string());
        set.insert("current_path".to_string());
        set.insert("current_role".to_string());
        set.insert("current_row".to_string());
        set.insert("current_schema".to_string());
        set.insert("current_time".to_string());
        set.insert("current_timestamp".to_string());
        set.insert("current_transform_group_for_type".to_string());
        set.insert("current_user".to_string());
        set.insert("cursor".to_string());
        set.insert("cursor_name".to_string());
        set.insert("cycle".to_string());
        set.insert("data".to_string());
        set.insert("database".to_string());
        set.insert("datalink".to_string());
        set.insert("date".to_string());
        set.insert("datetime_interval_code".to_string());
        set.insert("datetime_interval_precision".to_string());
        set.insert("day".to_string());
        set.insert("db".to_string());
        set.insert("deallocate".to_string());
        set.insert("dec".to_string());
        set.insert("decfloat".to_string());
        set.insert("decimal".to_string());
        set.insert("declare".to_string());
        set.insert("default".to_string());
        set.insert("defaults".to_string());
        set.insert("deferrable".to_string());
        set.insert("deferred".to_string());
        set.insert("define".to_string());
        set.insert("defined".to_string());
        set.insert("definer".to_string());
        set.insert("degree".to_string());
        set.insert("delete".to_string());
        set.insert("delimiter".to_string());
        set.insert("delimiters".to_string());
        set.insert("dense_rank".to_string());
        set.insert("depends".to_string());
        set.insert("depth".to_string());
        set.insert("deref".to_string());
        set.insert("derived".to_string());
        set.insert("desc".to_string());
        set.insert("describe".to_string());
        set.insert("descriptor".to_string());
        set.insert("detach".to_string());
        set.insert("deterministic".to_string());
        set.insert("diagnostics".to_string());
        set.insert("dictionary".to_string());
        set.insert("disable".to_string());
        set.insert("discard".to_string());
        set.insert("disconnect".to_string());
        set.insert("dispatch".to_string());
        set.insert("distinct".to_string());
        set.insert("dlnewcopy".to_string());
        set.insert("dlpreviouscopy".to_string());
        set.insert("dlurlcomplete".to_string());
        set.insert("dlurlcompleteonly".to_string());
        set.insert("dlurlcompletewrite".to_string());
        set.insert("dlurlpath".to_string());
        set.insert("dlurlpathonly".to_string());
        set.insert("dlurlpathwrite".to_string());
        set.insert("dlurlscheme".to_string());
        set.insert("dlurlserver".to_string());
        set.insert("dlvalue".to_string());
        set.insert("do".to_string());
        set.insert("document".to_string());
        set.insert("domain".to_string());
        set.insert("double".to_string());
        set.insert("drop".to_string());
        set.insert("dynamic".to_string());
        set.insert("dynamic_function".to_string());
        set.insert("dynamic_function_code".to_string());
        set.insert("each".to_string());
        set.insert("element".to_string());
        set.insert("else".to_string());
        set.insert("empty".to_string());
        set.insert("enable".to_string());
        set.insert("encoding".to_string());
        set.insert("encrypted".to_string());
        set.insert("end".to_string());
        set.insert("end".to_string());
        set.insert("end_frame".to_string());
        set.insert("end_partition".to_string());
        set.insert("enforced".to_string());
        set.insert("enum".to_string());
        set.insert("equals".to_string());
        set.insert("error".to_string());
        set.insert("escape".to_string());
        set.insert("event".to_string());
        set.insert("every".to_string());
        set.insert("except".to_string());
        set.insert("exception".to_string());
        set.insert("exclude".to_string());
        set.insert("excluding".to_string());
        set.insert("exclusive".to_string());
        set.insert("exec".to_string());
        set.insert("execute".to_string());
        set.insert("exists".to_string());
        set.insert("exp".to_string());
        set.insert("explain".to_string());
        set.insert("expression".to_string());
        set.insert("extension".to_string());
        set.insert("external".to_string());
        set.insert("extract".to_string());
        set.insert("false".to_string());
        set.insert("family".to_string());
        set.insert("fetch".to_string());
        set.insert("file".to_string());
        set.insert("filter".to_string());
        set.insert("final".to_string());
        set.insert("finalize".to_string());
        set.insert("finish".to_string());
        set.insert("first".to_string());
        set.insert("first_value".to_string());
        set.insert("flag".to_string());
        set.insert("float".to_string());
        set.insert("floor".to_string());
        set.insert("following".to_string());
        set.insert("for".to_string());
        set.insert("force".to_string());
        set.insert("foreign".to_string());
        set.insert("format".to_string());
        set.insert("fortran".to_string());
        set.insert("forward".to_string());
        set.insert("found".to_string());
        set.insert("frame_row".to_string());
        set.insert("free".to_string());
        set.insert("freeze".to_string());
        set.insert("from".to_string());
        set.insert("fs".to_string());
        set.insert("fulfill".to_string());
        set.insert("full".to_string());
        set.insert("function".to_string());
        set.insert("functions".to_string());
        set.insert("fusion".to_string());
        set.insert("g".to_string());
        set.insert("general".to_string());
        set.insert("generated".to_string());
        set.insert("get".to_string());
        set.insert("global".to_string());
        set.insert("go".to_string());
        set.insert("goto".to_string());
        set.insert("grant".to_string());
        set.insert("granted".to_string());
        set.insert("greatest".to_string());
        set.insert("group".to_string());
        set.insert("grouping".to_string());
        set.insert("groups".to_string());
        set.insert("handler".to_string());
        set.insert("having".to_string());
        set.insert("header".to_string());
        set.insert("hex".to_string());
        set.insert("hierarchy".to_string());
        set.insert("hold".to_string());
        set.insert("hour".to_string());
        set.insert("id".to_string());
        set.insert("identity".to_string());
        set.insert("if".to_string());
        set.insert("ignore".to_string());
        set.insert("ilike".to_string());
        set.insert("immediate".to_string());
        set.insert("immediately".to_string());
        set.insert("immutable".to_string());
        set.insert("implementation".to_string());
        set.insert("implicit".to_string());
        set.insert("import".to_string());
        set.insert("in".to_string());
        set.insert("include".to_string());
        set.insert("including".to_string());
        set.insert("increment".to_string());
        set.insert("indent".to_string());
        set.insert("index".to_string());
        set.insert("indexes".to_string());
        set.insert("indicator".to_string());
        set.insert("inherit".to_string());
        set.insert("inherits".to_string());
        set.insert("initial".to_string());
        set.insert("initially".to_string());
        set.insert("inline".to_string());
        set.insert("inner".to_string());
        set.insert("inout".to_string());
        set.insert("input".to_string());
        set.insert("insensitive".to_string());
        set.insert("insert".to_string());
        set.insert("instance".to_string());
        set.insert("instantiable".to_string());
        set.insert("instead".to_string());
        set.insert("int".to_string());
        set.insert("integer".to_string());
        set.insert("integrity".to_string());
        set.insert("intersect".to_string());
        set.insert("intersection".to_string());
        set.insert("interval".to_string());
        set.insert("into".to_string());
        set.insert("invoker".to_string());
        set.insert("is".to_string());
        set.insert("isnull".to_string());
        set.insert("isolation".to_string());
        set.insert("join".to_string());
        set.insert("json".to_string());
        set.insert("json_array".to_string());
        set.insert("json_arrayagg".to_string());
        set.insert("json_exists".to_string());
        set.insert("json_object".to_string());
        set.insert("json_objectagg".to_string());
        set.insert("json_query".to_string());
        set.insert("json_table".to_string());
        set.insert("json_table_primitive".to_string());
        set.insert("json_value".to_string());
        set.insert("k".to_string());
        set.insert("keep".to_string());
        set.insert("key".to_string());
        set.insert("keys".to_string());
        set.insert("key_member".to_string());
        set.insert("key_type".to_string());
        set.insert("label".to_string());
        set.insert("lag".to_string());
        set.insert("language".to_string());
        set.insert("large".to_string());
        set.insert("last".to_string());
        set.insert("last_value".to_string());
        set.insert("lateral".to_string());
        set.insert("lead".to_string());
        set.insert("leading".to_string());
        set.insert("leakproof".to_string());
        set.insert("least".to_string());
        set.insert("left".to_string());
        set.insert("length".to_string());
        set.insert("level".to_string());
        set.insert("library".to_string());
        set.insert("like".to_string());
        set.insert("like_regex".to_string());
        set.insert("limit".to_string());
        set.insert("link".to_string());
        set.insert("listagg".to_string());
        set.insert("listen".to_string());
        set.insert("ln".to_string());
        set.insert("load".to_string());
        set.insert("local".to_string());
        set.insert("localtime".to_string());
        set.insert("localtimestamp".to_string());
        set.insert("location".to_string());
        set.insert("locator".to_string());
        set.insert("lock".to_string());
        set.insert("locked".to_string());
        set.insert("log".to_string());
        set.insert("log10".to_string());
        set.insert("logged".to_string());
        set.insert("lower".to_string());
        set.insert("m".to_string());
        set.insert("map".to_string());
        set.insert("mapping".to_string());
        set.insert("match".to_string());
        set.insert("matched".to_string());
        set.insert("matches".to_string());
        set.insert("match_number".to_string());
        set.insert("match_recognize".to_string());
        set.insert("materialized".to_string());
        set.insert("max".to_string());
        set.insert("maxvalue".to_string());
        set.insert("measures".to_string());
        set.insert("member".to_string());
        set.insert("merge".to_string());
        set.insert("message_length".to_string());
        set.insert("message_octet_length".to_string());
        set.insert("message_text".to_string());
        set.insert("method".to_string());
        set.insert("min".to_string());
        set.insert("minute".to_string());
        set.insert("minvalue".to_string());
        set.insert("mod".to_string());
        set.insert("mode".to_string());
        set.insert("modifies".to_string());
        set.insert("module".to_string());
        set.insert("month".to_string());
        set.insert("more".to_string());
        set.insert("move".to_string());
        set.insert("multiset".to_string());
        set.insert("mumps".to_string());
        set.insert("name".to_string());
        set.insert("names".to_string());
        set.insert("namespace".to_string());
        set.insert("national".to_string());
        set.insert("natural".to_string());
        set.insert("nchar".to_string());
        set.insert("nclob".to_string());
        set.insert("nested".to_string());
        set.insert("nesting".to_string());
        set.insert("new".to_string());
        set.insert("next".to_string());
        set.insert("nfc".to_string());
        set.insert("nfd".to_string());
        set.insert("nfkc".to_string());
        set.insert("nfkd".to_string());
        set.insert("nil".to_string());
        set.insert("no".to_string());
        set.insert("none".to_string());
        set.insert("normalize".to_string());
        set.insert("normalized".to_string());
        set.insert("not".to_string());
        set.insert("nothing".to_string());
        set.insert("notify".to_string());
        set.insert("notnull".to_string());
        set.insert("nowait".to_string());
        set.insert("nth_value".to_string());
        set.insert("ntile".to_string());
        set.insert("null".to_string());
        set.insert("nullable".to_string());
        set.insert("nullif".to_string());
        set.insert("nulls".to_string());
        set.insert("number".to_string());
        set.insert("numeric".to_string());
        set.insert("object".to_string());
        set.insert("occurrences_regex".to_string());
        set.insert("octets".to_string());
        set.insert("octet_length".to_string());
        set.insert("of".to_string());
        set.insert("off".to_string());
        set.insert("offset".to_string());
        set.insert("oids".to_string());
        set.insert("old".to_string());
        set.insert("omit".to_string());
        set.insert("on".to_string());
        set.insert("one".to_string());
        set.insert("only".to_string());
        set.insert("open".to_string());
        set.insert("operator".to_string());
        set.insert("option".to_string());
        set.insert("options".to_string());
        set.insert("or".to_string());
        set.insert("order".to_string());
        set.insert("ordering".to_string());
        set.insert("ordinality".to_string());
        set.insert("others".to_string());
        set.insert("out".to_string());
        set.insert("outer".to_string());
        set.insert("output".to_string());
        set.insert("over".to_string());
        set.insert("overflow".to_string());
        set.insert("overlaps".to_string());
        set.insert("overlay".to_string());
        set.insert("overriding".to_string());
        set.insert("owned".to_string());
        set.insert("owner".to_string());
        set.insert("p".to_string());
        set.insert("pad".to_string());
        set.insert("parallel".to_string());
        set.insert("parameter".to_string());
        set.insert("parameter_mode".to_string());
        set.insert("parameter_name".to_string());
        set.insert("parameter_ordinal_position".to_string());
        set.insert("parameter_specific_catalog".to_string());
        set.insert("parameter_specific_name".to_string());
        set.insert("parameter_specific_schema".to_string());
        set.insert("parser".to_string());
        set.insert("partial".to_string());
        set.insert("partition".to_string());
        set.insert("pascal".to_string());
        set.insert("pass".to_string());
        set.insert("passing".to_string());
        set.insert("passthrough".to_string());
        set.insert("password".to_string());
        set.insert("past".to_string());
        set.insert("path".to_string());
        set.insert("pattern".to_string());
        set.insert("per".to_string());
        set.insert("percent".to_string());
        set.insert("percentile_cont".to_string());
        set.insert("percentile_disc".to_string());
        set.insert("percent_rank".to_string());
        set.insert("period".to_string());
        set.insert("permission".to_string());
        set.insert("permute".to_string());
        set.insert("placing".to_string());
        set.insert("plan".to_string());
        set.insert("plans".to_string());
        set.insert("pli".to_string());
        set.insert("policy".to_string());
        set.insert("portion".to_string());
        set.insert("position".to_string());
        set.insert("position_regex".to_string());
        set.insert("power".to_string());
        set.insert("precedes".to_string());
        set.insert("preceding".to_string());
        set.insert("precision".to_string());
        set.insert("prepare".to_string());
        set.insert("prepared".to_string());
        set.insert("preserve".to_string());
        set.insert("primary".to_string());
        set.insert("prior".to_string());
        set.insert("private".to_string());
        set.insert("privileges".to_string());
        set.insert("procedural".to_string());
        set.insert("procedure".to_string());
        set.insert("procedures".to_string());
        set.insert("program".to_string());
        set.insert("prune".to_string());
        set.insert("ptf".to_string());
        set.insert("public".to_string());
        set.insert("publication".to_string());
        set.insert("quote".to_string());
        set.insert("quotes".to_string());
        set.insert("range".to_string());
        set.insert("rank".to_string());
        set.insert("read".to_string());
        set.insert("reads".to_string());
        set.insert("real".to_string());
        set.insert("reassign".to_string());
        set.insert("recheck".to_string());
        set.insert("recovery".to_string());
        set.insert("recursive".to_string());
        set.insert("ref".to_string());
        set.insert("references".to_string());
        set.insert("referencing".to_string());
        set.insert("refresh".to_string());
        set.insert("regr_avgx".to_string());
        set.insert("regr_avgy".to_string());
        set.insert("regr_count".to_string());
        set.insert("regr_intercept".to_string());
        set.insert("regr_r2".to_string());
        set.insert("regr_slope".to_string());
        set.insert("regr_sxx".to_string());
        set.insert("regr_sxy".to_string());
        set.insert("regr_syy".to_string());
        set.insert("reindex".to_string());
        set.insert("relative".to_string());
        set.insert("release".to_string());
        set.insert("rename".to_string());
        set.insert("repeatable".to_string());
        set.insert("replace".to_string());
        set.insert("replica".to_string());
        set.insert("requiring".to_string());
        set.insert("reset".to_string());
        set.insert("respect".to_string());
        set.insert("restart".to_string());
        set.insert("restore".to_string());
        set.insert("restrict".to_string());
        set.insert("result".to_string());
        set.insert("return".to_string());
        set.insert("returned_cardinality".to_string());
        set.insert("returned_length".to_string());
        set.insert("returned_octet_length".to_string());
        set.insert("returned_sqlstate".to_string());
        set.insert("returning".to_string());
        set.insert("returns".to_string());
        set.insert("revoke".to_string());
        set.insert("right".to_string());
        set.insert("role".to_string());
        set.insert("rollback".to_string());
        set.insert("rollup".to_string());
        set.insert("routine".to_string());
        set.insert("routines".to_string());
        set.insert("routine_catalog".to_string());
        set.insert("routine_name".to_string());
        set.insert("routine_schema".to_string());
        set.insert("row".to_string());
        set.insert("rows".to_string());
        set.insert("row_count".to_string());
        set.insert("row_number".to_string());
        set.insert("rule".to_string());
        set.insert("running".to_string());
        set.insert("savepoint".to_string());
        set.insert("scalar".to_string());
        set.insert("scale".to_string());
        set.insert("schema".to_string());
        set.insert("schemas".to_string());
        set.insert("schema_name".to_string());
        set.insert("scope".to_string());
        set.insert("scope_catalog".to_string());
        set.insert("scope_name".to_string());
        set.insert("scope_schema".to_string());
        set.insert("scroll".to_string());
        set.insert("search".to_string());
        set.insert("second".to_string());
        set.insert("section".to_string());
        set.insert("security".to_string());
        set.insert("seek".to_string());
        set.insert("select".to_string());
        set.insert("selective".to_string());
        set.insert("self".to_string());
        set.insert("sensitive".to_string());
        set.insert("sequence".to_string());
        set.insert("sequences".to_string());
        set.insert("serializable".to_string());
        set.insert("server".to_string());
        set.insert("server_name".to_string());
        set.insert("session".to_string());
        set.insert("session_user".to_string());
        set.insert("set".to_string());
        set.insert("setof".to_string());
        set.insert("sets".to_string());
        set.insert("share".to_string());
        set.insert("show".to_string());
        set.insert("similar".to_string());
        set.insert("simple".to_string());
        set.insert("sin".to_string());
        set.insert("sinh".to_string());
        set.insert("size".to_string());
        set.insert("skip".to_string());
        set.insert("smallint".to_string());
        set.insert("snapshot".to_string());
        set.insert("some".to_string());
        set.insert("source".to_string());
        set.insert("space".to_string());
        set.insert("specific".to_string());
        set.insert("specifictype".to_string());
        set.insert("specific_name".to_string());
        set.insert("sql".to_string());
        set.insert("sqlcode".to_string());
        set.insert("sqlerror".to_string());
        set.insert("sqlexception".to_string());
        set.insert("sqlstate".to_string());
        set.insert("sqlwarning".to_string());
        set.insert("sqrt".to_string());
        set.insert("stable".to_string());
        set.insert("standalone".to_string());
        set.insert("start".to_string());
        set.insert("state".to_string());
        set.insert("statement".to_string());
        set.insert("static".to_string());
        set.insert("statistics".to_string());
        set.insert("stddev_pop".to_string());
        set.insert("stddev_samp".to_string());
        set.insert("stdin".to_string());
        set.insert("stdout".to_string());
        set.insert("storage".to_string());
        set.insert("stored".to_string());
        set.insert("strict".to_string());
        set.insert("string".to_string());
        set.insert("strip".to_string());
        set.insert("structure".to_string());
        set.insert("style".to_string());
        set.insert("subclass_origin".to_string());
        set.insert("submultiset".to_string());
        set.insert("subscription".to_string());
        set.insert("subset".to_string());
        set.insert("substring".to_string());
        set.insert("substring_regex".to_string());
        set.insert("succeeds".to_string());
        set.insert("sum".to_string());
        set.insert("support".to_string());
        set.insert("symmetric".to_string());
        set.insert("sysid".to_string());
        set.insert("system".to_string());
        set.insert("system_time".to_string());
        set.insert("system_user".to_string());
        set.insert("t".to_string());
        set.insert("table".to_string());
        set.insert("tables".to_string());
        set.insert("tablesample".to_string());
        set.insert("tablespace".to_string());
        set.insert("table_name".to_string());
        set.insert("tan".to_string());
        set.insert("tanh".to_string());
        set.insert("temp".to_string());
        set.insert("template".to_string());
        set.insert("temporary".to_string());
        set.insert("text".to_string());
        set.insert("then".to_string());
        set.insert("through".to_string());
        set.insert("ties".to_string());
        set.insert("time".to_string());
        set.insert("timestamp".to_string());
        set.insert("timezone_hour".to_string());
        set.insert("timezone_minute".to_string());
        set.insert("to".to_string());
        set.insert("token".to_string());
        set.insert("top_level_count".to_string());
        set.insert("trailing".to_string());
        set.insert("transaction".to_string());
        set.insert("transactions_committed".to_string());
        set.insert("transactions_rolled_back".to_string());
        set.insert("transaction_active".to_string());
        set.insert("transform".to_string());
        set.insert("transforms".to_string());
        set.insert("translate".to_string());
        set.insert("translate_regex".to_string());
        set.insert("translation".to_string());
        set.insert("treat".to_string());
        set.insert("trigger".to_string());
        set.insert("trigger_catalog".to_string());
        set.insert("trigger_name".to_string());
        set.insert("trigger_schema".to_string());
        set.insert("trim".to_string());
        set.insert("trim_array".to_string());
        set.insert("true".to_string());
        set.insert("truncate".to_string());
        set.insert("trusted".to_string());
        set.insert("type".to_string());
        set.insert("types".to_string());
        set.insert("uescape".to_string());
        set.insert("unbounded".to_string());
        set.insert("uncommitted".to_string());
        set.insert("unconditional".to_string());
        set.insert("under".to_string());
        set.insert("unencrypted".to_string());
        set.insert("union".to_string());
        set.insert("unique".to_string());
        set.insert("unknown".to_string());
        set.insert("unlink".to_string());
        set.insert("unlisten".to_string());
        set.insert("unlogged".to_string());
        set.insert("unmatched".to_string());
        set.insert("unnamed".to_string());
        set.insert("unnest".to_string());
        set.insert("until".to_string());
        set.insert("untyped".to_string());
        set.insert("update".to_string());
        set.insert("upper".to_string());
        set.insert("uri".to_string());
        set.insert("usage".to_string());
        set.insert("user".to_string());
        set.insert("user_defined_type_catalog".to_string());
        set.insert("user_defined_type_code".to_string());
        set.insert("user_defined_type_name".to_string());
        set.insert("user_defined_type_schema".to_string());
        set.insert("using".to_string());
        set.insert("utf16".to_string());
        set.insert("utf32".to_string());
        set.insert("utf8".to_string());
        set.insert("vacuum".to_string());
        set.insert("valid".to_string());
        set.insert("validate".to_string());
        set.insert("validator".to_string());
        set.insert("value".to_string());
        set.insert("values".to_string());
        set.insert("value_of".to_string());
        set.insert("varbinary".to_string());
        set.insert("varchar".to_string());
        set.insert("variadic".to_string());
        set.insert("varying".to_string());
        set.insert("var_pop".to_string());
        set.insert("var_samp".to_string());
        set.insert("verbose".to_string());
        set.insert("version".to_string());
        set.insert("versioning".to_string());
        set.insert("view".to_string());
        set.insert("views".to_string());
        set.insert("volatile".to_string());
        set.insert("when".to_string());
        set.insert("whenever".to_string());
        set.insert("where".to_string());
        set.insert("whitespace".to_string());
        set.insert("width_bucket".to_string());
        set.insert("window".to_string());
        set.insert("with".to_string());
        set.insert("within".to_string());
        set.insert("without".to_string());
        set.insert("work".to_string());
        set.insert("wrapper".to_string());
        set.insert("write".to_string());
        set.insert("xml".to_string());
        set.insert("xmlagg".to_string());
        set.insert("xmlattributes".to_string());
        set.insert("xmlbinary".to_string());
        set.insert("xmlcast".to_string());
        set.insert("xmlcomment".to_string());
        set.insert("xmlconcat".to_string());
        set.insert("xmldeclaration".to_string());
        set.insert("xmldocument".to_string());
        set.insert("xmlelement".to_string());
        set.insert("xmlexists".to_string());
        set.insert("xmlforest".to_string());
        set.insert("xmliterate".to_string());
        set.insert("xmlnamespaces".to_string());
        set.insert("xmlparse".to_string());
        set.insert("xmlpi".to_string());
        set.insert("xmlquery".to_string());
        set.insert("xmlroot".to_string());
        set.insert("xmlschema".to_string());
        set.insert("xmlserialize".to_string());
        set.insert("xmltable".to_string());
        set.insert("xmltext".to_string());
        set.insert("xmlvalidate".to_string());
        set.insert("year".to_string());
        set.insert("yes".to_string());
        set.insert("zone".to_string());
        set
    };
}
