/*-
 * SPDX-License-Identifier: BSD-2-Clause
 *
 * Copyright (c) 2021 Aleksandr Morozov, RELKOM s.r.o
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials provided
 *      with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

use std::fmt;

#[derive(Copy, Clone, Eq, PartialEq)]
pub enum PfErrCode
{
    None,
    RuntimeAssertion,
    LibcGetAddrInfo(i32),
    UnknownAddressFamily(i32),
    /// IOCTL error: title, code
    IoCtlError(&'static str, libc::c_ulong),
    /// Some error while processing input or probably errors in code
    InternalError,
    /// Incorrect input data
    Einval,
    /// Io errors,
    IOErr,
    /// tokenizer error
    Tokenizer,
}

pub struct PfError 
{
    /// Error code
    err_code: PfErrCode,
    /// Description
    message: String,
}

impl PfError
{
    /// Creates new instance without any code
    pub fn new(msg: String) -> Self
    {
        return PfError{err_code: PfErrCode::None, message: msg};
    }

    /// Creates new instance with code number and message
    pub fn new_code(code: PfErrCode, msg: String) -> Self
    {
        return PfError{err_code: code, message: msg};
    }

    /// returns the kind of the error as [`PfErrCode`]
    pub fn kind(&self) -> PfErrCode
    {
        return self.err_code;
    }
}

impl fmt::Display for PfError 
{
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result 
    {
        write!(f, "{}", self.message)
    }
}
impl fmt::Debug for PfError 
{
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result 
    {
        write!(f, "{}", self.message)
    }
}


pub type PfResult<T> = Result<T, PfError>;


#[macro_export]
macro_rules! error 
{
    ($($arg:tt)*) => (
        return std::result::Result::Err(PfError::new(format!($($arg)*)))
    )
}

#[macro_export]
macro_rules! map_error 
{
    ($($arg:tt)*) => (
        return PfError::new(format!($($arg)*))
    )
}


#[macro_export]
macro_rules! error_coded 
{
    ($code:expr,$($arg:tt)*) => (
        return std::result::Result::Err(PfError::new_code($code, format!($($arg)*)))
    )
}

#[macro_export]
macro_rules! map_error_coded 
{
    ($code:expr,$($arg:tt)*) => (
        return PfError::new_code($code, format!($($arg)*))
    )
}

