/*-
 * SPDX-License-Identifier: BSD-2-Clause
 *
 * Copyright (c) 2021 Aleksandr Morozov, RELKOM s.r.o
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials provided
 *      with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */
 
use std::ffi::{CStr, CString};
use std::fmt;
use std::net::{IpAddr};
use std::path::Path;
use std::fs;


use nix::NixPath;
use nix::fcntl::{open, OFlag};
use nix::sys::stat::Mode;

use super::pf_tokenizer::PfListTokenizer;
use crate::portable::net::{htonl, ntohl};
use crate::{error_coded, cfor, map_error_coded, runtime_exception::*};


const PF_TABLE_NAME_SIZE: usize = 32;
const PF_RULE_LABEL_SIZE: usize = 64;
const MAXPATHLEN: usize = 1024;
const IFNAMSIZ: usize = 16;
const IF_NAMESIZE: usize = 16;

//libc is missing some constants for the FreeBSD
const AI_NUMERICHOST: libc::c_int = 0x00000004;

pub enum PfCmd
{
    /// Adds a host/hostname to table
    Add{ hosts: Vec<String> }, 

    /// Removes a host/hostname from table
    Delete{ hosts: Vec<String> }, 

    /// Checks if host/hostname presents in table
    Test{ hosts: Vec<String> },

    /// Flushes the 
    Flush,
}



#[allow(non_camel_case_types)]
#[derive(Debug, Copy, Clone)]
#[repr(C)]
enum Pfrb
{	
    PFRB_TABLES = 1, 
    PFRB_TSTATS, 
    PFRB_ADDRS, 
    PFRB_ASTATS,
    PFRB_IFACES, 
    PFRB_TRANS, 
    PFRB_MAX 
}

#[allow(non_camel_case_types)]
#[derive(Debug, PartialEq, Eq)]
#[repr(C)]
enum PfAddr
{
    PF_ADDR_ADDRMASK = 0,
    PF_ADDR_NOROUTE, 
    PF_ADDR_DYNIFTL,
    PF_ADDR_TABLE,
    PF_ADDR_URPFFAILED,
    PF_ADDR_RANGE
}

impl Into<u8> for PfAddr
{
    fn into(self) -> u8 
    {
        return self as u8;
    }
}

#[allow(non_camel_case_types)]
#[repr(C)]
#[derive(Copy, Clone)]
union pf_addr_union
{
    v4: libc::in_addr,
    v6: libc::in6_addr,
    addr8: [u8; 16],
    addr16: [u16; 8],
    addr32: [u32; 4],
}

#[allow(non_camel_case_types)]
#[repr(C)]
#[derive(Copy, Clone)]
struct pf_addr
{
    pfa: pf_addr_union,
}

impl From<libc::in_addr> for pf_addr
{
    fn from(addr: libc::in_addr) -> Self 
    {
        // zero memory first
        let mut s: Self = unsafe{ std::mem::zeroed() };

        s.pfa.v4 = addr;
        
        return s;
    }
}

impl From<libc::in6_addr> for pf_addr
{
    fn from(addr: libc::in6_addr) -> Self 
    {
        // zero memory first
        let mut s: Self = unsafe{ std::mem::zeroed() };

        s.pfa.v6 = addr;
        
        return s;
    }
}

/// For IPv4
impl From<[u8; 4]> for pf_addr
{
    fn from(octets: [u8; 4]) -> Self 
    {
        let mut s: Self = unsafe{ std::mem::zeroed() };
        s.pfa.v4.s_addr = u32::from_le_bytes(octets);

        return s;
    }
}

/// For IPv6
impl From<[u8; 16]> for pf_addr
{
    fn from(octets: [u8; 16]) -> Self 
    {
        let mut s: Self = unsafe{ std::mem::zeroed() };
        s.pfa.v6.s6_addr = octets;

        return s;
    }
}

impl pf_addr
{
    fn new_empty() -> Self
    {
        return unsafe{ std::mem::zeroed() };
    }

    unsafe fn pfctl_addrprefix<'addr>(&mut self, addr: &'addr str) -> PfResult<()>
    {
        let (adr, prefix) = 
            if let Some(pos) = addr.chars().position(|c| c == '/') //C if ((p = strchr(addr, '/')) == NULL) return
            {
                let p =
                    match addr.char_indices().nth(pos+1)  // skip '/'
                    {
                        None => error_coded!(PfErrCode::InternalError, "can not get char indeces at: '{}' for: '{}'", pos, addr),
                        Some((idx, _)) => 
                        {
                            let prefix = 
                                i32::from_str_radix(&addr[idx..], 10).map_err(
                                    |e| map_error_coded!(PfErrCode::InternalError, "can not decode: '{}' to i32, {}", &addr[idx..], e)
                                )?;

                            if prefix < 0
                            {
                                error_coded!(PfErrCode::Einval, "decoded val: '{}' < minval: 0", prefix);
                            }
                            else if prefix > 128
                            {
                                error_coded!(PfErrCode::Einval, "decoded val: '{}' > maxval: 128", prefix);
                            }

                            prefix
                        }
                    };

                let adr = 
                    match addr.char_indices().nth(pos-1)  // don't count '/'
                    {
                        None => error_coded!(PfErrCode::InternalError, "can not get char indeces at: '{}' for: '{}'", pos, addr),
                        Some((idx, _)) => &addr[..=idx],
                    };

                (adr, p)
            }
            else
            {
                return Ok(());
            };

        let mut res: *mut libc::addrinfo = std::ptr::null_mut();
        let mut hints: libc::addrinfo = std::mem::zeroed();

        // prefix only with numeric addresses
        hints.ai_flags |= AI_NUMERICHOST;

        let c = CString::new(adr.to_string()).map_err(|e| map_error_coded!(PfErrCode::InternalError, "{}", e))?;
        let ret_ga = libc::getaddrinfo(c.as_ptr(), std::ptr::null(), &hints as *const _ as *const libc::addrinfo, &mut res);

        if ret_ga > 0
        {
            error_coded!(PfErrCode::LibcGetAddrInfo(ret_ga), "getaddrinfo: {}", CStr::from_ptr(libc::gai_strerror(ret_ga)).to_string_lossy());
        }
        else if res == std::ptr::null_mut()
        {
            error_coded!(PfErrCode::RuntimeAssertion, "getaddrinfo: returned NULL");
        }


        if (*res).ai_family == libc::AF_INET && prefix > 32
        {
            // drop the result
            libc::freeaddrinfo(res);

            error_coded!(PfErrCode::Einval, "prefix too long for AF_INET: {}", prefix);
        }
        else if (*res).ai_family == libc::AF_INET6 && prefix > 128
        {
            // drop the result
            libc::freeaddrinfo(res);

            error_coded!(PfErrCode::Einval, "prefix too long for AF_INET6: {}", prefix);
        }

        let q = prefix >> 3;
        let r = prefix & 7;

        self.pfa.addr32 = [0_u32; 4];

        match (*res).ai_family
        {
            libc::AF_INET => 
            {
                //C mask->v4.s_addr = htonl((u_int32_t) (0xffffffffffULL << (32 - prefix)));
                self.pfa.v4.s_addr = htonl(((0xffffffffff as i64) << ((32 - prefix) as i64)) as u32);
            },
            libc::AF_INET6 => 
            {
                if q > 0
                {
                    // memset((void *)&mask->v6, 0xff, q);
                    for i in 0..q as usize
                    {
                        self.pfa.v6.s6_addr[i] = 0xFF;
                    }
                }

                if r > 0
                {
                    //C *((u_char *)&mask->v6 + q) = (0xff00 >> r) & 0xff;
                    self.pfa.v6.s6_addr[q as usize] = (0xff00 >> r as u16) as u8 & 0xff;
                }
            },
            _ => 
            {
                // drop the result
                libc::freeaddrinfo(res);

                error_coded!(PfErrCode::UnknownAddressFamily((*res).ai_family), "unknown ai_family: {}", (*res).ai_family);
            }
        }

        // drop the result
        libc::freeaddrinfo(res);
        drop(res);

        return Ok(());
    }

    // todo test  https://github.com/freebsd/freebsd-src/blob/858937bea4599d254a97ee6321683f8629604e15/sbin/pfctl/pf_print_state.c#L365
    unsafe fn unmask(&self, af: libc::sa_family_t) -> libc::c_int
    {
        let mut i: usize = 31;
        let mut j: usize = 0;
        let mut b: libc::c_int = 0;
        let mut tmp: u32 = 0;

        while j < 4 && self.pfa.addr32[j] == 0xFFFFFFFF
        {
            b += 32;
            j += 1;
        }

        if j < 4
        {
            tmp = ntohl(self.pfa.addr32[j]); // tmp = ntohl(m->addr32[j]);
            cfor!(let mut i = 31; tmp & (1 << i) > 0; i -= 1; // or (i = 31; tmp & (1 << i); --i)
                {
                b += 1;   //C b++;
                }
            ); 
        }

        return b;
    }
}



#[allow(non_camel_case_types)]
#[repr(C)]
#[derive(Copy, Clone)]
struct pf_addr_wrap_1
{
    addr: pf_addr,
    mask: pf_addr,
}

// https://github.com/freebsd/freebsd-src/blob/27ab791a55191c0b6503391d411303b042b41047/sbin/pfctl/pfctl_parser.c :1127
impl pf_addr_wrap_1
{
    unsafe fn new_zeroed() -> Self
    {
        return std::mem::zeroed();
    }
}

#[allow(non_camel_case_types)]
#[repr(C)]
union pf_addr_wrap_v_union
{
    a: pf_addr_wrap_1,
    ifname: [u8; IFNAMSIZ],
    tblname: [u8; PF_TABLE_NAME_SIZE],
}

// sizeof(pf_addr_wrap) should be 48 (struct pf_addr_wrap)
#[allow(non_camel_case_types)]
#[repr(C)]
struct pf_addr_wrap
{
    v: pf_addr_wrap_v_union, // this one is 32
    p: [u8; 8], // skip this sizeof pf_addr_wrap.p is 8 on x86-64
    tp: u8,
    iflags: u8,
    // pad 4 because compiler is aligning to 44
    pad0: u32,
}

impl fmt::Debug for pf_addr_wrap
{
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result 
    {
        write!(f, "v: union ")?;
        write!(f, "p: {:?} ", self.p)?;
        write!(f, "tp: {} ", self.tp)?;
        write!(f, "iflags: {} ", self.iflags)
    }
}

impl pf_addr_wrap
{
    unsafe fn set_ipmask(pf_addr_type: PfAddr, bits: libc::c_int, mut addr: pf_addr) -> Self
    {

        let mut m = pf_addr::new_empty();
        let mut b = bits;
        let mut j: usize = 0;

        while b >= 32                     // while (b >= 32) {
        {
            m.pfa.addr32[j] = 0xFFFFFFFF;   // m->addr32[j++] = 0xffffffff;
            j += 1;                         // |
            b -= 32;                        // b -= 32;
        }

        cfor!(let mut i = 31; i > 31-b; i -= 1; // for (i = 31; i > 31-b; --i)
            {
                m.pfa.addr32[j] |= 1 << i;   // m->addr32[j] |= (1 << i);
            }
        );

        if b > 0 //if (b)
        {
            m.pfa.addr32[j] = htonl(m.pfa.addr32[j]); // m->addr32[j] = htonl(m->addr32[j]);
        }

        //...
        if pf_addr_type == PfAddr::PF_ADDR_ADDRMASK // if (h->addr.type == PF_ADDR_ADDRMASK)
        {
            cfor!(let mut i = 0; i < 4; i += 1; // for (i = 0; i < 4; i++)
                {
                    addr.pfa.addr32[i] = addr.pfa.addr32[i] & m.pfa.addr32[i]; // n->addr32[i] = n->addr32[i] & m->addr32[i];
                }
            );
        }


        let pf_addr = 
            pf_addr_wrap_1
            {
                addr: addr,
                mask: m,
            };

        let ret = 
            Self
            {
                v: pf_addr_wrap_v_union{ a: pf_addr },
                p: [0_u8; 8],
                tp: pf_addr_type.into(), 
                iflags: 0,
                pad0: 0,  
            };
        
        return ret;
    }
}

#[allow(non_camel_case_types)]
#[repr(C)]
struct node_host
{
    addr: pf_addr_wrap,
    bcast: pf_addr,
    peer: pf_addr,
    af: libc::sa_family_t,
    not: u8,
    ifindex: u32,
    ifname: *const u8,
    ifa_flags: libc::c_uint,
}

impl node_host
{

    // https://github.com/freebsd/freebsd-src/blob/27ab791a55191c0b6503391d411303b042b41047/sbin/pfctl/pfctl_parser.c :1648
    unsafe fn host_v4v6<'pfcmd>(s: &'pfcmd str, maskv4: libc::c_int, maskv6: libc::c_int) -> PfResult<Option<Vec<Self>>>
    {
        let ipaddr: IpAddr = 
            match s.parse()
            {
                Ok(r) => r,
                Err(e) => return Ok(None),
            };

        let mut node_host_list = Vec::new();

        match ipaddr
        {
            IpAddr::V4(ipv4) =>
            {
                let addr = pf_addr::from(ipv4.octets());

                let mut addr_wrap: pf_addr_wrap = pf_addr_wrap::set_ipmask(PfAddr::PF_ADDR_ADDRMASK, maskv4, addr);

                // node_host does not contain references, so zeroed() can be used
                //let mut h: node_host= std::mem::zeroed();
                let h = 
                    Self
                    {
                        addr: addr_wrap,
                        bcast: pf_addr::new_empty(),
                        peer: pf_addr::new_empty(),
                        af: libc::AF_INET as libc::sa_family_t,
                        not: 0,
                        ifindex: 0,
                        ifname: std::ptr::null(),
                        ifa_flags: 0,
                    };

                // add to list i.e h->next = NULL; h->tail = h;
                node_host_list.push(h);
            
                return Ok(Some( node_host_list ));
            },
            IpAddr::V6(_ipv6) => // https://github.com/freebsd/freebsd-src/blob/27ab791a55191c0b6503391d411303b042b41047/sbin/pfctl/pfctl_parser.c#L1677
            {
                let mut res: *mut libc::addrinfo = std::ptr::null_mut();
                let mut hints: libc::addrinfo = std::mem::zeroed();
        
                // prefix only with numeric addresses
                hints.ai_family = libc::AF_INET6;
                hints.ai_socktype = libc::SOCK_DGRAM; //dummy
                hints.ai_flags = AI_NUMERICHOST;
        
                let c = CString::new(s.to_string()).map_err(|e| map_error_coded!(PfErrCode::InternalError, "{}", e))?;
                let service = CString::new("0".to_string()).map_err(|e| map_error_coded!(PfErrCode::InternalError, "{}", e))?;

                // getaddrinfo(s, "0", &hints, &res
                let ret_ga = 
                    libc::getaddrinfo(c.as_ptr(), service.as_ptr(), &hints as *const _ as *const libc::addrinfo, &mut res);
        
                if ret_ga > 0
                {
                    //runtime_error!("getaddrinfo: {}", CStr::from_ptr(libc::gai_strerror(ret_ga)).to_string_lossy());
                    return Ok(None);
                }
                else if res == std::ptr::null_mut()
                {
                    // assertion
                    error_coded!(PfErrCode::RuntimeAssertion, "getaddrinfo: returned NULL");
                }

                //let mut h: node_host= std::mem::zeroed();

                //h.ifname = std::ptr::null();
                //h.af = libc::AF_INET6 as u16;

                //memcpy(&h->addr.v.a.addr,&((struct sockaddr_in6 *)res->ai_addr)->sin6_addr,sizeof(h->addr.v.a.addr));
/*                libc::memcpy(
                    &mut h.addr.v.a.addr as *mut _ as *mut libc::c_void, 
                    &(*s_in6).sin6_addr as *const _ as *const libc::c_void,
                    std::mem::size_of::<pf_addr>()
                );*/

                //h->ifindex = ((struct sockaddr_in6 *)res->ai_addr)->sin6_scope_id;
            // h.ifindex = (*s_in6).sin6_scope_id;
                let s_in6 = (*res).ai_addr as *const libc::sockaddr_in6;
                let addr = pf_addr::from((*s_in6).sin6_addr);
                let addr_wrap: pf_addr_wrap = pf_addr_wrap::set_ipmask(PfAddr::PF_ADDR_ADDRMASK, maskv6, addr);

                let h = 
                    Self
                    {
                        addr: addr_wrap,
                        bcast: pf_addr::new_empty(),
                        peer: pf_addr::new_empty(),
                        af: libc::AF_INET6 as libc::sa_family_t,
                        not: 0,
                        ifindex: (*s_in6).sin6_scope_id,
                        ifname: std::ptr::null(),
                        ifa_flags: 0,
                    };

                // add to list i.e h->next = NULL; h->tail = h;
                node_host_list.push(h);

                libc::freeaddrinfo(res);

                return Ok(Some( node_host_list ));
            },
        }
    }

    unsafe fn host_dns<'pfcmd>(s: &'pfcmd str, maskv4: libc::c_int, maskv6: libc::c_int) -> PfResult<Option<Vec<Self>>>
    {
        let mut noalias: libc::c_int = 0;

        let ps =
            if let Some(pos) = s.chars().position(|c| c == ':') // p = strrchr(ps, ':')) != NULL
            {
                match s.char_indices().nth(pos) 
                {
                    None => error_coded!(PfErrCode::InternalError, "can not get char indeces at: '{}' for: '{}'", pos, s),
                    Some((idx, _)) => 
                    {
                        if &s[idx..] == ":0" // && !strcmp(p, ":0") i.e strcmp(p, ":0") == 0
                        {
                            noalias = 1;

                            &s[..idx]
                        }
                        else
                        {
                            s
                        }
                    },
                }
            }
            else
            {
                s
            };

            let mut res: *mut libc::addrinfo = std::ptr::null_mut();
            let mut hints: libc::addrinfo = std::mem::zeroed();

            hints.ai_family = libc::PF_UNSPEC;
            hints.ai_socktype = libc::SOCK_STREAM; // DUMMY

            let c_ps = CString::new(s.to_string()).map_err(|e| map_error_coded!(PfErrCode::InternalError, "{}", e))?;

            // getaddrinfo(s, "0", &hints, &res
            let ret_ga = 
                libc::getaddrinfo(
                    c_ps.as_ptr(), 
                    std::ptr::null(), 
                    &hints as *const _ as *const libc::addrinfo, 
                    &mut res
                );
    
            if ret_ga > 0
            {
                //runtime_error!("getaddrinfo: {}", CStr::from_ptr(libc::gai_strerror(ret_ga)).to_string_lossy());
                return Ok(None);
            }
            else if res == std::ptr::null_mut()
            {
                error_coded!(PfErrCode::RuntimeAssertion, "getaddrinfo: returned NULL");
            }

            let mut got4: libc::c_int = 0;
            let mut got6: libc::c_int = 0;
            let mut node_host_list: Vec<node_host> = Vec::new();

            cfor!(let mut p_resp = res; p_resp != std::ptr::null_mut(); p_resp = (*p_resp).ai_next;
                {
                    let resp = *p_resp;

                    if resp.ai_family != libc::AF_INET &&
                        resp.ai_family != libc::AF_INET6
                    {
                        continue;
                    }
                

                    if noalias > 0
                    {
                        if resp.ai_family == libc::AF_INET
                        {
                            if got4 > 0
                            {
                                continue;
                            }

                            got4 = 1;
                        }
                        else
                        {
                            if got6 > 0
                            {
                                continue;
                            }

                            got6 = 1;
                        }
                    } // if noalias

                    let mut ifindex: u32 = 0;

                    let addr_wrap: pf_addr_wrap = 
                        if resp.ai_family == libc::AF_INET
                        {
                            let s_in4: libc::sockaddr_in = *(resp.ai_addr as *const libc::sockaddr_in);
                            let addr = pf_addr::from(s_in4.sin_addr);

                            pf_addr_wrap::set_ipmask(PfAddr::PF_ADDR_ADDRMASK, maskv4, addr)
                        }
                        else 
                        {
                            let s_in6 = resp.ai_addr as *const libc::sockaddr_in6;

                            ifindex = (*s_in6).sin6_scope_id;

                            let addr = pf_addr::from((*s_in6).sin6_addr);

                            pf_addr_wrap::set_ipmask(PfAddr::PF_ADDR_ADDRMASK, maskv6, addr)
                        };

                    let n: Self = 
                        Self
                        {
                            addr: addr_wrap,
                            bcast: pf_addr::new_empty(),
                            peer: pf_addr::new_empty(),
                            af: resp.ai_family as libc::sa_family_t,
                            not: 0,
                            ifindex: ifindex,
                            ifname: std::ptr::null(),
                            ifa_flags: 0,
                        };

                    node_host_list.push(n);
                }
            );

            libc::freeaddrinfo(res);

            return Ok(
                if node_host_list.is_empty() == true
                {
                    None
                }
                else
                {
                    Some(node_host_list)
                }
            );
    }

    unsafe fn host<'pfcmd>(s: &'pfcmd str) -> PfResult<Vec<Self>>
    {
        let mut v4mask: libc::c_int = 32;
        let mut v6mask: libc::c_int = 128;
        let mut mask: libc::c_int = -1;

        // search for the '/' for the netmask
        let str_ip = 
            if let Some(pos) = s.chars().position(|c| c == '/')
            { 
                // position was found

                let str_ip =
                    match s.char_indices().nth(pos) 
                    {
                        None => error_coded!(PfErrCode::InternalError, "can not get char indeces at: '{}' for: '{}'", pos, s),
                        Some((idx, _)) => &s[..idx],
                    };
                
                let str_mask = 
                    match s.char_indices().nth(pos+1) // skip '/' = pos + 1 
                    {
                        None => error_coded!(PfErrCode::InternalError, "can not get char indeces at: '{}' for: '{}'", pos, s),
                        Some((idx, _)) => &s[idx..],
                    };

                // convert mask to int
                mask = 
                    match libc::c_int::from_str_radix(str_mask, 10)
                    {
                        Ok(m) =>
                        {
                            if m > 128
                            {
                                error_coded!(PfErrCode::Einval, "invalid netmask: '{}' > 128", m);
                            }

                            m
                        },
                        Err(e) => error_coded!(PfErrCode::Einval, "invalid netmask: '{}', {}", str_mask, e),
                    };

                v4mask = mask;
                v6mask = mask;

                str_ip
            }
            else
            {
                // not found
                s
            };

        // test for host_if and host_dns are skipped as it will work with IP only
        #[cfg(feature = "log_to_stdout")]
        println!("pf: IP address/host: {} mask4: {} mask6: {}", str_ip, v4mask, v6mask);

        if let Some(r) = Self::host_v4v6(str_ip, v4mask, v6mask)?
        {
            return Ok(r);
        }
        /*else if let Some(r) => Self::host_if(str_ip, mask)
        {

        }*/
        else if let Some(r) = Self::host_dns(str_ip, v4mask, v6mask)?
        {
            return Ok(r);
        }
        else
        {
            error_coded!(PfErrCode::Einval, "invalid address: {} not IP or DNS or IF", str_ip);
        }

        
    }

}

#[allow(non_camel_case_types)]
#[repr(C)]
union pfr_addr_in_addr
{
    _pfra_ip4addr: libc::in_addr,
    _pfra_ip6addr: libc::in6_addr
}

#[allow(non_camel_case_types)]
#[repr(C)]
struct pfr_addr
{
    pfra_u: pfr_addr_in_addr,
    pfra_af: u8,
    pfra_net: u8,
    pfra_not: u8,
    pfra_fback: u8,
}

impl pfr_addr
{
    // append_addr_host
    unsafe fn new(mut n: node_host, is_test: libc::c_int, not: libc::c_int) -> PfResult<Self>
    {
        let mut bits: libc::c_int = 0;

        let pfra_u = 
            match n.af as i32             // switch (n->af) {
            {
                libc::AF_INET => 
                {
                    bits = 32;
                    //C addr.pfra_ip4addr.s_addr = n->addr.v.a.addr.addr32[0];
                    pfr_addr_in_addr{ _pfra_ip4addr: libc::in_addr{ s_addr: n.addr.v.a.addr.pfa.addr32[0] } } 
                },
                libc::AF_INET6 =>
                {
                    bits = 128;
                    //C memcpy(&addr.pfra_ip6addr, &n->addr.v.a.addr.v6, sizeof(struct in6_addr));
                    pfr_addr_in_addr{ _pfra_ip6addr: libc::in6_addr{ s6_addr: n.addr.v.a.addr.pfa.addr8 } }
                },
                _ => error_coded!(PfErrCode::UnknownAddressFamily(n.af as i32), "Unknown AF flag in node_host: '{}'", n.af),
            };

        let mut addr =
            Self
            {
                pfra_u: pfra_u,
                pfra_af: n.af as u8,
                pfra_net: n.addr.v.a.mask.unmask(n.af) as u8,
                pfra_not: (n.not ^ (not as u8)),
                pfra_fback: 0,
            };


        if is_test > 0 && 
            (not > 0 || (addr.pfra_net as i32) != bits) || 
            (addr.pfra_net as i32) > bits
        {
            error_coded!(PfErrCode::Einval, "error at expresion: params: is_test: {}, not: {}, addr.pfra_net: {}, bits: {}",
                is_test, not, addr.pfra_net, bits);
        }

        return Ok(addr);
    }
}

#[derive(Debug)]
#[repr(C)]
struct pf_rule_addr 
{
    addr: pf_addr_wrap,
    port: [u16; 2],
    neg: u8,
    port_op: u8,
}

/// Keep synced with struct pf_kstate.
#[derive(Debug)]
#[repr(C)]
struct pf_state_cmp 
{
    id: u64,
    creatorid: u32,
    direction: u8,
    pad: [u8; 3],
}

#[derive(Debug)]
#[repr(C)]
struct pfioc_state_kill 
{
    psk_pfcmp: pf_state_cmp,
    psk_af: libc::sa_family_t,
    psk_proto: libc::c_int,
    psk_src: pf_rule_addr,
    psk_dst: pf_rule_addr,
    psk_ifname: [u8; IFNAMSIZ],
    psk_label: [u8; PF_RULE_LABEL_SIZE],
    psk_killed: libc::c_uint,
}


impl pfioc_state_kill
{
    unsafe fn new<H: AsRef<str>>(fd: libc::c_int, is_test: bool, src: H, dest: Option<H>) -> PfResult<()>
    {
        let host = src.as_ref();

        // psk does not use references, zeroed is safe to use because 
        // psk contains psk_ifname, psk_label which are string ended with 0
        let mut psk: Self = std::mem::zeroed();

        //psk.psk_src.addr.v.a = pf_addr_wrap_1::new_zeroed();
        psk.psk_src.addr.v.a.mask.pfa.addr32 = [0xFFFFFFFF_u32; 4];

        psk.psk_src.addr.v.a.mask.pfctl_addrprefix(host)?;


        let mut res: *mut libc::addrinfo = std::ptr::null_mut();

        let c = CString::new(host.to_string()).map_err(|e| map_error_coded!(PfErrCode::InternalError, "{}", e))?;
        let ret_ga = libc::getaddrinfo(c.as_ptr(), std::ptr::null(), std::ptr::null(), &mut res);

        if ret_ga > 0
        {
            error_coded!(PfErrCode::LibcGetAddrInfo(ret_ga), "getaddrinfo: {}", CStr::from_ptr(libc::gai_strerror(ret_ga)).to_string_lossy());
        }
        else if res == std::ptr::null_mut()
        {
            error_coded!(PfErrCode::RuntimeAssertion, "getaddrinfo: returned NULL");
        }

        // sockaddr does not contains refereces so it is safe to use zeroed()
        let mut last_src: libc::sockaddr = std::mem::zeroed();
            libc::memset(&mut last_src as *mut _ as *mut libc::c_void, 0xff, std::mem::size_of::<libc::sockaddr>());
        let mut last_dst: libc::sockaddr = std::mem::zeroed();
            libc::memset(&mut last_dst as *mut _ as *mut libc::c_void, 0xff, std::mem::size_of::<libc::sockaddr>());

        let mut sources: libc::c_int = 0;
        let mut killed: u32 = 0;
        let mut dests: libc::c_int = 0;

        
        cfor!(let mut p_resp = res; p_resp != std::ptr::null_mut(); p_resp = (*p_resp).ai_next;
            {
                let resp = *p_resp;

                if resp.ai_addr == std::ptr::null_mut()
                {
                    continue;
                }

                // We get lots of duplicates.  Catch the easy ones

                if 0 == libc::memcmp(
                        &last_src as *const _ as *const libc::c_void, 
                        resp.ai_addr as *const libc::c_void, 
                        std::mem::size_of::<libc::sockaddr>()
                    )
                {
                    continue;
                }

                last_src = *resp.ai_addr;

                psk.psk_af = resp.ai_family as libc::sa_family_t;// freebsd u8;
                sources += 1;

                if psk.psk_af as i32 == libc::AF_INET
                {
                    psk.psk_src.addr.v.a.addr.pfa.v4 = (*(resp.ai_addr as *const libc::sockaddr_in)).sin_addr;
                }
                else if psk.psk_af as i32 == libc::AF_INET6
                {
                    psk.psk_src.addr.v.a.addr.pfa.v6 = (*(resp.ai_addr as *const libc::sockaddr_in6)).sin6_addr;
                }
                else
                {
                    libc::freeaddrinfo(res);
                    error_coded!(PfErrCode::UnknownAddressFamily(psk.psk_af as i32), "unknown address family {}", psk.psk_af);
                }

                //C if (state_killers > 1) {
                if dest.is_some() == true
                {
                    let r_host2 = dest.as_ref().unwrap().as_ref();
                    dests = 0;

                    libc::memset(&mut psk.psk_dst.addr.v.a.mask as *mut _ as *mut libc::c_void, 0xff, std::mem::size_of_val(&psk.psk_dst.addr.v.a.mask));
                    libc::memset(&mut last_dst as *mut _ as *mut libc::c_void, 0xff, std::mem::size_of_val(&last_dst));

                    psk.psk_dst.addr.v.a.mask.pfctl_addrprefix(r_host2)?;

                    // init new addrinfo pointer stuct addrinfo * res1;
                    let mut res1: *mut libc::addrinfo = std::ptr::null_mut();

                    let c1 = CString::new(host.to_string()).map_err(|e| map_error_coded!(PfErrCode::InternalError, "{}", e))?;

                    // call getaddrinfo
                    let ret_ga = libc::getaddrinfo(c1.as_ptr(), std::ptr::null(), std::ptr::null(), &mut res1);

                    if ret_ga > 0
                    {
                        libc::freeaddrinfo(res);
                        error_coded!(PfErrCode::LibcGetAddrInfo(ret_ga), "getaddrinfo: {}", CStr::from_ptr(libc::gai_strerror(ret_ga)).to_string_lossy());
                    }
                    else if res1 == std::ptr::null_mut()
                    {
                        libc::freeaddrinfo(res);
                        error_coded!(PfErrCode::RuntimeAssertion, "getaddrinfo: returned NULL");
                    }

                    // at this point is must be inited

                    cfor!(let mut p_resp1 = res1; p_resp1 != std::ptr::null_mut(); p_resp1 = (*p_resp1).ai_next;
                        {
                            let resp1 = *p_resp1;

                            if resp1.ai_addr == std::ptr::null_mut()
                            {
                                continue;
                            }

                            if psk.psk_af as i32 != resp1.ai_family
                            {
                                continue;
                            }

                            // We get lots of duplicates.  Catch the easy ones
                            if 0 == libc::memcmp(
                                    &last_dst as *const _ as *const libc::c_void, 
                                    resp1.ai_addr as *const libc::c_void, 
                                    std::mem::size_of::<libc::sockaddr>()
                                )
                            {
                                continue;
                            }

                            last_dst = *resp1.ai_addr;
                            dests += 1;

                            if psk.psk_af as i32 == libc::AF_INET
                            {
                                psk.psk_dst.addr.v.a.addr.pfa.v4 = (*(resp1.ai_addr as *const libc::sockaddr_in)).sin_addr;
                            }
                            else if psk.psk_af as i32 == libc::AF_INET6
                            {
                                psk.psk_dst.addr.v.a.addr.pfa.v6 = (*(resp1.ai_addr as *const libc::sockaddr_in6)).sin6_addr;
                            }
                            else
                            {
                                libc::freeaddrinfo(res);
                                libc::freeaddrinfo(res1);
                                error_coded!(PfErrCode::UnknownAddressFamily(psk.psk_af as i32), "unknown address family {}", psk.psk_af);
                            }

                            if is_test == false
                            {
                                if libc::ioctl(fd, IoCtl::DIOCKILLSTATES, &mut psk as *mut _, std::ptr::null::<libc::c_void>()) != 0
                                {
                                    //pfr_report_error(tbl, &io, "add table");
                                    libc::freeaddrinfo(res);
                                    libc::freeaddrinfo(res1);

                                    error_coded!(PfErrCode::IoCtlError("DIOCKILLSTATES", IoCtl::DIOCKILLSTATES), "ioctl DIOCKILLSTATES failed");
                                }
                            }
                            else
                            {
                                println!("Fake ioctl DIOCKILLSTATES: {}\n{:?}", IoCtl::DIOCKILLSTATES, psk);
                            }

                            killed += psk.psk_killed;

                        }
                    );

                    libc::freeaddrinfo(res1);
                }
                else
                {
                    if is_test == false
                    {
                        if libc::ioctl(fd, IoCtl::DIOCKILLSTATES, &mut psk as *mut _, std::ptr::null::<libc::c_void>()) != 0
                        {
                            //pfr_report_error(tbl, &io, "add table");
                            libc::freeaddrinfo(res);

                            error_coded!(PfErrCode::IoCtlError("DIOCKILLSTATES", IoCtl::DIOCKILLSTATES), "ioctl DIOCKILLSTATES failed");
                        }
                    }
                    else
                    {
                        println!("Fake ioctl DIOCKILLSTATES: {}\n{:?}", IoCtl::DIOCKILLSTATES, psk);
                    }

                    killed += psk.psk_killed;
                }
            }
        );

        libc::freeaddrinfo(res);
        
        #[cfg(feature = "log_to_stdout")]
        println!("killed {} states from {} sources and {} destinations", killed, sources, dests);

        return Ok(());
    }
}

#[derive(Debug)]
#[repr(C)]
struct pfioc_table
{
    pfrio_table: pfr_table,
    pfrio_buffer: *mut libc::c_void,
    pfrio_esize: libc::c_int,
    pfrio_size: libc::c_int,
    pfrio_size2: libc::c_int,
    pfrio_nadd: libc::c_int,
    pfrio_ndel: libc::c_int,
    pfrio_nchange: libc::c_int,
    pfrio_flags: libc::c_int,
    pfrio_ticket: libc::c_uint,
}

impl pfioc_table
{
    unsafe fn new_add_tables(tbl: pfr_table, size: libc::c_int, flags: libc::c_int) -> Self
    {
        let mut io: Self = std::mem::zeroed();

        io.pfrio_flags = flags;
        io.pfrio_buffer = Box::into_raw(Box::new(tbl)) as *mut libc::c_void;//tbl as *mut libc::c_void;
        io.pfrio_esize = std::mem::size_of::<pfr_table>() as libc::c_int;
        io.pfrio_size = size;

        return io;
    }

    unsafe fn new_addrs(tbl: pfr_table, addr: *mut libc::c_void, size: libc::c_int, flags: libc::c_int) -> Self
    {
        let mut io: Self = std::mem::zeroed();

        io.pfrio_flags = flags;
        io.pfrio_table = tbl;
        io.pfrio_buffer = addr as *mut libc::c_void;
        io.pfrio_esize = std::mem::size_of::<pfr_addr>() as libc::c_int;
        io.pfrio_size = size;

        return io;
    }

    unsafe fn new_flush(tbl: pfr_table, flags: libc::c_int) -> Self
    {
        let mut io: Self = std::mem::zeroed();
        
        io.pfrio_flags = flags;
        io.pfrio_table = tbl;

        return io;
    }
}

struct IoCtl{}
impl IoCtl
{
    /// no parameters
    const IOC_VOID: libc::c_ulong = 0x20000000;
    /// copy out parameters
    const IOC_OUT: libc::c_ulong = 0x40000000;
    /// copy in parameters
    const IOC_IN: libc::c_ulong = 0x80000000;
    /// copy parameters in and out
    const IOC_INOUT: libc::c_ulong = (Self::IOC_IN|Self::IOC_OUT);
    /// mask for IN/OUT/VOID
    const IOC_DIRMASK: libc::c_ulong = (Self::IOC_VOID|Self::IOC_OUT|Self::IOC_IN);

    /// number of bits for ioctl size
    const IOCPARM_SHIFT: libc::c_ulong = 13;
    /// parameter length mask
    const IOCPARM_MASK: libc::c_ulong = ((1 << Self::IOCPARM_SHIFT) - 1);

    fn IOCPARM_LEN(x: libc::c_ulong) -> libc::c_ulong
    {
        return ((x) >> 16) & Self::IOCPARM_MASK;
    }

    fn IOCBASECMD(x: libc::c_ulong) -> libc::c_ulong
    {
        return (x) & !(Self::IOCPARM_MASK << 16);
    }	

    fn IOCGROUP(x: libc::c_uint) -> libc::c_uint
    {
        return ((x) >> 8) & 0xff;
    }

    /// returns unsigned long = 8
    const fn iowr(group: char, num: libc::c_ulong, len: usize) -> libc::c_ulong
    {   
        return Self::IOC_INOUT | (((len as libc::c_ulong) & Self::IOCPARM_MASK) << 16) | ((group as libc::c_ulong) << 8) | (num);
    }

    const DIOCRADDTABLES: libc::c_ulong = Self::iowr('D', 61, std::mem::size_of::<pfioc_table>());
    const DIOCRADDADDRS: libc::c_ulong = Self::iowr('D', 67, std::mem::size_of::<pfioc_table>());
    const DIOCRDELADDRS: libc::c_ulong = Self::iowr('D', 68, std::mem::size_of::<pfioc_table>());
    const DIOCRTSTADDRS: libc::c_ulong = Self::iowr('D', 73, std::mem::size_of::<pfioc_table>());
    const DIOCKILLSTATES: libc::c_ulong = Self::iowr('D', 41, std::mem::size_of::<pfioc_state_kill>());
    const DIOCRCLRADDRS: libc::c_ulong = Self::iowr('D', 66, std::mem::size_of::<pfioc_table>());
}

#[derive(Debug, Clone, Copy)]
#[repr(C)]
struct pfr_table 
{
    pub pfrt_anchor: [u8; MAXPATHLEN], // libc::c_char
    pub pfrt_name: [u8; PF_TABLE_NAME_SIZE], // libc::c_char
    pub pfrt_flags: u32, // u_int32_t
    pub pfrt_fback: u8, // u_int8_t
}

impl pfr_table
{
    const PFR_TFLAG_PERSIST: libc::c_uint = 0x00000001;
    const PFR_TFLAG_CONST: libc::c_uint = 0x00000002;
    const PFR_TFLAG_ACTIVE: libc::c_uint = 0x00000004;
    const PFR_TFLAG_INACTIVE: libc::c_uint = 0x00000008;
    const PFR_TFLAG_REFERENCED: libc::c_uint = 0x00000010;
    const PFR_TFLAG_REFDANCHOR: libc::c_uint = 0x00000020;
    const PFR_TFLAG_COUNTERS: libc::c_uint = 0x00000040;

    unsafe 
    fn new(pfrt_name: &str) -> PfResult<Self>
    {
        if pfrt_name.len() >= PF_TABLE_NAME_SIZE-1
        {
            error_coded!(PfErrCode::Einval, "pfrt_name: {} too long: {} >= {}", pfrt_name, pfrt_name.len(), PF_TABLE_NAME_SIZE-1);
        }

        let c_pfrt_name = CString::new(pfrt_name.to_string()).map_err(|e| map_error_coded!(PfErrCode::InternalError, "{}", e))?;
        let len = c_pfrt_name.to_bytes().len();

        let mut table = 
            pfr_table
            {
                pfrt_anchor: [0_u8; MAXPATHLEN],
                pfrt_name: [0_u8; PF_TABLE_NAME_SIZE],
                pfrt_flags: 0,
                pfrt_fback: 0,
            };

        table.pfrt_name[0..len].copy_from_slice(c_pfrt_name.to_bytes());

        return Ok(table);
    }

    unsafe 
    fn pfr_add_tables(&mut self, fd: libc::c_int, size: libc::c_int, nadd: &mut libc::c_int, flags: libc::c_int, test: bool) -> PfResult<()>
    {
        if size < 0
        {
            error_coded!(PfErrCode::Einval, "pfr_add_tables() EINVAL size: {}", size);
        }

        let mut io = pfioc_table::new_add_tables(self.clone(), size, flags);

        if test == false
        {
            if libc::ioctl(fd, IoCtl::DIOCRADDTABLES, &mut io as *mut _, std::ptr::null::<libc::c_void>()) != 0
            {
                //pfr_report_error(tbl, &io, "add table");
                error_coded!(PfErrCode::IoCtlError("DIOCRADDTABLES", IoCtl::DIOCRADDTABLES), "ioctl DIOCRADDTABLES failed");
            }
        }
        else
        {
            println!("Fake IOCTL DIOCRADDTABLES:\n{:?}", io);
        }

        *nadd = io.pfrio_nadd;

        return Ok(());
    }

    unsafe 
    fn pfr_add_addrs(
        &mut self, 
        fd: libc::c_int, 
        mut b: PfrBuffer, 
        nadd: &mut libc::c_int, 
        flags: libc::c_int,
        test: bool,
    ) -> PfResult<()>
    {   
        if b.pfrb_caddr.len() == 0
        {
            error_coded!(PfErrCode::Einval, "pfr_add_addrs() EINVAL size: {}", b.pfrb_caddr.len());
        }

        let mut io = 
            pfioc_table::new_addrs(
                self.clone(), 
                b.pfrb_caddr.as_mut_ptr() as *mut libc::c_void, 
                b.pfrb_caddr.len() as libc::c_int, 
                flags
            );    
        
        if test == false
        {
            if libc::ioctl(fd, IoCtl::DIOCRADDADDRS, &mut io as *mut _, std::ptr::null::<libc::c_void>()) != 0
            {
                //pfr_report_error(tbl, &io, "add table");
                error_coded!(PfErrCode::IoCtlError("DIOCRADDADDRS", IoCtl::DIOCRADDADDRS), "ioctl DIOCRADDADDRS failed");
            }
        }
        else
        {
            println!("fake IOCTL DIOCRADDADDRS pfioc_table contains:\n{:?}", io);
        }

        *nadd = io.pfrio_nadd;

        return Ok(());
    }

    unsafe 
    fn pfr_del_addrs(&mut self, fd: libc::c_int, mut b: PfrBuffer, ndel: &mut libc::c_int, flags: libc::c_int) -> PfResult<()>
    {   
        if b.pfrb_caddr.len() == 0
        {
            error_coded!(PfErrCode::Einval, "pfr_del_addrs() EINVAL size: {}", b.pfrb_caddr.len());
        }

        let mut io = 
            pfioc_table::new_addrs(
                self.clone(), 
                b.pfrb_caddr.as_mut_ptr() as *mut libc::c_void, 
                b.pfrb_caddr.len() as libc::c_int, 
                flags
            );    
        
        if libc::ioctl(fd, IoCtl::DIOCRDELADDRS, &mut io as *mut _, std::ptr::null::<libc::c_void>()) != 0
        {
            //pfr_report_error(tbl, &io, "add table");
            error_coded!(PfErrCode::IoCtlError("DIOCRDELADDRS", IoCtl::DIOCRDELADDRS), "ioctl DIOCRDELADDRS failed");
        }

        *ndel = io.pfrio_ndel;

        return Ok(());
    }

    unsafe 
    fn pfr_tst_addrs(&mut self, fd: libc::c_int, mut b: PfrBuffer, nmatch: &mut libc::c_int, flags: libc::c_int) -> PfResult<()>
    {   
        if b.pfrb_caddr.len() == 0
        {
            error_coded!(PfErrCode::Einval, "pfr_tst_addrs() EINVAL size: {}", b.pfrb_caddr.len());
        }

        let mut io = 
            pfioc_table::new_addrs(
                self.clone(), 
                b.pfrb_caddr.as_mut_ptr() as *mut libc::c_void, 
                b.pfrb_caddr.len() as libc::c_int, 
                flags
            );    
        
        if libc::ioctl(fd, IoCtl::DIOCRTSTADDRS, &mut io as *mut _, std::ptr::null::<libc::c_void>()) != 0
        {
            //pfr_report_error(tbl, &io, "add table");
            error_coded!(PfErrCode::IoCtlError("DIOCRTSTADDRS", IoCtl::DIOCRTSTADDRS), "ioctl DIOCRDELADDRS failed");
        }

        *nmatch = io.pfrio_nadd; //C *nmatch = io.pfrio_nmatch; #define	pfrio_nmatch	pfrio_nadd

        return Ok(());
    }

    unsafe 
    fn pfr_clr_addrs(&mut self, fd: libc::c_int, flags: libc::c_int, ndel: &mut libc::c_int) -> PfResult<()>
    {
        let mut io = pfioc_table::new_flush(self.clone(), flags);

        if libc::ioctl(fd, IoCtl::DIOCRCLRADDRS, &mut io as *mut _, std::ptr::null::<libc::c_void>()) != 0
        {
            //pfr_report_error(tbl, &io, "add table");
            error_coded!(PfErrCode::IoCtlError("DIOCRCLRADDRS", IoCtl::DIOCRCLRADDRS), "ioctl DIOCRCLRADDRS failed");
        }

        *ndel = io.pfrio_ndel; //C *nmatch = io.pfrio_nmatch; #define	pfrio_nmatch	pfrio_nadd

        return Ok(());
    }

    unsafe 
    fn create_table(&mut self, fd: libc::c_int, test: bool) -> PfResult<()>
    {
        let mut nadd: libc::c_int = 0;
        let flags: libc::c_int = 0;

        self.pfrt_flags |= Self::PFR_TFLAG_PERSIST;

        self.pfr_add_tables(fd, 1, &mut nadd, flags, test)?;

        #[cfg(feature = "log_to_stdout")]
        if nadd > 0
        {
            println!("{} table created", nadd);
        }

        self.pfrt_flags &= !Self::PFR_TFLAG_PERSIST;

        return Ok(());
    }

    unsafe 
    fn add_addrs(&mut self, fd: libc::c_int, b: PfrBuffer, test: bool) -> PfResult<i32>
    {
        let mut nadd: libc::c_int = 0;
        let flags: libc::c_int = 0;

        let pfrb_size = b.pfrb_caddr.len();
        self.pfr_add_addrs(fd, b, &mut nadd, flags, test)?;

        #[cfg(feature = "log_to_stdout")]
        println!("{}/{} addresses added", nadd, pfrb_size);

        return Ok(nadd);
    }

    unsafe 
    fn del_addrs(&mut self, fd: libc::c_int, b: PfrBuffer) -> PfResult<i32>
    {
        let mut ndel: libc::c_int = 0;
        let flags: libc::c_int = 0;

        let pfrb_size = b.pfrb_caddr.len();
        self.pfr_del_addrs(fd, b, &mut ndel, flags)?;

        #[cfg(feature = "log_to_stdout")]
        println!("{}/{} addresses deleted", nadd, pfrb_size);

        return Ok(ndel);
    }

    unsafe 
    fn tst_addrs(&mut self, fd: libc::c_int, b: PfrBuffer) -> PfResult<i32>
    {
        let mut nmatch = 0;
        let flags: libc::c_int = 0;

        let pfrb_size = b.pfrb_caddr.len();
        self.pfr_tst_addrs(fd, b, &mut nmatch, flags)?;

        #[cfg(feature = "log_to_stdout")]
        println!("{}/{} addresses test", nmatch, pfrb_size);

        return Ok(nmatch);
    }

    unsafe 
    fn fls_addrs(&mut self, fd: libc::c_int) -> PfResult<i32>
    {
        let mut ndel = 0;
        let flags: libc::c_int = 0;

        self.pfr_clr_addrs(fd, flags, &mut ndel)?;

        #[cfg(feature = "log_to_stdout")]
        println!("{} addresses removed", ndel);

        return Ok(ndel);
    }

}

// TODO: switch to Vec and use as_mut_ptr to remove the malloc calls OK
// TODO: add enum for other types i.e pfr_addr ...
struct PfrBuffer 
{
    /// type of content
    pfrb_type: Pfrb,	
    
    // pfrb_size number of objects in buffer
    // pfrb_msize maximum number of objects in buffer

    /// rust allocated memory
    pfrb_caddr: Vec<pfr_addr>,//*mut libc::c_void,
}

impl Drop for PfrBuffer
{
    fn drop(&mut self) 
    {
        self.pfrb_caddr.clear();
    }
}

impl PfrBuffer
{
    fn new(pfrb_type: Pfrb) -> Self
    {
        return Self{ pfrb_type: pfrb_type, pfrb_caddr: Vec::new() };
    }

    //https://github.com/freebsd/freebsd-src/blob/27ab791a55191c0b6503391d411303b042b41047/sbin/pfctl/pfctl_parser.c :1787
    /// Convert a hostname to a list of addresses and put them in the [PfrBuffer]
    /// ard
    unsafe fn append_addr<'pfcmd>(&mut self, s: &'pfcmd str, is_test: libc::c_int) -> PfResult<()>
    {
        let mut not: libc::c_int = 0;

        let r = 
            if let Some(pos) = s.chars().position(|c| c == '!') // for (r = s; *r == '!'; r++)
            {
                not = !not; // not = !not;

                match s.char_indices().nth(pos+1)  // skip '!'
                {
                    None => error_coded!(PfErrCode::InternalError, "can not get char indeces at: '{}' for: '{}'", pos, s),
                    Some((idx, _)) => &s[idx..],
                }
            }
            else
            {
                s
            };
    

        let n = node_host::host(r)?; // if ((n = host(r)) == NULL) { errno = 0; return (-1); }

        self.append_addr_host(n, is_test, not)?; // rv = append_addr_host(b, n, test, not);

        return Ok(());
    }

    //https://github.com/freebsd/freebsd-src/blob/373ffc62c158e52cde86a5b934ab4a51307f9f2e/sbin/pfctl/pfctl_table.c : 416
    unsafe fn load_addr_from_str<H: AsRef<str>>(&mut self, s: H, is_test: libc::c_int) -> PfResult<()>
    {
        //while (argc--)
        self.append_addr(s.as_ref(), is_test)?;

        //todo (from file if needed)
        /*if (pfr_buf_load(b, file, nonetwork, append_addr)) 
        {
            warn("cannot load %s", file);
            return (-1);
        }*/

        return Ok(());
    }

    
    unsafe fn load_addr_from_file<P: AsRef<Path>>(&mut self, path: P, is_test: libc::c_int) -> PfResult<()>
    {

        let s_path = path.as_ref();

        if s_path.as_os_str().len() == 0
        {
            #[cfg(feature = "log_to_stdout")]
            println!("load_addr_from_file() path is empty!");

            return Ok(());
        }
        
        let mut stream: Box<dyn std::io::Read + 'static> = 
            if s_path.as_os_str() == "-"
            {
                // read from stdin
                Box::new(std::io::stdin())
            }
            else
            {   
                if s_path.exists() == true
                {
                    error_coded!(PfErrCode::IOErr, "path does not exist: {}", s_path.display());
                }
                else if s_path.is_dir() == true
                {
                    error_coded!(PfErrCode::IOErr, "path is dir: {}", s_path.display());
                }

                Box::new(fs::File::open(s_path).map_err(|e| map_error_coded!(PfErrCode::IOErr, "{}", e))?)
            };

        //read from stream all
        let mut file_buf = String::new();
        stream.read_to_string(&mut file_buf).map_err(|e| map_error_coded!(PfErrCode::IOErr, "{}", e))?;

        let token = PfListTokenizer::from_str(&file_buf)?;
        
        if token.is_none() == true
        {
            return Ok(());
        } 

        let mut token = token.unwrap();

        loop
        {
            let tok = token.read_next()?;

            match tok
            {
                Some(s) => 
                {
                    self.append_addr(s, is_test)?;
                },
                None => break,
            }
        }

        //while (argc--)
        

        //todo (from file if needed)
        /*if (pfr_buf_load(b, file, nonetwork, append_addr)) 
        {
            warn("cannot load %s", file);
            return (-1);
        }*/

        return Ok(());
    }
    

    // https://github.com/freebsd/freebsd-src/blob/373ffc62c158e52cde86a5b934ab4a51307f9f2e/sbin/pfctl/pfctl_radix.c#L457
    fn pfr_buf_add(&mut self, addr: pfr_addr) -> PfResult<()>
    {
        // this should not happen, because it is converted to the enum, but for compat.
        let pfrb_type = self.pfrb_type as i32;

        if pfrb_type <= 0 || pfrb_type >= Pfrb::PFRB_MAX as i32
        {
            error_coded!(PfErrCode::Einval, "pfrb_type unknown: {}", self.pfrb_type as i32);
        }

        self.pfrb_caddr.push(addr);

        return Ok(());
    }

    unsafe fn append_addr_host(&mut self, vn: Vec<node_host>, is_test: libc::c_int, not: libc::c_int) -> PfResult<()>
    {
        for n in vn
        {
            let addr = pfr_addr::new(n, is_test, not)?;

            self.pfr_buf_add(addr)?;
        }

        return Ok(());
    }
}

/*fn pfctl_fopen(file: &Path) -> PfResult<fs::File>
{
    if file.is_dir() == true
    {
        runtime_error!("path is dir: {}", file.display());
    }

    return Ok(fs::File::open(file).map_err(|e| map_runtime_error!("{}", e))?);
}*/


pub struct Pf
{
    fd: i32,
    test: bool
}

impl Drop for Pf
{
    fn drop(&mut self) 
    {
        if self.fd >= 0
        {
            unsafe { libc::close(self.fd) };
        }    
    }
}


impl Pf
{
    const DEFAULT_PF_DEVICE: &'static str = "/dev/pf";

    pub 
    fn new(test: bool) -> PfResult<Self>
    {
        let fd = 
            if test == false
            {
                open(Self::DEFAULT_PF_DEVICE, OFlag::O_NONBLOCK | OFlag::O_RDWR, Mode::empty())
                    .map_err(|e| map_error_coded!(PfErrCode::IOErr, "open() failed path {} err: {}", Self::DEFAULT_PF_DEVICE, e))?
            }
            else
            {
                -1
            };

        return Ok(Self{ fd: fd, test: test });
    }

    // internal functions to be called from public interface
    // https://github.com/freebsd/freebsd-src/blob/373ffc62c158e52cde86a5b934ab4a51307f9f2e/sbin/pfctl/pfctl.c#L638
    pub unsafe 
    fn pfctl_kill_state<H: AsRef<str>>(&self, src: H, dest: Option<H>) -> PfResult<()>
    {
        return pfioc_state_kill::new(self.fd, self.test, src, dest);
    }

    pub unsafe 
    fn pfctl_table<T: AsRef<str>>(&self, table_name: T, cmd: PfCmd) -> PfResult<i32>
    {
        let tname = table_name.as_ref();

        let mut table: pfr_table = pfr_table::new(tname)?;
        //let mut b: PfrBuffer = PfrBuffer::new();
        //let mut b2: PfrBuffer = PfrBuffer::new();

        if tname.len() >= PF_TABLE_NAME_SIZE
        {
            error_coded!(PfErrCode::Einval, "table name: '{}' is too long!", tname);
        }

        // verify that onlu ASCII chars are in tname
        if tname.chars().all(|c| c.is_ascii()) == false
        {
            error_coded!(PfErrCode::Einval, "non ascii character in table_name: {}", tname);
        }

        //table.pfrt_name.copy_from_slice(tname.as_bytes());

        // https://github.com/freebsd/freebsd-src/blob/373ffc62c158e52cde86a5b934ab4a51307f9f2e/sbin/pfctl/pfctl_table.c :201
        let ret_val = 
            match cmd
            {
                PfCmd::Add{ hosts } =>
                {
                    let mut b: PfrBuffer = PfrBuffer::new(Pfrb::PFRB_ADDRS);
                    //b.pfrb_type = Pfrb::PFRB_ADDRS as libc::c_int; //C b.pfrb_type = PFRB_ADDRS;

                    for host in hosts
                    {
                        b.load_addr_from_str(host, 0)?; //C if (load_addr(&b, argc, argv, file, 0))
                    }

                    table.create_table(self.fd, self.test)?;

                    table.add_addrs(self.fd, b, self.test)?
                },
                PfCmd::Delete{ hosts } => 
                {
                    let mut b: PfrBuffer = PfrBuffer::new(Pfrb::PFRB_ADDRS);
                    //b.pfrb_type = Pfrb::PFRB_ADDRS as libc::c_int; //C b.pfrb_type = PFRB_ADDRS;

                    for host in hosts
                    {
                        b.load_addr_from_str(host, 0)?; //C if (load_addr(&b, argc, argv, file, 0))
                    }

                    table.del_addrs(self.fd, b)?
                },
                PfCmd::Test{ hosts } => 
                {
                    let mut b: PfrBuffer = PfrBuffer::new(Pfrb::PFRB_ADDRS);
                    //b.pfrb_type = Pfrb::PFRB_ADDRS as libc::c_int; //C b.pfrb_type = PFRB_ADDRS;
                    //b2.pfrb_type = Pfrb::PFRB_ADDRS as libc::c_int; //C b.pfrb_type = PFRB_ADDRS;

                    for host in hosts
                    {
                        b.load_addr_from_str(host, 1)?; //C if (load_addr(&b, argc, argv, file, 1))
                    }

                    table.tst_addrs(self.fd, b)?
                },
                PfCmd::Flush => 
                {
                    table.fls_addrs(self.fd)?
                }
            }; // match

        return Ok(ret_val);
    }

    
}

#[test]
fn test_real_add()
{
    let pf = Pf::new(false).unwrap();

    unsafe { pf.pfctl_table("test", PfCmd::Add{ hosts: vec!["192.168.2.1".to_string()] }).unwrap() };
}

#[test]
fn test_real_check()
{
    let pf = Pf::new(false).unwrap();

    unsafe { pf.pfctl_table("test", PfCmd::Test{ hosts: vec!["192.168.2.1".to_string()] }).unwrap() };
}

#[test]
fn test_real_del() 
{
    let pf = Pf::new(false).unwrap();

    unsafe { pf.pfctl_table("test", PfCmd::Delete{ hosts: vec!["192.168.2.1".to_string()] }).unwrap() };
}

#[test]
fn test_kill_state()
{
    let pf = Pf::new(false).unwrap();

    unsafe { pf.pfctl_kill_state("192.168.2.104", None).unwrap() };
}

#[test]
fn fake_test_real_kill_state()
{
    let pf = Pf::new(true).unwrap();

    unsafe { pf.pfctl_kill_state("192.168.2.1", None).unwrap() };
}

#[test]
fn fake_test_add()
{
    let pf = Pf::new(true).unwrap();

    unsafe { pf.pfctl_table("test", PfCmd::Add{ hosts: vec!["192.168.2.1".to_string()] }).unwrap() };
}

#[test]
fn fake_test2_add()
{
    let pf = Pf::new(true).unwrap();

    unsafe { pf.pfctl_table("test", PfCmd::Add{ hosts: vec!["192.168.2.0/24".to_string()] }).unwrap() };
}

#[test]
fn test_struct_sizes()
{
    let s = std::mem::size_of::<pfioc_table>();
    println!("sizeof(struct pfioc_table = 1104) ?= {}", s);
    assert_eq!(1104, s);

    let s = std::mem::size_of::<pfr_table>();
    println!("sizeof(struct pfr_table = 1064) ?= {}", s);
    assert_eq!(1064, s);

    let s = std::mem::size_of::<pfioc_state_kill>();
    println!("sizeof(struct pfioc_state_kill = 224) ?= {}", s);
    assert_eq!(224, s);
    
    let psk: pfioc_state_kill = unsafe {std::mem::zeroed()};
    let s = std::mem::size_of_val(&psk.psk_af);
    println!("sizeof(psk.psk_af) = {}", s);
    assert_eq!(1, s);

    let s = std::mem::size_of_val(&psk.psk_src);
    println!("sizeof(psk.psk_src) = {}", s);
    assert_eq!(56, s);

    let s = std::mem::size_of_val(&psk.psk_dst);
    println!("sizeof(psk.psk_dst) = {}", std::mem::size_of_val(&psk.psk_dst));
    assert_eq!(56, s);

    let s = std::mem::size_of_val(&psk.psk_ifname);
    println!("sizeof(psk.psk_ifname) = {}", std::mem::size_of_val(&psk.psk_ifname));
    assert_eq!(16, s);

    let s = std::mem::size_of_val(&psk.psk_label);
    println!("sizeof(psk.psk_label) = {}", std::mem::size_of_val(&psk.psk_label));
    assert_eq!(64, s);

    let s = std::mem::size_of_val(&psk.psk_killed);
    println!("sizeof(psk.psk_killed) = {}", std::mem::size_of_val(&psk.psk_killed));
    assert_eq!(4, s);

    let s = std::mem::size_of_val(&psk.psk_proto);
    println!("sizeof(psk.psk_proto) = {}", std::mem::size_of_val(&psk.psk_proto));
    assert_eq!(4, s);

    let s = std::mem::size_of_val(&psk.psk_pfcmp);
    println!("sizeof(psk.psk_pfcmp) = {}", std::mem::size_of_val(&psk.psk_pfcmp));
    assert_eq!(16, s);

    let s = std::mem::size_of_val(&psk.psk_src.addr);
    println!("sizeof(psk.psk_src.addr) = {}", std::mem::size_of_val(&psk.psk_src.addr));
    assert_eq!(48, s);

    let s = std::mem::size_of_val(&psk.psk_src.addr.v);
    println!("sizeof(psk.psk_src.addr.v) = {}", std::mem::size_of_val(&psk.psk_src.addr.v));
    assert_eq!(32, s);

    let s = std::mem::size_of_val(&psk.psk_src.addr.p);
    println!("sizeof(psk.psk_src.addr.p) = {}", std::mem::size_of_val(&psk.psk_src.addr.p));
    assert_eq!(8, s);

    let s = std::mem::size_of_val(&psk.psk_src.addr.tp);
    println!("sizeof(psk.psk_src.addr.tp) = {}", std::mem::size_of_val(&psk.psk_src.addr.tp));
    assert_eq!(1, s);

    let s = std::mem::size_of_val(&psk.psk_src.addr.iflags);
    println!("sizeof(psk.psk_src.addr.iflags) = {}", std::mem::size_of_val(&psk.psk_src.addr.iflags));
    assert_eq!(1, s);
}

/*
DIOCKILLSTATES: 3235922985 sizeof(224)
psk.psk_af size: 1
psk.psk_src size: 56
psk.psk_dst size: 56
psk.psk_ifname size: 16
psk.psk_label size: 64
psk.psk_killed size: 4
psk.psk_proto size: 4
psk.psk_pfcmp size: 16
psk.psk_src.addr size: 48
psk.psk_src.addr size: 48

psk.psk_src.addr.v size: 32
psk.psk_src.addr.p size: 8
psk.psk_src.addr.type size: 1
psk.psk_src.addr.iflags size: 1

*/
#[test]
fn test_unmask()
{
    use std::net::Ipv4Addr;

    fn create(ip: &'static str) -> libc::c_int
    {
        let ipv4: Ipv4Addr = ip.parse().unwrap();

        let ip = pf_addr::from(ipv4.octets());

        return unsafe {ip.unmask(0)};
    }


    assert_eq!(create("255.255.255.255"), 32);
    assert_eq!(create("192.168.1.1"), 2);
    assert_eq!(create("10.0.0.0"), 0);
    assert_eq!(create("10.5.6.3"), 0);
    assert_eq!(create("134.156.43.230"), 1);
    assert_eq!(create("8.8.8.8"), 0);
    assert_eq!(create("244.255.255.255"), 4);
    assert_eq!(create("0.0.0.0"), 0);
    assert_eq!(create("100.200.255.255"), 0);
}

#[test]
fn test_set_ipmask()
{
    unsafe
    {
        let vh = node_host::host("192.168.2.1").unwrap();

        let h = &vh[0];
        assert_eq!(h.af, 2);
        assert_eq!(h.addr.v.a.addr.pfa.addr32[0], 16951488);
        assert_eq!(h.addr.v.a.mask.pfa.addr32[0], 4294967295);
    }
}

#[test]
fn test_set_ipmask_cidr()
{
    unsafe
    {
        let vh = node_host::host("192.168.2.0/24").unwrap();

        let h = &vh[0];

        assert_eq!(h.af, 2);
        assert_eq!(h.addr.v.a.addr.pfa.addr32[0], 174272);
        assert_eq!(h.addr.v.a.mask.pfa.addr32[0], 16777215);
    }
}

