/*-
 * SPDX-License-Identifier: BSD-2-Clause
 *
 * Copyright (c) 2021 Aleksandr Morozov, RELKOM s.r.o
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials provided
 *      with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

use std::iter::Peekable;
use std::str::{Chars};

use crate::{runtime_error, runtime_exception::*};

pub(crate) struct PfListTokenizer<'token>
{
    srcmsg: &'token str,
    chars: Peekable<Chars<'token>>,
    pos: usize,
    curchar: Option<char>,
}

impl<'token> PfListTokenizer<'token>
{
    pub fn from_str(text_buf: &'token str) -> PfResult<Option<Self>>
    {
        let mut chars: Peekable<Chars> = text_buf.chars().peekable();
        let cur: Option<char>;

        match chars.next()
        {
            Some(r) =>
            {
                cur = Some(r);
            },
            None => runtime_error!("unexpected EOF"),
        }  

        let inst = 
            Self
            {
                srcmsg: text_buf,
                chars: chars,
                pos: 0,
                curchar: cur,
            };

        return Ok(Some(inst));
    }

    #[inline] 
    fn move_next(&mut self)
    {
        self.pos += 1;

        self.curchar = self.chars.next();
    }

    #[inline]
    fn get_cur_char(&self) -> Option<char>
    {
        return self.curchar;
    }

    #[inline]
    fn foresee_char(&mut self) -> Option<char>
    {
        return match self.chars.peek()
        {
            Some(c) => Some(*c),
            None => None
        };
    }

    #[inline]
    fn is_eof(&mut self) -> bool
    {
        return self.curchar.is_none();
    }

    pub fn read_next(&mut self) -> PfResult<Option<&str>>
    {
        loop
        {
            // skip spaces
            while self.is_eof() == false && 
                    self.get_cur_char().unwrap().is_whitespace() == true
            {
                self.move_next();
            }

            if self.is_eof() == true
            {
                return Ok(None);
            }

            if self.get_cur_char().unwrap() == '#'
            {
                self.move_next();

                while let Some(c) = self.get_cur_char()
                {
                    if c == '\n'
                    {
                        self.move_next();
                        break; // break from while
                    }
                }
            }
            else
            {
                // break from loop
                break;
            }
        }

        let initpos = self.pos;

        loop
        {
            match self.get_cur_char()
            {
                None => break,
                Some(ref c) if c.is_whitespace() == true =>
                {
                    break;
                },
                Some(ref c) if c.is_control() == true =>
                {
                    self.move_next();
                    break;
                },
                Some(c) => 
                {
                    self.move_next();
                }
            }
        }

        let ret = &self.srcmsg[initpos..self.pos];

        return Ok(Some(ret));
    }
}

#[test]
fn test()
{
    let t = "192.168.2.1/32\n192.168.4.33\n   192.168.3.42   \n  192.168.6.200/31 \n \n test.domain.tld";

    let tok = PfListTokenizer::from_str(t).unwrap();

    assert_eq!(tok.is_some(), true);

    let mut tok = tok.unwrap();

    let one = tok.read_next().unwrap();
    assert_eq!(one.is_some(), true);
    assert_eq!(one.unwrap(), "192.168.2.1/32");

    let one = tok.read_next().unwrap();
    assert_eq!(one.is_some(), true);
    assert_eq!(one.unwrap(), "192.168.4.33");

    let one = tok.read_next().unwrap();
    assert_eq!(one.is_some(), true);
    assert_eq!(one.unwrap(), "192.168.3.42");

    let one = tok.read_next().unwrap();
    assert_eq!(one.is_some(), true);
    assert_eq!(one.unwrap(), "192.168.6.200/31");

    let one = tok.read_next().unwrap();
    assert_eq!(one.is_some(), true);
    assert_eq!(one.unwrap(), "test.domain.tld");
}
