/********************************************************************************
 *   Prometheus exporter for monitoring network connectivity using icmp pings   *
 *                                                                              *
 *   Copyright (C) 2019-2020 Jan Christian Grünhage                             *
 *   Copyright (C) 2020 Famedly GmbH                                            *
 *                                                                              *
 *   This program is free software: you can redistribute it and/or modify       *
 *   it under the terms of the GNU Affero General Public License as             *
 *   published by the Free Software Foundation, either version 3 of the         *
 *   License, or (at your option) any later version.                            *
 *                                                                              *
 *   This program is distributed in the hope that it will be useful,            *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               *
 *   GNU Affero General Public License for more details.                        *
 *                                                                              *
 *   You should have received a copy of the GNU Affero General Public License   *
 *   along with this program.  If not, see <https://www.gnu.org/licenses/>.     *
 ********************************************************************************/
use crate::config::Config;
use anyhow::{Context, Result};
use hyper::{
    header::CONTENT_TYPE,
    server::Server,
    service::{make_service_fn, service_fn},
    Body, Request, Response,
};
use lazy_static::lazy_static;
use log::info;
use prometheus::*;
use prometheus::{Counter, Gauge, HistogramVec, TextEncoder};

lazy_static! {
    static ref HTTP_COUNTER: Counter = register_counter!(opts!(
        "http_requests_total",
        "Total number of HTTP requests made.",
        labels! {"handler" => "all",}
    ))
    .unwrap();
    static ref HTTP_BODY_GAUGE: Gauge = register_gauge!(opts!(
        "http_response_size_bytes",
        "The HTTP response sizes in bytes.",
        labels! {"handler" => "all",}
    ))
    .unwrap();
    static ref HTTP_REQ_HISTOGRAM: HistogramVec = register_histogram_vec!(
        "http_request_duration_seconds",
        "The HTTP request latencies in seconds.",
        &["handler"]
    )
    .unwrap();
}

async fn serve_req(req: Request<Body>) -> Result<Response<Body>> {
    match req.uri().path() {
        "/metrics" => serve_metrics().await,
        "/health" => serve_health_check().await,
        _ => serve_not_found().await,
    }
}

async fn serve_metrics() -> Result<Response<Body>> {
    let encoder = TextEncoder::new();

    HTTP_COUNTER.inc();
    let timer = HTTP_REQ_HISTOGRAM.with_label_values(&["all"]).start_timer();

    let metric_families = prometheus::gather();
    let mut buffer = vec![];
    encoder.encode(&metric_families, &mut buffer).unwrap();
    HTTP_BODY_GAUGE.set(buffer.len() as f64);

    let response = Response::builder()
        .status(200)
        .header(CONTENT_TYPE, encoder.format_type())
        .body(Body::from(buffer))
        .context("Couldn't build metrics response")?;

    timer.observe_duration();

    Ok(response)
}

async fn serve_health_check() -> Result<Response<Body>> {
    Ok(Response::builder()
        .status(200)
        .body(Body::from(vec![]))
        .context("Couldn't build health check response")?)
}

async fn serve_not_found() -> Result<Response<Body>> {
    Ok(Response::builder()
        .status(404)
        .body(Body::from(vec![]))
        .context("Couldn't build not found response")?)
}

pub(crate) async fn start_serving_metrics(config: &Config) -> Result<()> {
    let serve_future = Server::bind(&config.listener).serve(make_service_fn(|_| async {
        Ok::<_, hyper::Error>(service_fn(serve_req))
    }));
    info!("Listening on {}", &config.listener);
    Ok(serve_future.await?)
}
