use persy::{Config, Persy};
use persy_expimp::{export_json, import_json, Error};
use std::fs::OpenOptions;
use std::io::{stdin, stdout};
use std::path::PathBuf;
use structopt::StructOpt;

#[derive(StructOpt)]
#[structopt(
    name = "persy_expimp",
    author = "tglman",
    about = "export or import data from a persy storage"
)]
struct Cli {
    #[structopt(long, short, default_value = "json")]
    format: String,

    #[structopt(long, short)]
    import: bool,

    #[structopt(long, short)]
    export: bool,

    #[structopt(parse(from_os_str))]
    persy_file: PathBuf,

    #[structopt(parse(from_os_str))]
    file: Option<PathBuf>,
}

pub fn main() -> Result<(), Error> {
    let cli = Cli::from_args();

    assert_eq!(cli.format, "json", "only supported JSON format");

    if cli.export {
        let persy = Persy::open(cli.persy_file.to_str().unwrap(), Config::new())?;
        if let Some(f) = cli.file {
            let mut output = OpenOptions::new().write(true).create(true).open(f)?;
            export_json(&persy, &mut output)?;
        } else {
            export_json(&persy, &mut stdout())?;
        }
    } else if cli.import {
        Persy::create(cli.persy_file.to_str().unwrap())?;
        let persy = Persy::open(cli.persy_file.to_str().unwrap(), Config::new())?;
        if let Some(f) = cli.file {
            let mut input = OpenOptions::new().read(true).open(f)?;
            import_json(&persy, &mut input)?;
        } else {
            import_json(&persy, &mut stdin())?;
        }
    } else {
        println!("please specify if export or import");
    }

    Ok(())
}
