use crate::{
    device::{free_flag_set, is_free, Page, ReadPage, SizeTool, UpdateList, FLAGS_OFFSET, NEXT_OFFSET, PREV_OFFSET},
    error::PERes,
    io::{read_u64, write_u64, ReadFormat, WriteFormat},
};
use std::{
    io::{Read, Seek, SeekFrom, Write},
    sync::Arc,
};

pub(crate) fn load_page<T: Read + Seek>(fl: &mut T, page: u64) -> PERes<ReadPage> {
    // add 2 to skip the metadata
    fl.seek(SeekFrom::Start(page))?;
    let exp = fl.read_u8()?;
    let size = 1 << exp; //EXP - (size_exp+size_mitigator);
    let mut ve = vec![0u8; size];
    ve[0] = exp;
    fl.read_exact(&mut ve[1..size])?;
    Ok(ReadPage::new(Arc::new(ve), 2, page, exp))
}

pub(crate) fn create_page_raw<T: Write + Seek>(fl: &mut T, exp: u8) -> PERes<u64> {
    let size = 1 << exp; //EXP - (size_exp+size_mitigator);
    let ve = vec![0u8; size];
    let offset = fl.seek(SeekFrom::End(0))?;
    fl.write_all(&ve)?;
    Ok(offset)
}

pub(crate) fn load_page_raw<T: Read + Seek>(fl: &mut T, page: u64, size_exp: u8) -> PERes<Page> {
    let mut ve;
    let size;
    {
        fl.seek(SeekFrom::Start(page))?;
        size = 1 << size_exp; //EXP - (size_exp+size_mitigator);
        ve = vec![0u8; size];
        fl.read_exact(&mut ve[0..size])?;
    }
    Ok(Page::new(ve, 0, page, size_exp))
}

pub(crate) fn flush_page<T: Write + Seek>(fl: &mut T, page: &Page) -> PERes<()> {
    fl.seek(SeekFrom::Start(page.index))?;
    fl.write_all(&page.buff)?;
    Ok(())
}

pub(crate) fn create_page<T: Write + Seek>(fl: &mut T, exp: u8) -> PERes<Page> {
    let size = 1 << exp; //EXP - (size_exp+size_mitigator);
    let mut ve = vec![0u8; size];
    ve[0] = exp;
    ve[size - 1] = exp;
    let offset = fl.seek(SeekFrom::End(0))?;
    fl.write_all(&ve)?;
    Ok(Page::new(ve, 2, offset, exp))
}

pub(crate) fn mark_allocated<T: Write + Read + Seek>(fl: &mut T, page: u64) -> PERes<u64> {
    fl.seek(SeekFrom::Start(page + FLAGS_OFFSET as u64))?;
    let mut moderator = fl.read_u8()?;
    fl.seek(SeekFrom::Start(page + NEXT_OFFSET as u64))?;
    // Free pages are a linked list reading the next page to return
    let next = fl.read_u64()?;
    moderator = free_flag_set(moderator, false);
    fl.seek(SeekFrom::Start(page + 1))?;
    fl.write_u8(moderator)?;
    fl.write_u64(0)?; // next
    fl.write_u64(0)?; // previous
    if next != 0 {
        fl.seek(SeekFrom::Start(next + PREV_OFFSET as u64))?;
        fl.write_u64(0)?;
    }
    Ok(next)
}

pub(crate) fn trim_or_free_page<T: Write + Read + Seek + SizeTool>(
    fl: &mut T,
    page: u64,
    update_list: &mut dyn UpdateList,
) -> PERes<()> {
    fl.seek(SeekFrom::Start(page))?;
    let exp = fl.read_u8()?;
    let mut mitigator = fl.read_u8()?;
    debug_assert!(!is_free(mitigator), "freeing: {} already freed ", page);
    let size = (1 << exp) as u64; //EXP - (size_exp+size_mitigator);
    if page + size == fl.len()? {
        fl.set_len(page)?;
        while check_and_trim(fl, update_list)? {}
    } else {
        let next = update_list.update(exp, page)?;
        fl.seek(SeekFrom::Start(page + 1))?;
        mitigator = free_flag_set(mitigator, true);
        let mut data: [u8; 9] = [0; 9];
        data[0] = mitigator;
        write_u64(&mut data[1..9], next);
        fl.seek(SeekFrom::Start(page + 1))?;
        fl.write_all(&data)?;
        fl.write_all(&[0u8; 8])?;

        if next != 0 {
            let mut data = [0; 8];
            write_u64(&mut data, page);
            fl.seek(SeekFrom::Start(next + PREV_OFFSET as u64))?;
            fl.write_all(&data)?;
        }
    }
    Ok(())
}

pub(crate) fn check_and_trim<T: Write + Read + Seek + SizeTool>(
    fl: &mut T,
    update_list: &mut dyn UpdateList,
) -> PERes<bool> {
    let len = fl.seek(SeekFrom::End(0))?;
    if len == 0 {
        return Ok(false);
    }

    fl.seek(SeekFrom::Start(len - 1))?;
    let exp = fl.read_u8()?;
    if exp == 0 {
        return Ok(false);
    }
    let size = (1 << exp) as u64;
    let mut header = [0u8; 18];
    let page = len - size;
    if page == 0 {
        return Ok(false);
    }
    fl.seek(SeekFrom::Start(page))?;
    fl.read_exact(&mut header)?;
    if is_free(header[1]) {
        let next = read_u64(&header[NEXT_OFFSET..(NEXT_OFFSET + 8)]);
        let prev = read_u64(&header[PREV_OFFSET..(PREV_OFFSET + 8)]);
        if prev == 0 {
            update_list.remove(exp, page, next)?;
        } else {
            let mut data = [0u8; 8];
            write_u64(&mut data, next);
            fl.seek(SeekFrom::Start(prev + NEXT_OFFSET as u64))?;
            fl.write_all(&data)?;
        }
        if next != 0 {
            let mut data = [0u8; 8];
            write_u64(&mut data, prev);
            fl.seek(SeekFrom::Start(next + PREV_OFFSET as u64))?;
            fl.write_all(&data)?;
        }
        fl.set_len(page)?;
        Ok(true)
    } else {
        Ok(false)
    }
}
