use serde_json::Value;
use std::fmt::{Display, Formatter};
use std::num::ParseIntError;
use std::str::FromStr;

use crate::SerializedEvent;

/// Used to upcast and event from an older type or version to the current form. This is needed
/// to modify the structure of events older versions are already persisted.
pub trait EventUpcaster: Send + Sync {
    /// Examines and event type and version to understand if the event should be upcasted.
    fn can_upcast(&self, event_type: &str, event_version: &str) -> bool;

    /// Modifies the serialized event to conform the the new structure.
    fn upcast(&self, event: SerializedEvent) -> SerializedEvent;
}

pub type SemanticVersionEventUpcasterFunc = dyn Fn(Value) -> Value + Send + Sync;

#[derive(Debug, PartialOrd, PartialEq)]
pub struct SemanticVersion {
    major_version: u32,
    minor_version: u32,
    patch: u32,
}

impl SemanticVersion {
    pub fn supersedes(&self, other: &SemanticVersion) -> bool {
        if other.major_version < self.major_version {
            return true;
        }
        if other.major_version == self.major_version {
            if other.minor_version < self.minor_version {
                return true;
            }
            if other.minor_version == self.minor_version && other.patch < self.patch {
                return true;
            }
        }
        false
    }
}

impl Display for SemanticVersion {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(
            f,
            "{}.{}.{}",
            self.major_version, self.minor_version, self.patch
        )
    }
}
impl FromStr for SemanticVersion {
    type Err = SemanticVersionError;

    fn from_str(event_version: &str) -> Result<Self, Self::Err> {
        let split_version: Vec<&str> = event_version.split('.').collect();
        if split_version.len() < 3 {
            return Err(SemanticVersionError);
        }
        let major_version = u32::from_str(split_version.get(0).unwrap())?;
        let minor_version = u32::from_str(split_version.get(1).unwrap())?;
        // TODO: account for '_', '-', additional values, etc.
        let patch = u32::from_str(split_version.get(2).unwrap())?;
        Ok(Self {
            major_version,
            minor_version,
            patch,
        })
    }
}

#[derive(Debug, PartialOrd, PartialEq)]
pub struct SemanticVersionError;

impl From<ParseIntError> for SemanticVersionError {
    fn from(_: ParseIntError) -> Self {
        Self
    }
}

/// This upcasts any event that has the same `event_type` and an `event_version` that is less than the
/// version configured on the upcaster.
///
/// ```
/// use persist_es::{EventUpcaster,SemanticVersionEventUpcaster};
/// use serde_json::Value;
/// use persist_es::SerializedEvent;
///
/// let upcast_function = Box::new(|payload: Value| match payload {
///             Value::Object(mut object_map) => {
///                 object_map.insert("country".to_string(), "USA".into());
///                 Value::Object(object_map)
///             }
///             _ => {
///                 panic!("the event payload is not an object")
///             }
///         });
/// let upcaster = SemanticVersionEventUpcaster::new("EventX", "2.3.4", upcast_function);
///
/// let payload: Value = serde_json::from_str(
///             r#"{
///                     "zip code": 98103,
///                     "state": "Washington"
///                    }"#,
///         ).unwrap();
///  let event = SerializedEvent::new(
///             "".to_string(),
///             0,
///             "".to_string(),
///             "".to_string(),
///             "".to_string(),
///             payload,
///             Default::default(),
///         );
/// let upcasted_event = upcaster.upcast(event);
///
/// let expected_payload: Value = serde_json::from_str(
///             r#"{
///                     "zip code": 98103,
///                     "state": "Washington",
///                     "country": "USA"
///                    }"#,
///         ).unwrap();
/// let expected_event = SerializedEvent::new(
///             "".to_string(),
///             0,
///             "".to_string(),
///             "".to_string(),
///             "2.3.4".to_string(),
///             expected_payload,
///             Default::default(),
///         );
///
/// assert_eq!(upcasted_event, expected_event);
/// ```
pub struct SemanticVersionEventUpcaster {
    event_type: String,
    event_version: SemanticVersion,
    f: Box<SemanticVersionEventUpcasterFunc>,
}

impl SemanticVersionEventUpcaster {
    /// Creates a `SemanticVersionEventUpcaster`
    pub fn new(
        event_type: &str,
        event_version: &str,
        f: Box<SemanticVersionEventUpcasterFunc>,
    ) -> Self {
        let event_version: SemanticVersion = SemanticVersion::from_str(event_version)
            .expect("event_version is not a valid semantic version");
        Self {
            event_type: event_type.to_string(),
            event_version,
            f,
        }
    }
}

impl EventUpcaster for SemanticVersionEventUpcaster {
    fn can_upcast(&self, event_type: &str, event_version: &str) -> bool {
        if event_type != self.event_type.as_str() {
            return false;
        }
        let event_version = match SemanticVersion::from_str(event_version) {
            Ok(result) => result,
            Err(_) => {
                return false;
            }
        };
        self.event_version.supersedes(&event_version)
    }

    fn upcast(&self, event: SerializedEvent) -> SerializedEvent {
        let upcasted_payload = (self.f)(event.payload);
        SerializedEvent {
            aggregate_id: event.aggregate_id,
            sequence: event.sequence,
            aggregate_type: event.aggregate_type,
            event_type: event.event_type,
            event_version: self.event_version.to_string(),
            payload: upcasted_payload,
            metadata: event.metadata,
        }
    }
}

#[cfg(test)]
mod test {
    use std::str::FromStr;

    use crate::SerializedEvent;
    use serde_json::json;
    use serde_json::Value;

    use crate::upcaster::{
        EventUpcaster, SemanticVersion, SemanticVersionError, SemanticVersionEventUpcaster,
        SemanticVersionEventUpcasterFunc,
    };

    fn semantic_version(major_version: u32, minor_version: u32, patch: u32) -> SemanticVersion {
        SemanticVersion {
            major_version,
            minor_version,
            patch,
        }
    }
    #[test]
    fn parse_version() {
        assert_eq!(
            semantic_version(2, 3, 4),
            SemanticVersion::from_str("2.3.4").unwrap()
        );
        assert_eq!(
            semantic_version(2, 3, 4),
            SemanticVersion::from_str("2.3.4.5").unwrap()
        );
        assert_eq!(
            Err(SemanticVersionError),
            SemanticVersion::from_str("not_a_version")
        );
        // TODO: support missing minor version/patch
        assert_eq!(Err(SemanticVersionError), SemanticVersion::from_str("2.3"));
        assert_eq!(Err(SemanticVersionError), SemanticVersion::from_str("2"));
    }

    #[test]
    fn simple_upcaster_can_upcast() {
        let upcaster =
            SemanticVersionEventUpcaster::new("EventX", "2.3.4", Box::new(|event| event));
        assert!(upcaster.can_upcast("EventX", "1.12.35"));
        assert!(upcaster.can_upcast("EventX", "2.3.3"));
        assert!(!upcaster.can_upcast("AnotherEvent", "1.12.35"));
        assert!(!upcaster.can_upcast("EventX", "2.3.4"));
        assert!(!upcaster.can_upcast("EventX", "2.3.5"));
        assert!(!upcaster.can_upcast("EventX", "2.4.0"));
        assert!(!upcaster.can_upcast("EventX", "3.0.0"));
    }

    #[test]
    fn semantic_version_upcaster_can_upcast() {
        SemanticVersionEventUpcaster::new("EventX", "2.3.4", test_upcast());
    }

    #[test]
    #[should_panic]
    fn semantic_version_upcaster_invalid_version() {
        SemanticVersionEventUpcaster::new("EventX", "2", test_upcast());
    }

    #[test]
    fn semantic_version_upcaster_upcast() {
        let upcaster = SemanticVersionEventUpcaster::new("EventX", "2.3.4", test_upcast());
        let payload: Value = serde_json::from_str(
            r#"{
    "id": 4829,
    "name": "George Steinbrenner"
}"#,
        )
        .unwrap();
        let event = SerializedEvent::new(
            "".to_string(),
            0,
            "".to_string(),
            "".to_string(),
            "".to_string(),
            payload,
            Default::default(),
        );
        println!("{}", event.payload.to_string());
        let upcasted_event = upcaster.upcast(event);
        println!("{}", upcasted_event.payload.to_string());
    }
    #[test]
    fn semantic_version_upcaster_upcast_for_documentation() {
        let upcast_function = Box::new(|payload: Value| match payload {
            Value::Object(mut object_map) => {
                object_map.insert("country".to_string(), "USA".into());
                Value::Object(object_map)
            }
            _ => {
                panic!("the event payload is not an object")
            }
        });
        let upcaster = SemanticVersionEventUpcaster::new("EventX", "2.3.4", upcast_function);

        let payload: Value = serde_json::from_str(
            r#"{
                    "zip code": 98103,
                    "state": "Washington"
                   }"#,
        )
        .unwrap();
        let event = SerializedEvent::new(
            "".to_string(),
            0,
            "".to_string(),
            "".to_string(),
            "".to_string(),
            payload,
            Default::default(),
        );
        let upcasted_event = upcaster.upcast(event);

        let expected_payload: Value = serde_json::from_str(
            r#"{
                    "zip code": 98103,
                    "state": "Washington",
                    "country": "USA"
                   }"#,
        )
        .unwrap();
        let expected_event = SerializedEvent::new(
            "".to_string(),
            0,
            "".to_string(),
            "".to_string(),
            "2.3.4".to_string(),
            expected_payload,
            Default::default(),
        );

        assert_eq!(upcasted_event, expected_event);
    }

    fn test_upcast() -> Box<SemanticVersionEventUpcasterFunc> {
        Box::new(|mut payload| {
            let current_id = payload.get("id").unwrap().to_string();
            let updated_id = format!("CUST{}", current_id);
            *payload.get_mut("id").unwrap() = json!(updated_id);
            payload
        })
    }
}
