use crate::errors::*;

/// Represents all the different states that can be generated for a single template, allowing amalgamation logic to be run with the knowledge
/// of what did what (rather than blindly working on a vector).
#[derive(Default, Debug)]
pub struct States {
    /// Any state generated by the *build state* strategy.
    pub build_state: Option<String>,
    /// Any state generated by the *request state* strategy.
    pub request_state: Option<String>,
}
impl States {
    /// Creates a new instance of the states, setting both to `None`.
    pub fn new() -> Self {
        Self::default()
    }
    /// Checks if both request state and build state are defined.
    pub fn both_defined(&self) -> bool {
        self.build_state.is_some() && self.request_state.is_some()
    }
    /// Gets the only defined state if only one is defined. If no states are defined, this will just return `None`. If both are defined,
    /// this will return an error.
    pub fn get_defined(&self) -> Result<Option<String>, ServeError> {
        if self.both_defined() {
            return Err(ServeError::BothStatesDefined);
        }

        if self.build_state.is_some() {
            Ok(self.build_state.clone())
        } else if self.request_state.is_some() {
            Ok(self.request_state.clone())
        } else {
            Ok(None)
        }
    }
}
