use super::*;

/// A set of register fields
///
/// # Created by:
/// - Combining (with `|`) fields generated by the [`periph!`] or the [`register!`] macro.
/// - `Default` or [`Fields::empty`]: a set with no fields.
///
/// # Usable with:
/// - [`Reg::fields`] or [`Value`]` & `[`Fields`]: read some [`FieldValues`] from the register. They can then be compared or written back.
/// - [`Reg::toggle`] or [`Value`]` ^ `[`Fields`]: toggle these fields (only for single-bit fields).
///
/// These fields be combined together with `|`, `&` and `^`. For operators that returns the same
/// type as their first operand, the "assign" form (i.e. `|=`, `&=` and `^=`) can also be used.
///
/// # Example
///
/// ```
/// use peripherals::{register, Fields, Value};
///
/// register! {
///     Register: u8 = 0b1001 {
///         BIT1: 0 = struct Bit1(bool);
///         BIT2: 1 = struct Bit2(bool);
///         TWO_BITS: 2..3 = struct TwoBits(u8);
///     }
/// }
///
/// // Obtain it with the `empty` method or by combining fields
/// let empty = Fields::<Register>::empty();
/// assert_eq!(empty.mask(), 0b0000);
/// let bits_12 = Register::BIT1 | Register::BIT2;
/// assert_eq!(bits_12.mask(), 0b0011);
/// let fields = Register::BIT1 | Register::TWO_BITS;
/// assert_eq!(fields.mask(), 0b1101);
///
/// // Combine fields together
/// assert_eq!(bits_12 | fields, Register::BIT1 | Register::BIT2 | Register::TWO_BITS);
/// assert_eq!(bits_12 & fields, Register::BIT1);
/// assert_eq!(bits_12 ^ fields, Register::BIT2 | Register::TWO_BITS);
///
/// assert_eq!(bits_12 | empty, bits_12);
/// assert_eq!(bits_12 & empty, empty);
/// assert_eq!(bits_12 ^ empty, bits_12);
///
/// let mut value = Value::reset();
/// assert_eq!(value.value(), 0b1001);
///
/// // Use it to read fields
/// let value_12 = value & (Register::BIT1 | Register::BIT2);
/// assert_eq!(value_12.bits(), 0b001);
/// assert_eq!((value & Register::TWO_BITS).bits(), 0b1000);
///
/// // Toggle single-bit fields
/// value ^= bits_12;
/// assert_eq!(value.value(), 0b1010);
/// value ^= Register::BIT1;
/// assert_eq!(value.value(), 0b1011);
///
/// // Write back bits previously read
/// value |= value_12;
/// assert_eq!(value.value(), 0b1001);
/// ```

pub struct Fields<R: RegisterValue, T = ()> {
    mask: R::Int,
    _reg: PhantomData<R>,
    _type: PhantomData<T>,
}

impl<R: RegisterValue, T> Fields<R, T> {
    /// Get the raw mask
    #[inline]
    pub fn mask(self) -> R::Int {
        self.mask
    }

    /// Build from raw mask
    ///
    /// # Safety
    ///
    /// You should ensure the mask is valid for the fields of the associated register.
    #[inline]
    pub unsafe fn from_raw(mask: R::Int) -> Fields<R, T> {
        Fields {
            mask,
            _reg: PhantomData,
            _type: PhantomData,
        }
    }

    /// An empty set with no fields
    #[inline]
    pub fn empty() -> Fields<R, T> {
        Fields {
            mask: R::Int::default(),
            _reg: PhantomData,
            _type: PhantomData,
        }
    }
}

impl<R: RegisterValue, T> Clone for Fields<R, T> {
    #[inline]
    fn clone(&self) -> Fields<R, T> {
        Fields {
            mask: self.mask,
            _reg: PhantomData,
            _type: PhantomData,
        }
    }
}

impl<R: RegisterValue, T> Copy for Fields<R, T> {}

impl<R: RegisterValue, T> Default for Fields<R, T> {
    #[inline]
    fn default() -> Fields<R, T> {
        Fields {
            mask: R::Int::default(),
            _reg: PhantomData,
            _type: PhantomData,
        }
    }
}

impl<R: RegisterValue, T> Debug for Fields<R, T> {
    fn fmt(&self, fmt: &mut fmt::Formatter) -> fmt::Result {
        if fmt.alternate() {
            write!(
                fmt,
                "Fields<{}>(0b{:02$b})",
                R::NAME,
                self.mask,
                <R::Int as Int>::WIDTH
            )
        } else {
            write!(
                fmt,
                "Fields<{}>(0x{:02$x})",
                R::NAME,
                self.mask,
                <R::Int as Int>::WIDTH / 4
            )
        }
    }
}

impl<R: RegisterValue, T: Into<Fields<R>> + Copy, U> PartialEq<T> for Fields<R, U> {
    #[inline]
    fn eq(&self, other: &T) -> bool {
        let other = (*other).into();
        self.mask == other.mask
    }
}

impl<R: RegisterValue> From<Fields<R, Toggle>> for Fields<R, ()> {
    #[inline]
    fn from(fields: Fields<R, Toggle>) -> Fields<R, ()> {
        Fields {
            mask: fields.mask,
            _reg: PhantomData,
            _type: PhantomData,
        }
    }
}

impl<R: RegisterValue, T> MayToggle for Fields<R, T> {
    type Toggle = T;
}

impl<R: RegisterValue, T: Into<Fields<R>>, U> BitOr<T> for Fields<R, U>
where
    T: Both<U>,
{
    type Output = Fields<R, T::Output>;

    #[inline]
    fn bitor(self, other: T) -> Self::Output {
        let other = other.into();
        Fields {
            mask: self.mask | other.mask,
            _reg: PhantomData,
            _type: PhantomData,
        }
    }
}

impl<R: RegisterValue, T: Into<Fields<R>>, U> BitOrAssign<T> for Fields<R, U>
where
    T: Both<U, Output = U>,
{
    #[inline]
    fn bitor_assign(&mut self, other: T) {
        let other = other.into();
        self.mask = self.mask | other.mask;
    }
}

impl<R: RegisterValue, T: Into<Fields<R>>, U> BitAnd<T> for Fields<R, U>
where
    T: Either<U>,
{
    type Output = Fields<R, T::Output>;

    #[inline]
    fn bitand(self, other: T) -> Self::Output {
        let other = other.into();
        Fields {
            mask: self.mask & other.mask,
            _reg: PhantomData,
            _type: PhantomData,
        }
    }
}

impl<R: RegisterValue, T: Into<Fields<R>>, U> BitAndAssign<T> for Fields<R, U>
where
    T: Either<U, Output = U>, // Not needed for soundness, but for consistency
{
    #[inline]
    fn bitand_assign(&mut self, other: T) {
        let other = other.into();
        self.mask = self.mask & other.mask;
    }
}

impl<R: RegisterValue, T: Into<Fields<R>>, U> BitXor<T> for Fields<R, U>
where
    T: Both<U>,
{
    type Output = Fields<R, T::Output>;

    #[inline]
    fn bitxor(self, other: T) -> Self::Output {
        let other = other.into();
        Fields {
            mask: self.mask ^ other.mask,
            _reg: PhantomData,
            _type: PhantomData,
        }
    }
}

impl<R: RegisterValue, T: Into<Fields<R>>, U> BitXorAssign<T> for Fields<R, U>
where
    T: Both<U, Output = U>,
{
    #[inline]
    fn bitxor_assign(&mut self, other: T) {
        let other = other.into();
        self.mask = self.mask ^ other.mask;
    }
}
