use super::*;

/// Some fields' values read from or to be written to a register
///
/// # Created by:
/// - Combining (with `|`) fields values generated by the [`periph!`], [`register!`] or [`field_type!`] macros.
/// - [`Reg::fields`] or [`Value`]` & `[`Fields`]: read some fields from the register.
/// - `Default` or [`FieldValues::empty`]: no fields' values.
///
/// # Used with:
/// - [`Reg::modify`] or [`Value`]` | `[`FieldValues`]: change the value of the specified fields.
///
/// These field values can be combined together with `|`.
///
/// # Example
///
/// ```
/// use peripherals::{register, FieldValues, Fields, Value};
///
/// register! {
///     Register: u8 = 0b1001 {
///         BIT1: 0 = struct Bit1(bool);
///         BIT2: 1 = struct Bit2(bool);
///         TWO_BITS: 2..3 = struct TwoBits(u8);
///     }
/// }
///
/// // Obtain it with the `empty` method or by combining field values
/// let empty = FieldValues::<Register>::empty();
/// assert_eq!(empty.bits(), 0b0000);
/// assert_eq!(empty.fields(), Fields::<Register>::empty());
/// let bits_12 = Bit1(false) | Bit2(true);
/// assert_eq!(bits_12.bits(), 0b0010);
/// assert_eq!(bits_12.fields(), Register::BIT1 | Register::BIT2);
/// let values = Bit1(true) | TwoBits(0b01);
/// assert_eq!(values.bits(), 0b0101);
/// assert_eq!(values.fields(), Register::BIT1 | Register::TWO_BITS);
///
/// // Use it to modify values
/// let mut value = Value::<Register>::reset();
/// assert_eq!(value.value(), 0b1001);
/// value |= bits_12;
/// assert_eq!(value.value(), 0b1010);
/// value |= values;
/// assert_eq!(value.value(), 0b0111);
/// ```

pub struct FieldValues<R: RegisterValue, T = ()> {
    bits: R::Int,
    mask: R::Int,
    _reg: PhantomData<R>,
    _toggle: PhantomData<T>,
}

impl<R: RegisterValue, T> FieldValues<R, T> {
    /// Get the raw bits to set
    #[inline]
    pub fn bits(self) -> R::Int {
        self.bits
    }

    /// Get the raw mask
    #[inline]
    pub fn mask(self) -> R::Int {
        self.mask
    }

    /// Get the fields specified in this value
    #[inline]
    pub fn fields(self) -> Fields<R, T> {
        unsafe { Fields::from_raw(self.mask) }
    }

    /// Build from raw bits and mask
    ///
    /// # Safety
    ///
    /// You must ensure the value and mask are valid for the fields of the associated register.
    #[inline]
    pub unsafe fn from_raw(bits: R::Int, mask: R::Int) -> FieldValues<R, T> {
        FieldValues {
            bits,
            mask,
            _reg: PhantomData,
            _toggle: PhantomData,
        }
    }

    /// No fields' values (doesn't change anything when used to modify a register)
    #[inline]
    pub fn empty() -> FieldValues<R, T> {
        FieldValues {
            bits: R::Int::default(),
            mask: R::Int::default(),
            _reg: PhantomData,
            _toggle: PhantomData,
        }
    }
}

impl<R: RegisterValue, T> Clone for FieldValues<R, T> {
    #[inline]
    fn clone(&self) -> FieldValues<R, T> {
        FieldValues {
            bits: self.bits,
            mask: self.mask,
            _reg: PhantomData,
            _toggle: PhantomData,
        }
    }
}

impl<R: RegisterValue, T> Copy for FieldValues<R, T> {}

impl<R: RegisterValue, T> Default for FieldValues<R, T> {
    #[inline]
    fn default() -> FieldValues<R, T> {
        FieldValues {
            bits: R::Int::default(),
            mask: R::Int::default(),
            _reg: PhantomData,
            _toggle: PhantomData,
        }
    }
}

impl<R: RegisterValue, T> Debug for FieldValues<R, T> {
    fn fmt(&self, fmt: &mut fmt::Formatter) -> fmt::Result {
        if fmt.alternate() {
            write!(
                fmt,
                "FieldValues<{}>(0b{:03$b}, 0b{:03$x})",
                R::NAME,
                self.bits,
                self.mask,
                <R::Int as Int>::WIDTH
            )
        } else {
            write!(
                fmt,
                "FieldValues<{}>(0x{:03$x}, 0x{:03$x})",
                R::NAME,
                self.bits,
                self.mask,
                <R::Int as Int>::WIDTH / 4
            )
        }
    }
}

impl<R: RegisterValue, T: Into<FieldValues<R>> + Copy, U> PartialEq<T> for FieldValues<R, U> {
    #[inline]
    fn eq(&self, other: &T) -> bool {
        let other = (*other).into();
        self.bits == other.bits && self.mask == other.mask
    }
}

impl<R: RegisterValue> Eq for FieldValues<R, ()> {}
impl<R: RegisterValue> Eq for FieldValues<R, Toggle> {}

impl<R: RegisterValue, T: Into<FieldValues<R, Toggle>>> From<T> for FieldValues<R, ()> {
    #[inline]
    fn from(bits: T) -> FieldValues<R, ()> {
        let bits = bits.into();
        FieldValues {
            bits: bits.bits,
            mask: bits.mask,
            _reg: PhantomData,
            _toggle: PhantomData,
        }
    }
}

impl<R: RegisterValue, T> MayToggle for FieldValues<R, T> {
    type Toggle = T;
}

impl<R: RegisterValue> Not for FieldValues<R, Toggle> {
    type Output = Self;

    #[inline]
    fn not(self) -> Self {
        FieldValues {
            bits: !self.bits & self.mask,
            mask: self.mask,
            _reg: PhantomData,
            _toggle: PhantomData,
        }
    }
}

impl<R: RegisterValue, T: Into<FieldValues<R>>, U> BitOr<T> for FieldValues<R, U>
where
    T: Both<U>,
{
    type Output = FieldValues<R, T::Output>;

    #[inline]
    fn bitor(self, other: T) -> Self::Output {
        let other = other.into();
        FieldValues {
            bits: self.bits & !other.mask | other.bits,
            mask: self.mask | other.mask,
            _reg: PhantomData,
            _toggle: PhantomData,
        }
    }
}

impl<R: RegisterValue, T: Into<FieldValues<R>>, U> BitOrAssign<T> for FieldValues<R, U>
where
    T: Both<U, Output = U>,
{
    #[inline]
    fn bitor_assign(&mut self, other: T) {
        let other = other.into();
        self.bits = self.bits & !other.mask | other.bits;
        self.mask = self.mask | other.mask;
    }
}

impl<R: RegisterValue, T: Into<Fields<R>>, U> BitAnd<T> for FieldValues<R, U>
where
    T: Either<U>,
{
    type Output = FieldValues<R, T::Output>;

    #[inline]
    fn bitand(self, other: T) -> Self::Output {
        let other = other.into();
        FieldValues {
            bits: self.bits & other.mask(),
            mask: self.mask & other.mask(),
            _reg: PhantomData,
            _toggle: PhantomData,
        }
    }
}

impl<R: RegisterValue, T: Into<Fields<R>>, U> BitAndAssign<T> for FieldValues<R, U>
where
    T: Either<U, Output = U>, // Not need for soundness, but for consistency
{
    #[inline]
    fn bitand_assign(&mut self, other: T) {
        let other = other.into();
        self.bits = self.bits & other.mask();
        self.mask = self.mask & other.mask();
    }
}

impl<R: RegisterValue, T: Into<Fields<R>>, U> BitXor<T> for FieldValues<R, U>
where
    T: Either<U, Output = Toggle>,
{
    type Output = FieldValues<R, U>;

    #[inline]
    fn bitxor(self, other: T) -> Self::Output {
        let other = other.into();
        FieldValues {
            bits: self.bits ^ (self.mask & other.mask()),
            mask: self.mask,
            _reg: PhantomData,
            _toggle: PhantomData,
        }
    }
}

impl<R: RegisterValue, T: Into<Fields<R>>, U> BitXorAssign<T> for FieldValues<R, U>
where
    T: Either<U, Output = Toggle>,
{
    #[inline]
    fn bitxor_assign(&mut self, other: T) {
        let other = other.into();
        self.bits = self.bits ^ (self.mask & other.mask());
    }
}
