# Perfdata (Performance Data)

This library was created to fit basic needs for parsing and creating
Performance Data commonly used by check commands from monitoring engines
like Icinga2 or Nagios.

I'm using this in close-to-production software, but would generally **consider
this library still in a pre-stable stage**, since it's still evolving.

## Usage
Usually you will want to create performance data in check commands called by
monitoring engines, after executing the command from the data collected.
```rust
// simple label with a value
let perfdata = Perfdata::unit("label", 23);

// with warn, crit, min and max thresholds
let with_thresholds = Perfdata::unit("thresholds", 42)
    .with_warn(ThresholdRange::inside(33,50))
    .with_crit(ThresholdRange::above(50))
    .with_min(0)
    .with_max(100);

// check thresholds
if with_thresholds.is_warn() {
    println("{} in warning threshold", with_threshold.label())
}
if with_thresholds.is_crit() {
    println("{} in critical threshold", with_threshold.label())
}
   
```

Perfdata can be created with several units of measurements. And will be
formatted to the spec of the
[Nagios Plugin Development Guidelines](https://nagios-plugins.org/doc/guidelines.html#AEN200).
```rust
// This will be formatted as 'seconds'=10s
Perfdata::seconds("seconds", 10);

// This will be formatted as 'percent'=50%
Perfdata::percent("seconds", 50);

// This will be formatted as 'bytes'=23b
Perfdata::bytes("bytes", 23);

// This will be formatted as 'counter'=10c;@20:30;30;0;100
Perfdata::percent("counter", 10)
  .with_warn(ThresholdRange::inside(20,30))
  .with_crit(ThresholdRange::above_pos(30))
  .with_min(0)
  .with_max(100);
```

Multiple Perfdata points can be combined into a `PerfdataSet`, which provides some utilities for usage in monitoring
checks, most notably the `MonitoringStatus` enum.

```rust
// PerfdataSets can be built from iterators over Perfdata
let mut pds: PerfdataSet = [Perfdata::unit("this is fine", 42), Perfdata::percent("this is also fine")].iter().collect();
let critical = Perfdata::unit(100).with_crit(ThresholdRange::above(50));

// Additional perfdata can be added
pds.add(critical);

// This is MonitoringStatus::Critical
let status = pds.status;()

// Exit with code 2, interpreted by most monitoring engines as "critical" check result
std::process::exit(status.exit_code());
```

This library provides also a basic parser, for dealing with perfdata generated by one of the
myriad of check commands for common monitoring engines.
```rust
let input = "'some perf'=42;@75:80;80";
let perfdata = Perfdata::try_from(input).unwrap();
```

Usually more than one Performance Datapoint are generated in a space delimited list.
These can be parsed using `PerfdataSet::try_from()`;
```rust
let input = "'some perf'=42;66;75;0;100; 'foo'=23 bar=10"
let pds = PerfdataSet::try_from(input).unwrap();
```

### License
Licensed under either of [Apache License](./LICENSE-APACHE), Version 2.0 or [MIT license](./LICENSE-MIT) at your option.
Unless you explicitly state otherwise, any contribution intentionally submitted for
inclusion in this crate by you, as defined in the Apache-2.0 license, shall be dual
licensed as above, without any additional terms or conditions. 