use crate::generator::TrafficGenerator;
use crate::scenario::Scenario;

use crossbeam_channel::{bounded, Receiver, Sender};
use reqwest;
use rgb::RGB8;
use std::sync::{
    atomic::{AtomicUsize, Ordering},
    Arc,
};
use std::time::{Duration, Instant};
use textplots::{Chart, ColorPlot, Shape};
use tokio;
use tokio::sync::Semaphore;
/// Very simple reqwest based generator. This generator is very sensitive to
/// latency and should mainly be used to test servers running on the same (or neighboring)
/// machine. This Generator will reuse TCP connections for requests, and
/// all request will be sent from the same IP.
pub struct ReqwestGenerator<S: Scenario> {
    pub scenario: S,
    /// This is the maximum amount of TCP file descriptors **being used** at any one time.
    /// The actual number of open connections will be higher, because it takes
    /// time to purge old fd's.
    pub max_open_connections: usize,
    /// Let this generator warm up to the rate at t=0 before running the test.
    pub warmup: bool,
    /// The request to send.
    pub req: reqwest::Request,
    /// Holds time series data on the send rate of the generator
    send_rate_channel: (Sender<(f32, f32)>, Receiver<(f32, f32)>),
    /// Sends the total number of successfull requests sent.
    send_num_packets_channel: (Sender<usize>, Receiver<usize>),
}

impl<T: Scenario> TrafficGenerator<T> for ReqwestGenerator<T> {
    fn run_scenario(&mut self) {
        let rt = tokio::runtime::Runtime::new().unwrap();
        rt.block_on(self.run_reqwests())
    }
    fn fire_hose(&mut self) {
        let rt = tokio::runtime::Runtime::new().unwrap();
        rt.block_on(self.fire())
    }

    fn set_scenario(&mut self, schem: T) {
        self.scenario = schem;
    }
    fn send_packet(&mut self) {
        let client = reqwest::blocking::Client::new();
        let _ = client.get(self.req.url().clone()).send();
    }
    fn get_data_rate_channel(&self) -> Receiver<(f32, f32)> {
        self.send_rate_channel.1.clone()
    }

    fn get_scenario(&self) -> &T {
        &self.scenario
    }

    fn get_sent_packets_channel(&self) -> Receiver<usize> {
        self.send_num_packets_channel.1.clone()
    }
}

impl<S: Scenario> ReqwestGenerator<S> {
    pub fn new(
        scenario: S,
        max_open_connections: usize,
        warmup: bool,
        req: reqwest::Request,
    ) -> Self {
        ReqwestGenerator {
            scenario,
            max_open_connections,
            req,
            warmup,
            send_rate_channel: bounded(10),
            send_num_packets_channel: bounded(10),
        }
    }

    #[allow(dead_code)]
    fn print_graph(&self, send_data: &Vec<(f32, f32)>, total_run_time: f32) {
        Chart::new_with_y_range(300, 100, 0., total_run_time, 0., 45000.)
            .linecolorplot(
                &Shape::Lines(&send_data.as_slice()),
                RGB8 {
                    r: 255_u8,
                    g: 255_u8,
                    b: 255_u8,
                },
            )
            .linecolorplot(
                &Shape::Continuous(Box::new(|x| {
                    self.scenario
                        .rate(Duration::from_millis((x as usize).try_into().unwrap()))
                })),
                RGB8 {
                    r: 255_u8,
                    g: 255_u8,
                    b: 0_u8,
                },
            )
            .display();
    }

    async fn run_reqwests(&mut self) {
        let max_tokio_spawn: usize = self.max_open_connections;
        let mut send_rate: f32 = self.scenario.rate(Duration::from_millis(0));
        let client = reqwest::Client::new();
        let semaphore = Arc::new(Semaphore::new(max_tokio_spawn as usize));
        let num_successful = Arc::new(AtomicUsize::new(0));
        let mut send_delay = Duration::from_micros(1000000 / send_rate as u64);
        let mut previos_count: usize = 0;
        let mut previos_time = Instant::now();
        let mut start = Instant::now();
        let mut elasped = start.elapsed();
        let test_duration = self.scenario.duration();
        let mut has_warmed_up = false;

        while elasped.as_millis() < test_duration {
            for _ in 0..10 {
                let permit = semaphore.clone().acquire_owned().await.unwrap();
                let ns = num_successful.clone();
                let fut = client.execute(self.req.try_clone().unwrap());
                tokio::spawn(async move {
                    let res = fut.await;
                    drop(permit);
                    match res {
                        Ok(_) => {
                            ns.fetch_add(1, Ordering::Relaxed);
                        }
                        _ => {}
                    }
                });
            }
            std::thread::sleep(send_delay);

            if previos_time.elapsed().as_millis() > 100 {
                elasped = start.elapsed();
                let suc = num_successful.fetch_add(0, Ordering::Relaxed);
                let delta: f64 = suc as f64 - previos_count as f64;
                let rate = delta as u128 * 1000000 / (previos_time.elapsed().as_micros());
                send_rate = self.scenario.rate(elasped);

                // println!("Send Rate: {}", rate);

                previos_time = Instant::now();
                previos_count = suc;

                let rate_delta = rate as f64 - send_rate as f64;

                // Logisticall Tune rate to target
                // https://en.wikipedia.org/wiki/Logistic_function
                let adjustment = 2. / (1. + (-(rate_delta as f64) / 10000.).exp());

                send_delay =
                    Duration::from_nanos((send_delay.as_nanos() as f64 * adjustment) as u64);

                if self.warmup && !has_warmed_up {
                    if (adjustment - 1.).abs() < 0.01 {
                        has_warmed_up = true;
                        start = Instant::now();
                    }
                } else {
                    let _ = self
                        .send_rate_channel
                        .0
                        .try_send((elasped.as_millis() as f32, rate as f32));
                    let _ = self.send_num_packets_channel.0.try_send(suc);
                }
            }
        }

        // Wait for the last permits to get used
        let _ = semaphore.acquire_many(max_tokio_spawn as u32).await;
        // let tot_time = start.elapsed().as_millis();
        // let tot_sent = *num_successful.read().await;

        // println!("Number of Successful queries: {}", tot_sent);
        // println!(
        //     "{} requests in {} seconds or {} Req/s",
        //     tot_sent,
        //     tot_time as f64 / 1000.,
        //     tot_sent as f64 / (start.elapsed().as_millis() as f64 / 1000.)
        // );
        // self.print_graph(&send_data, tot_time as f32);
    }

    async fn fire(&self) {
        let max_tokio_spawn: usize = self.max_open_connections;

        let client = reqwest::Client::new();
        let semaphore = Arc::new(Semaphore::new(max_tokio_spawn as usize));
        let num_successful = Arc::new(AtomicUsize::new(0));
        let num_sent = Arc::new(AtomicUsize::new(0));

        let test_duration = self.scenario.duration();

        let previos_count: usize = 0;
        let previos_time = Instant::now();
        let start = Instant::now();
        let mut elasped;

        while start.elapsed().as_millis() < test_duration {
            for _ in 0..100 {
                let permit = semaphore.clone().acquire_owned().await.unwrap();
                let suc_lock = num_successful.clone();
                let tot_lock = num_sent.clone();
                let fut = client.execute(self.req.try_clone().unwrap());
                tokio::spawn(async move {
                    let res = fut.await;
                    drop(permit);
                    tot_lock.fetch_add(1, Ordering::Relaxed);
                    match res {
                        Ok(_) => suc_lock.fetch_add(1, Ordering::Relaxed),
                        Err(e) => {
                            panic!("{}", e)
                        }
                    }
                });
            }
            if previos_time.elapsed().as_millis() > 100 {
                elasped = start.elapsed();
                let suc = num_successful.fetch_add(0, Ordering::Relaxed);
                let delta: f64 = suc as f64 - previos_count as f64;
                let rate = delta as u128 * 1000000 / (previos_time.elapsed().as_micros());
                let _ = self
                    .send_rate_channel
                    .0
                    .try_send((elasped.as_millis() as f32, rate as f32));
                let _ = self.send_num_packets_channel.0.try_send(suc);
            }
        }

        // Wait for the last permits to get used
        let _ = semaphore.acquire_many(max_tokio_spawn as u32).await;
        // let tot_time = start.elapsed().as_millis();
        // let tot_suc = num_successful.fetch_add(0, Ordering::Relaxed);
        // let tot_sent = num_sent.fetch_add(0, Ordering::Relaxed);

        // println!(
        //     "Number of Successful queries: {}, Total sent {}",
        //     tot_suc, tot_sent
        // );
        // println!(
        //     "{} requests in {} seconds or {} Req/s",
        //     tot_suc,
        //     tot_time as f64 / 1000.,
        //     tot_suc as f64 / (start.elapsed().as_millis() as f64 / 1000.),
        // );
    }
}
