#![allow(dead_code)]
#[cfg(feature = "rayon")]
use rayon::prelude::*;

/// Gets the index (into Int. Crys. Handbook Vol A 2016) for the given symbol. First it is
/// interpreted as a Herman Mauguin symbol, if that is unsuccessful it is interpreted as a
/// Hall symbol.
pub fn get_index_for_symbol(symbol: &str) -> Option<usize> {
    if let Some(index) = HERMANN_MAUGUIN_SYMBOL.iter().position(|i| *i == symbol) {
        Some(index + 1)
    } else {
        HALL_SYMBOL.iter().position(|i| *i == symbol).map(|n| n + 1)
    }
}

/// Gets the index (into Int. Crys. Handbook Vol A 2016) for the given symbol in parallel. First it is
/// interpreted as a Herman Mauguin symbol, if that is unsuccessful it is interpreted as a
/// Hall symbol.
#[cfg(feature = "rayon")]
pub fn par_get_index_for_symbol(symbol: &str) -> Option<usize> {
    if let Some(index) = HERMANN_MAUGUIN_SYMBOL
        .par_iter()
        .position_any(|i| *i == symbol)
    {
        Some(index + 1)
    } else {
        HALL_SYMBOL
            .par_iter()
            .position_any(|i| *i == symbol)
            .map(|n| n + 1)
    }
}

/// Gets the Herman Mauguin symbol for the given index (into Int. Crys. Handbook Vol A 2016)
pub fn get_herman_mauguin_symbol_for_index(index: usize) -> Option<&'static str> {
    HERMANN_MAUGUIN_SYMBOL.get(index - 1).copied()
}

/// Gets the Hall symbol for the given index (into Int. Crys. Handbook Vol A 2016)
pub fn get_hall_symbol_for_index(index: usize) -> Option<&'static str> {
    HALL_SYMBOL.get(index - 1).copied()
}

/// Gets the transformations given an index (into Int. Crys. Handbook Vol A 2016) for the given space group
pub fn get_transformation(index: usize) -> Option<&'static [[[f64; 4]; 3]]> {
    SYMBOL_TRANSFORMATION.get(index - 1).copied()
}

/// Gets the atomic number for the given element. It is case insensitive for the element name.
pub fn get_atomic_number(element: &str) -> Option<usize> {
    let mut counter = 1;
    let element = element.to_ascii_uppercase();
    for item in ELEMENT_SYMBOLS {
        if item == &element {
            return Some(counter);
        }
        counter += 1;
    }
    None
}

/// Gets the atomic radius for the given atomic number (defined up until 'Cm' 96) in Å.
/// Source: Martin Rahm, Roald Hoffmann, and N. W. Ashcroft. Atomic and Ionic Radii of Elements 1-96. Chemistry - A European Journal, 22(41):14625–14632, oct 2016. URL: http://doi.org/10.1002/chem.201602949, doi:10.1002/chem.201602949.
/// Updated to the corrigendum: <https://doi.org/10.1002/chem.201700610>
pub fn get_atomic_radius(atomic_number: usize) -> Option<f64> {
    ELEMENT_ATOMIC_RADII.get(atomic_number - 1).copied()
}

/// Gets the van der Waals radius for the given atomic number (defined up until 'Es' 99) in Å.
/// Source: Alvarez, S. (2013). A cartography of the van der Waals territories. Dalton Transactions, 42(24), 8617. <https://doi.org/10.1039/c3dt50599e>
pub fn get_vanderwaals_radius(atomic_number: usize) -> Option<f64> {
    ELEMENT_VANDERWAALS_RADII.get(atomic_number - 1).copied()
}

/// Gets the covalent bond radii for the given atomic number (defined for all elements (<=118)).
/// The result is the radius for a single, double and triple bond, where the last two are optional.
/// All values are given in picometers.
/// Sources:
///  * P. Pyykkö; M. Atsumi (2009). "Molecular Single-Bond Covalent Radii for Elements 1-118". Chemistry: A European Journal. 15 (1): 186–197. doi:10.1002/chem.200800987
///  * P. Pyykkö; M. Atsumi (2009). "Molecular Double-Bond Covalent Radii for Elements Li–E112". Chemistry: A European Journal. 15 (46): 12770–12779. doi:10.1002/chem.200901472
///  * P. Pyykkö; S. Riedel; M. Patzschke (2005). "Triple-Bond Covalent Radii". Chemistry: A European Journal. 11 (12): 3511–3520. doi:10.1002/chem.200401299
pub fn get_covalent_bond_radii(atomic_number: usize) -> (usize, Option<usize>, Option<usize>) {
    *ELEMENT_BOND_RADII
        .get(atomic_number - 1)
        .expect("Invalid atomic number provided for element bond radius lookup. The number should be less than or equal to 118.")
}

/// Gets the amino acid number into the table, effectively providing the recognition of it being an amino acid or not
pub fn get_amino_acid_number(aa: &str) -> Option<usize> {
    let mut counter = 1;
    for item in AMINO_ACIDS {
        if *item == aa {
            return Some(counter);
        }
        counter += 1;
    }
    None
}

/// Returns if the given atom name is a name for an atom in the backbone of a protein
pub fn is_backbone(name: &str) -> bool {
    BACKBONE_NAMES.contains(&name)
}

/// Returns if the given number is a valid remark-type-number (according to wwPDB v 3.30)
pub fn valid_remark_type_number(number: usize) -> bool {
    REMARK_TYPES.contains(&number)
}

/// The valid remark type numbers as of PDB v3.30
const REMARK_TYPES: [usize; 41] = [
    0, 1, 2, 3, 4, 5, 100, 200, 205, 210, 215, 217, 230, 240, 245, 247, 250, 265, 280, 285, 290,
    300, 350, 375, 450, 465, 470, 475, 480, 500, 525, 600, 610, 615, 620, 630, 650, 700, 800, 900,
    999,
];

/// The symbols/names of the elements of the periodic table
const ELEMENT_SYMBOLS: &[&str] = &[
    "H", "HE", "LI", "BE", "B", "C", "N", "O", "F", "NE", "NA", "MG", "AL", "SI", "P", "S", "CL",
    "AR", "K", "CA", "SC", "TI", "V", "CR", "MN", "FE", "CO", "NI", "CU", "ZN", "GA", "GE", "AS",
    "SE", "BR", "KR", "RB", "SR", "Y", "ZR", "NB", "MO", "TC", "RU", "RH", "PD", "AG", "CD", "IN",
    "SN", "SB", "TE", "I", "XE", "CS", "BA", "LA", "CE", "PR", "ND", "PM", "SM", "EU", "GD", "TB",
    "DY", "HO", "ER", "TM", "YB", "LU", "HF", "TA", "W", "RE", "OS", "IR", "PT", "AU", "HG", "TL",
    "PB", "BI", "PO", "AT", "RN", "FR", "RA", "AC", "TH", "PA", "U", "NP", "PU", "AM", "CM", "BK",
    "CF", "ES", "FM", "MD", "NO", "LR", "RF", "DB", "SG", "BH", "HS", "MT", "DS", "RG", "CN", "NH",
    "FL", "MC", "LV", "TS", "OG",
];

/// The radii of the elements up to Cs 96
const ELEMENT_ATOMIC_RADII: &[f64] = &[
    1.54, 1.34, 2.20, 2.19, 2.05, 1.90, 1.79, 1.71, 1.63, 1.56, 2.25, 2.40, 2.39, 2.32, 2.23, 2.14,
    2.06, 1.97, 2.34, 2.70, 2.63, 2.57, 2.52, 2.33, 2.42, 2.37, 2.33, 2.29, 2.17, 2.22, 2.33, 2.34,
    2.31, 2.24, 2.19, 2.12, 2.40, 2.79, 2.74, 2.69, 2.51, 2.44, 2.52, 2.37, 2.33, 2.15, 2.25, 2.38,
    2.46, 2.48, 2.46, 2.42, 2.38, 2.32, 2.49, 2.93, 2.84, 2.82, 2.86, 2.84, 2.83, 2.80, 2.80, 2.77,
    2.76, 2.75, 2.73, 2.72, 2.71, 2.77, 2.70, 2.64, 2.58, 2.53, 2.49, 2.44, 2.40, 2.30, 2.26, 2.29,
    2.42, 2.49, 2.50, 2.50, 2.47, 2.43, 2.58, 2.92, 2.93, 2.88, 2.85, 2.83, 2.81, 2.78, 2.76, 2.64,
];

/// The van der waals radii of the elements up to Es 99
const ELEMENT_VANDERWAALS_RADII: &[f64] = &[
    1.20, 1.43, 2.12, 1.98, 1.91, 1.77, 1.66, 1.50, 1.46, 1.58, 2.50, 2.51, 2.25, 2.19, 1.90, 1.89,
    1.82, 1.83, 2.73, 2.62, 2.58, 2.46, 2.42, 2.45, 2.45, 2.44, 2.40, 2.40, 2.38, 2.39, 2.32, 2.29,
    1.88, 1.82, 1.86, 2.25, 3.21, 2.84, 2.75, 2.52, 2.56, 2.45, 2.44, 2.46, 2.44, 2.15, 2.53, 2.49,
    2.43, 2.42, 2.47, 1.99, 2.04, 2.06, 3.48, 3.03, 2.98, 2.88, 2.92, 2.95, 2.90, 2.87, 2.83, 2.79,
    2.87, 2.81, 2.83, 2.79, 2.80, 2.74, 2.63, 2.53, 2.57, 2.49, 2.48, 2.41, 2.29, 2.32, 2.45, 2.47,
    2.60, 2.54, 2.80, 2.93, 2.88, 2.71, 2.82, 2.81, 2.83, 3.05, 3.40, 3.05, 2.70,
];

/// The bond radii of all elements
const ELEMENT_BOND_RADII: &[(usize, Option<usize>, Option<usize>)] = &[
    (32, None, None),
    (46, None, None),
    (133, Some(124), None),
    (102, Some(90), Some(85)),
    (85, Some(78), Some(73)),
    (75, Some(67), Some(60)),
    (71, Some(60), Some(54)),
    (63, Some(57), Some(53)),
    (64, Some(59), Some(53)),
    (67, Some(96), None),
    (155, Some(160), None),
    (139, Some(132), Some(127)),
    (126, Some(113), Some(111)),
    (116, Some(107), Some(102)),
    (111, Some(102), Some(94)),
    (103, Some(94), Some(95)),
    (99, Some(95), Some(93)),
    (96, Some(107), Some(96)),
    (196, Some(193), None),
    (171, Some(147), Some(133)),
    (148, Some(116), Some(114)),
    (136, Some(117), Some(108)),
    (134, Some(112), Some(106)),
    (122, Some(111), Some(103)),
    (119, Some(105), Some(103)),
    (116, Some(109), Some(102)),
    (111, Some(103), Some(96)),
    (110, Some(101), Some(101)),
    (112, Some(115), Some(120)),
    (118, Some(120), None),
    (124, Some(117), Some(121)),
    (121, Some(111), Some(114)),
    (121, Some(114), Some(106)),
    (116, Some(107), Some(107)),
    (114, Some(109), Some(110)),
    (117, Some(121), Some(108)),
    (210, Some(202), None),
    (185, Some(157), Some(139)),
    (163, Some(130), Some(124)),
    (154, Some(127), Some(121)),
    (147, Some(125), Some(116)),
    (138, Some(121), Some(113)),
    (128, Some(120), Some(110)),
    (125, Some(114), Some(103)),
    (125, Some(110), Some(106)),
    (120, Some(117), Some(112)),
    (128, Some(139), Some(137)),
    (136, Some(144), None),
    (142, Some(136), Some(146)),
    (140, Some(130), Some(132)),
    (140, Some(133), Some(127)),
    (136, Some(128), Some(121)),
    (133, Some(129), Some(125)),
    (131, Some(135), Some(122)),
    (232, Some(209), None),
    (196, Some(161), Some(149)),
    (180, Some(139), Some(139)),
    (163, Some(137), Some(131)),
    (176, Some(138), Some(128)),
    (174, Some(137), None),
    (173, Some(135), None),
    (172, Some(134), None),
    (168, Some(134), None),
    (169, Some(135), Some(132)),
    (168, Some(135), None),
    (167, Some(133), None),
    (166, Some(133), None),
    (165, Some(133), None),
    (164, Some(131), None),
    (170, Some(129), None),
    (162, Some(131), Some(131)),
    (152, Some(128), Some(122)),
    (146, Some(126), Some(119)),
    (137, Some(120), Some(115)),
    (131, Some(119), Some(110)),
    (129, Some(116), Some(109)),
    (122, Some(115), Some(107)),
    (123, Some(112), Some(110)),
    (124, Some(121), Some(123)),
    (133, Some(142), None),
    (144, Some(142), Some(150)),
    (144, Some(135), Some(137)),
    (151, Some(141), Some(135)),
    (145, Some(135), Some(129)),
    (147, Some(138), Some(138)),
    (142, Some(145), Some(133)),
    (223, Some(218), None),
    (201, Some(173), Some(159)),
    (186, Some(153), Some(140)),
    (175, Some(143), Some(136)),
    (169, Some(138), Some(129)),
    (170, Some(134), Some(118)),
    (171, Some(136), Some(116)),
    (172, Some(135), None),
    (166, Some(135), None),
    (166, Some(136), None),
    (168, Some(139), None),
    (168, Some(140), None),
    (165, Some(140), None),
    (167, Some(167), Some(167)),
    (173, Some(139), None),
    (176, Some(176), Some(176)),
    (161, Some(141), None),
    (157, Some(140), Some(131)),
    (149, Some(136), Some(126)),
    (143, Some(128), Some(121)),
    (141, Some(128), Some(119)),
    (134, Some(125), Some(118)),
    (129, Some(125), Some(113)),
    (128, Some(116), Some(112)),
    (121, Some(116), Some(118)),
    (122, Some(137), Some(130)),
    (136, None, None),
    (143, None, None),
    (162, None, None),
    (175, None, None),
    (165, None, None),
    (157, None, None),
];

/// All amino acids. Includes Amber-specific naming conventions for (de-)protonated versions, CYS involved in
/// disulfide bonding and the like.
const AMINO_ACIDS: &[&str] = &[
    "ALA", "ARG", "ASH", "ASN", "ASP", "CYS", "CYX", "GLH", "GLN", "GLU", "GLY", "HID", "HIE",
    "HIM", "HIP", "HIS", "ILE", "LEU", "LYN", "LYS", "MET", "PHE", "PRO", "SER", "THR", "TRP",
    "TYR", "VAL",
];

/// The names of atom in the backbone of proteins
const BACKBONE_NAMES: &[&str] = &[
    "N", "CA", "C", "O", "H", "H1", "H2", "H3", "HA", "HA2", "HA3",
];

/// The list of Hermann Mauguin symbols in the same order as in the handbook
const HERMANN_MAUGUIN_SYMBOL: &[&str] = &[
    "P 1",
    "P -1",
    "P 1 2 1",
    "P 1 21 1",
    "C 1 2 1",
    "P 1 m 1",
    "P 1 c 1",
    "C 1 m 1",
    "C 1 c 1",
    "P 1 2/m 1",
    "P 1 21/m 1",
    "C 1 2/m 1",
    "P 1 2/c 1",
    "P 1 21/c 1",
    "C 1 2/c 1",
    "P 2 2 2",
    "P 2 2 21",
    "P 21 21 2",
    "P 21 21 21",
    "C 2 2 21",
    "C 2 2 2",
    "F 2 2 2",
    "I 2 2 2",
    "I 21 21 21",
    "P m m 2",
    "P m c 21",
    "P c c 2",
    "P m a 2",
    "P c a 21",
    "P n c 2",
    "P m n 21",
    "P b a 2",
    "P n a 21",
    "P n n 2",
    "C m m 2",
    "C m c 21",
    "C c c 2",
    "A m m 2",
    "A b m 2",
    "A m a 2",
    "A b a 2",
    "F m m 2",
    "F d d 2",
    "I m m 2",
    "I b a 2",
    "I m a 2",
    "P m m m",
    "P n n n :2",
    "P c c m",
    "P b a n :2",
    "P m m a",
    "P n n a",
    "P m n a",
    "P c c a",
    "P b a m",
    "P c c n",
    "P b c m",
    "P n n m",
    "P m m n :2",
    "P b c n",
    "P b c a",
    "P n m a",
    "C m c m",
    "C m c a",
    "C m m m",
    "C c c m",
    "C m m a",
    "C c c a :2",
    "F m m m",
    "F d d d :2",
    "I m m m",
    "I b a m",
    "I b c a",
    "I m m a",
    "P 4",
    "P 41",
    "P 42",
    "P 43",
    "I 4",
    "I 41",
    "P -4",
    "I -4",
    "P 4/m",
    "P 42/m",
    "P 4/n :2",
    "P 42/n :2",
    "I 4/m",
    "I 41/a :2",
    "P 4 2 2",
    "P 4 21 2",
    "P 41 2 2",
    "P 41 21 2",
    "P 42 2 2",
    "P 42 21 2",
    "P 43 2 2",
    "P 43 21 2",
    "I 4 2 2",
    "I 41 2 2",
    "P 4 m m",
    "P 4 b m",
    "P 42 c m",
    "P 42 n m",
    "P 4 c c",
    "P 4 n c",
    "P 42 m c",
    "P 42 b c",
    "I 4 m m",
    "I 4 c m",
    "I 41 m d",
    "I 41 c d",
    "P -4 2 m",
    "P -4 2 c",
    "P -4 21 m",
    "P -4 21 c",
    "P -4 m 2",
    "P -4 c 2",
    "P -4 b 2",
    "P -4 n 2",
    "I -4 m 2",
    "I -4 c 2",
    "I -4 2 m",
    "I -4 2 d",
    "P 4/m m m",
    "P 4/m c c",
    "P 4/n b m :2",
    "P 4/n n c :2",
    "P 4/m b m",
    "P 4/m n c",
    "P 4/n m m :2",
    "P 4/n c c :2",
    "P 42/m m c",
    "P 42/m c m",
    "P 42/n b c :2",
    "P 42/n n m :2",
    "P 42/m b c",
    "P 42/m n m",
    "P 42/n m c :2",
    "P 42/n c m :2",
    "I 4/m m m",
    "I 4/m c m",
    "I 41/a m d :2",
    "I 41/a c d :2",
    "P 3",
    "P 31",
    "P 32",
    "R 3 :h",
    "P -3",
    "R -3 :h",
    "P 3 1 2",
    "P 3 2 1",
    "P 31 1 2",
    "P 31 2 1",
    "P 32 1 2",
    "P 32 2 1",
    "R 3 2 :h",
    "P 3 m 1",
    "P 3 1 m",
    "P 3 c 1",
    "P 3 1 c",
    "R 3 m :h",
    "R 3 c :h",
    "P -3 1 m",
    "P -3 1 c",
    "P -3 m 1",
    "P -3 c 1",
    "R -3 m :h",
    "R -3 c :h",
    "P 6",
    "P 61",
    "P 65",
    "P 62",
    "P 64",
    "P 63",
    "P -6",
    "P 6/m",
    "P 63/m",
    "P 6 2 2",
    "P 61 2 2",
    "P 65 2 2",
    "P 62 2 2",
    "P 64 2 2",
    "P 63 2 2",
    "P 6 m m",
    "P 6 c c",
    "P 63 c m",
    "P 63 m c",
    "P -6 m 2",
    "P -6 c 2",
    "P -6 2 m",
    "P -6 2 c",
    "P 6/m m m",
    "P 6/m c c",
    "P 63/m c m",
    "P 63/m m c",
    "P 2 3",
    "F 2 3",
    "I 2 3",
    "P 21 3",
    "I 21 3",
    "P m -3",
    "P n -3 :2",
    "F m -3",
    "F d -3 :2",
    "I m -3",
    "P a -3",
    "I a -3",
    "P 4 3 2",
    "P 42 3 2",
    "F 4 3 2",
    "F 41 3 2",
    "I 4 3 2",
    "P 43 3 2",
    "P 41 3 2",
    "I 41 3 2",
    "P -4 3 m",
    "F -4 3 m",
    "I -4 3 m",
    "P -4 3 n",
    "F -4 3 c",
    "I -4 3 d",
    "P m -3 m",
    "P n -3 n :2",
    "P m -3 n",
    "P n -3 m :2",
    "F m -3 m",
    "F m -3 c",
    "F d -3 m :2",
    "F d -3 c :2",
    "I m -3 m",
    "I a -3 d",
];

/// The list of Hall crystal symmetry symbols in the same order as in the handbook (and the Hermann Mauguin table above)
const HALL_SYMBOL: &[&str] = &[
    "P 1",
    "-P 1",
    "P 2y",
    "P 2yb",
    "C 2y",
    "P -2y",
    "P -2yc",
    "C -2y",
    "C -2yc",
    "-P 2y",
    "-P 2yb",
    "-C 2y",
    "-P 2yc",
    "-P 2ybc",
    "-C 2yc",
    " P 2 2",
    "P 2c 2",
    "P 2 2ab",
    "P 2ac 2ab",
    "C 2c 2",
    "C 2 2",
    "F 2 2",
    "I 2 2",
    "I 2b 2c",
    "P 2 -2",
    "P 2c -2",
    "P 2 -2c",
    "P 2 -2a",
    "P 2c -2ac",
    "P 2 -2bc",
    "P 2ac -2",
    "P 2 -2ab",
    "P 2c -2n",
    "P 2 -2n",
    "C 2 -2",
    "C 2c -2",
    "C 2 -2c",
    "A 2 -2",
    "A 2 -2b",
    "A 2 -2a",
    "A 2 -2ab",
    "F 2 -2",
    "F 2 -2d",
    "I 2 -2",
    "I 2 -2c",
    "I 2 -2a",
    "-P 2 2",
    "-P 2ab 2bc",
    "-P 2 2c",
    "-P 2ab 2b",
    "-P 2a 2a",
    "-P 2a 2bc",
    "-P 2ac 2",
    "-P 2a 2ac",
    "-P 2 2ab",
    "-P 2ab 2ac",
    "-P 2c 2b",
    "-P 2 2n",
    "-P 2ab 2a",
    "-P 2n 2ab",
    "-P 2ac 2ab",
    "-P 2ac 2n",
    "-C 2c 2",
    "-C 2ac 2",
    "-C 2 2",
    "-C 2 2c",
    "-C 2a 2",
    "-C 2a 2ac",
    "-F 2 2",
    "-F 2uv 2vw",
    "-I 2 2",
    "-I 2 2c",
    "-I 2b 2c",
    "-I 2b 2",
    "P 4",
    "P 4w",
    "P 4c",
    "P 4cw",
    "I 4",
    "I 4bw",
    "P -4",
    "I -4",
    "-P 4",
    "-P 4c",
    "-P 4a",
    "-P 4bc",
    "-I 4",
    "-I 4ad",
    "P 4 2",
    "P 4ab 2ab",
    "P 4w 2c",
    "P 4abw 2nw",
    "P 4c 2",
    "P 4n 2n",
    "P 4cw 2c",
    "P 4nw 2abw",
    "I 4 2",
    "I 4bw 2bw",
    "P 4 -2",
    "P 4 -2ab",
    "P 4c -2c",
    "P 4n -2n",
    "P 4 -2c",
    "P 4 -2n",
    "P 4c -2",
    "P 4c -2ab",
    "I 4 -2",
    "I 4 -2c",
    "I 4bw -2",
    "I 4bw -2c",
    "P -4 2",
    "P -4 2c",
    "P -4 2ab",
    "P -4 2n",
    "P -4 -2",
    "P -4 -2c",
    "P -4 -2ab",
    "P -4 -2n",
    "I -4 -2",
    "I -4 -2c",
    "I -4 2",
    "I -4 2bw",
    "-P 4 2",
    "-P 4 2c",
    "-P 4a 2b",
    "-P 4a 2bc",
    "-P 4 2ab",
    "-P 4 2n",
    "-P 4a 2a",
    "-P 4a 2ac",
    "-P 4c 2",
    "-P 4c 2c",
    "-P 4ac 2b",
    "-P 4ac 2bc",
    "-P 4c 2ab",
    "-P 4n 2n",
    "-P 4ac 2a",
    "-P 4ac 2ac",
    "-I 4 2",
    "-I 4 2c",
    "-I 4bd 2",
    "-I 4bd 2c",
    "P 3",
    "P 31",
    "P 32",
    "R 3",
    "-P 3",
    "-R 3",
    "P 3 2",
    "P 3 2\"",
    "P 31 2 (0 0 4)",
    "P 31 2\"",
    "P 32 2 (0 0 2)",
    "P 32 2\"",
    "R 3 2\"",
    "P 3 -2\"",
    "P 3 -2",
    "P 3 -2\"c",
    "P 3 -2c",
    "R 3 -2\"",
    "R 3 -2\"c",
    "-P 3 2",
    "-P 3 2c",
    "-P 3 2\"",
    "-P 3 2\"c",
    "-R 3 2\"",
    "-R 3 2\"c",
    "P 6",
    "P 61",
    "P 65",
    "P 62",
    "P 64",
    "P 6c",
    "P -6",
    "-P 6",
    "-P 6c",
    "P 6 2",
    "P 61 2 (0 0 5)",
    "P 65 2 (0 0 1)",
    "P 62 2 (0 0 4)",
    "P 64 2 (0 0 2)",
    "P 6c 2c",
    "P 6 -2",
    "P 6 -2c",
    "P 6c -2",
    "P 6c -2c",
    "P -6 2",
    "P -6c 2",
    "P -6 -2",
    "P -6c -2c",
    "-P 6 2",
    "-P 6 2c",
    "-P 6c 2",
    "-P 6c 2c",
    "P 2 2 3",
    "F 2 2 3",
    "I 2 2 3",
    "P 2ac 2ab 3",
    "I 2b 2c 3",
    "-P 2 2 3",
    "-P 2ab 2bc 3",
    "-F 2 2 3",
    "-F 2uv 2vw 3",
    "-I 2 2 3",
    "-P 2ac 2ab 3",
    "-I 2b 2c 3",
    "P 4 2 3",
    "P 4n 2 3",
    "F 4 2 3",
    "F 4d 2 3",
    "I 4 2 3",
    "P 4acd 2ab 3",
    "P 4bd 2ab 3",
    "I 4bd 2c 3",
    "P -4 2 3",
    "F -4 2 3",
    "I -4 2 3",
    "P -4n 2 3",
    "F -4a 2 3",
    "I -4bd 2c 3",
    "-P 4 2 3",
    "-P 4a 2bc 3",
    "-P 4n 2 3",
    "-P 4bc 2bc 3",
    "-F 4 2 3",
    "-F 4a 2 3",
    "-F 4vw 2vw 3",
    "-F 4ud 2vw 3",
    "-I 4 2 3",
    "-I 4bd 2c 3",
];

/// Reworked from CCTBX output (Jan 2021)
const SYMBOL_TRANSFORMATION: &[&[[[f64; 4]; 3]]] = &[
    &[],
    &[[
        [-1.0, 0.0, 0.0, 0.0],
        [0.0, -1.0, 0.0, 0.0],
        [0.0, 0.0, -1.0, 0.0],
    ]],
    &[[
        [-1.0, 0.0, 0.0, 0.0],
        [0.0, 1.0, 0.0, 0.0],
        [0.0, 0.0, -1.0, 0.0],
    ]],
    &[[
        [-1.0, 0.0, 0.0, 0.0],
        [0.0, 1.0, 0.0, 0.5],
        [0.0, 0.0, -1.0, 0.0],
    ]],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[[
        [1.0, 0.0, 0.0, 0.0],
        [0.0, -1.0, 0.0, 0.0],
        [0.0, 0.0, 1.0, 0.0],
    ]],
    &[[
        [1.0, 0.0, 0.0, 0.0],
        [0.0, -1.0, 0.0, 0.0],
        [0.0, 0.0, 1.0, 0.5],
    ]],
    &[
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 1.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 1.25],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 1.25],
            [0.0, -1.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.25],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.25],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.25],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.75],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.75],
            [1.0, 0.0, 0.0, -0.75],
            [0.0, 0.0, -1.0, -0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 1.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.25],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.25],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.75],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.25],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.75],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.25],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, -0.75],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, -0.75],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, -0.25],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, -0.75],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, -0.75],
            [0.0, 0.0, 1.0, -0.75],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, -1.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.8333333333333334],
        ],
        [
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.8333333333333334],
        ],
        [
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.8333333333333334],
        ],
        [
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 1.1666666666666667],
        ],
        [
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 1.1666666666666667],
        ],
        [
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 1.1666666666666667],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.8333333333333334],
        ],
        [
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.8333333333333334],
        ],
        [
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.8333333333333334],
        ],
        [
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, -0.16666666666666666],
        ],
        [
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, -0.16666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 0.0, 1.0, -0.16666666666666666],
        ],
        [
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 1.1666666666666667],
        ],
        [
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 1.1666666666666667],
        ],
        [
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 1.1666666666666667],
        ],
        [
            [-1.0, 0.0, 0.0, 0.3333333333333333],
            [0.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [0.0, 1.0, 0.0, 0.3333333333333333],
            [-1.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [1.0, -1.0, 0.0, 0.3333333333333333],
            [1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.3333333333333333],
            [-1.0, 0.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.16666666666666666],
        ],
        [
            [1.0, 0.0, 0.0, 0.3333333333333333],
            [1.0, -1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.16666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.3333333333333333],
            [0.0, 1.0, 0.0, 0.6666666666666666],
            [0.0, 0.0, 1.0, 0.16666666666666666],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.16666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.8333333333333334],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.8333333333333334],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.16666666666666666],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.16666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.8333333333333334],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.8333333333333334],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.16666666666666666],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.8333333333333334],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.16666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.16666666666666666],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.8333333333333334],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.6666666666666666],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.3333333333333333],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.3333333333333333],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.6666666666666666],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
    ],
    &[
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [-1.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, -1.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [1.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 1.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.25],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.25],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.75],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, -0.25],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.25],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.25],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.25],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.25],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, -0.25],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.75],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, -0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.25],
            [1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.0],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, -1.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, -0.5],
        ],
    ],
    &[
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 1.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
            [0.0, 1.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
            [0.0, -1.0, 0.0, 1.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 1.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, 1.25],
            [1.0, 0.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [0.0, 1.0, 0.0, 1.25],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 1.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, 1.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 1.25],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, -1.0, 1.25],
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, -1.0, 0.75],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, -1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 1.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 1.25],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 1.25],
            [0.0, 1.0, 0.0, 1.25],
            [1.0, 0.0, 0.0, 0.25],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.25],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, -1.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 1.25],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
            [0.0, 1.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 1.25],
            [0.0, -1.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 1.25],
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, 1.0, 1.25],
            [0.0, -1.0, 0.0, 1.25],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 1.25],
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 1.25],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 1.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, 1.0, 1.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, 1.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 1.0],
            [0.0, -1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, 1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.75],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.75],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 1.25],
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.25],
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 1.25],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 1.25],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 1.25],
            [0.0, 1.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 1.25],
            [0.0, 1.0, 0.0, 1.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [0.0, -1.0, 0.0, 1.25],
            [1.0, 0.0, 0.0, 1.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, 1.0, 1.25],
            [0.0, -1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 1.25],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, -1.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 1.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
            [1.0, 0.0, 0.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [0.0, 1.0, 0.0, 1.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, -0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.75],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.75],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.75],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [0.0, -1.0, 0.0, -0.75],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.75],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.25],
            [0.0, -1.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.75],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, -1.0, 0.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.75],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, -0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 1.0, 0.0, -0.25],
        ],
        [
            [-1.0, 0.0, 0.0, -0.75],
            [0.0, 0.0, -1.0, -0.25],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, -0.75],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.75],
            [1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, -1.0, -0.25],
            [0.0, -1.0, 0.0, -0.75],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.5],
            [1.0, 0.0, 0.0, -0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.25],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, -0.25],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [0.0, -1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, -0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.25],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 1.0],
            [-1.0, 0.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.75],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.75],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.75],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, -0.75],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.75],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.75],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.75],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, -0.75],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, -0.75],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.25],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, -0.25],
            [0.0, -1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, -0.25],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 1.25],
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 1.25],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.25],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 1.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 1.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 1.25],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.25],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, -0.25],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.25],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, -0.25],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.75],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, -0.75],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, -0.5],
            [1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, -0.25],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 1.25],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 1.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 1.25],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 1.25],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 1.25],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 1.25],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 1.0],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, -0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.25],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, -0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, -0.25],
        ],
        [
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, -0.25],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, -0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, -0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, -0.75],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 1.0, 0.0, -0.75],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.75],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.25],
            [0.0, -1.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.25],
            [1.0, 0.0, 0.0, -0.75],
            [0.0, -1.0, 0.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.75],
            [0.0, 0.0, 1.0, -0.5],
            [0.0, -1.0, 0.0, -0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 1.0, 0.0, -0.25],
        ],
        [
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.25],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, -0.25],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.25],
            [1.0, 0.0, 0.0, -0.75],
        ],
        [
            [0.0, 0.0, -1.0, -0.25],
            [0.0, -1.0, 0.0, -0.75],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.75],
            [0.0, 0.0, -1.0, -0.5],
            [1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.25],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, -0.25],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, -0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, -0.75],
            [-1.0, 0.0, 0.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 1.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [-1.0, 0.0, 0.0, 1.25],
            [0.0, 1.0, 0.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 1.0],
            [0.0, 1.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.75],
            [-1.0, 0.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 1.0],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.75],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 1.25],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.75],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.75],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, -0.25],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.25],
            [1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, -1.0, -0.25],
            [0.0, -1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.75],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, -0.25],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.25],
            [-1.0, 0.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.25],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, -0.75],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.25],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [1.0, 0.0, 0.0, -0.75],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.5],
            [0.0, -1.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, -0.25],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [-1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, -0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, -0.75],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.5],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, -0.25],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, -0.75],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [-1.0, 0.0, 0.0, 1.25],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, -1.0, 1.0],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 1.25],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 1.0],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [0.0, -1.0, 0.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 1.25],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, -1.0, 0.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, -0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 1.0, 0.0, -0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.25],
            [1.0, 0.0, 0.0, -0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, -0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, -0.25],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
    ],
    &[
        [
            [0.0, -1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, 1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, 0.25],
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, -0.75],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, -0.5],
        ],
        [
            [0.0, -1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, -0.75],
            [0.0, 0.0, 1.0, -0.75],
        ],
        [
            [1.0, 0.0, 0.0, -0.5],
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, 1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.75],
            [0.0, -1.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.25],
            [0.0, -1.0, 0.0, -0.75],
        ],
        [
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, -0.5],
        ],
        [
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, -0.75],
            [0.0, 1.0, 0.0, -0.75],
        ],
        [
            [0.0, 0.0, 1.0, -0.5],
            [-1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.25],
            [0.0, 1.0, 0.0, -0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, -0.5],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, -0.75],
            [0.0, -1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, -1.0, -0.5],
            [1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, 1.0, -0.25],
            [0.0, 1.0, 0.0, -0.25],
            [1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 1.0, 0.0, -0.5],
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, -0.5],
        ],
        [
            [0.0, 0.0, -1.0, -0.75],
            [0.0, 1.0, 0.0, -0.75],
            [-1.0, 0.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.75],
            [0.0, -1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, -0.75],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, 1.0, 0.75],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 1.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 1.0],
        ],
        [
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 1.25],
            [0.0, 0.0, -1.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 1.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 1.25],
            [0.0, 1.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
            [0.0, 1.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 1.0],
        ],
        [
            [1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, 1.0, 1.25],
            [0.0, -1.0, 0.0, 1.25],
        ],
        [
            [0.0, 0.0, -1.0, 1.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [-1.0, 0.0, 0.0, 0.75],
            [0.0, 0.0, -1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 1.0],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, 1.0, 1.25],
            [0.0, 1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 1.0],
            [-1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, -1.0, 0.75],
            [0.0, -1.0, 0.0, 0.75],
            [-1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, -1.0, 0.0, 1.0],
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 1.0],
        ],
        [
            [0.0, 0.0, 1.0, 1.25],
            [0.0, -1.0, 0.0, 1.25],
            [1.0, 0.0, 0.0, 0.75],
        ],
        [
            [0.0, 0.0, -1.0, 1.25],
            [0.0, 1.0, 0.0, 0.75],
            [1.0, 0.0, 0.0, 1.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, -1.0, 0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, -0.25],
        ],
        [
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.0],
        ],
        [
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, -0.25],
            [0.0, 0.0, 1.0, -0.25],
        ],
        [
            [1.0, 0.0, 0.0, 0.0],
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, 1.0, 0.5],
        ],
        [
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, -0.25],
            [0.0, -1.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, -1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, -1.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, -1.0, 0.5],
            [1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.0],
        ],
        [
            [-1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, -1.0, -0.25],
            [0.0, 1.0, 0.0, -0.25],
        ],
        [
            [0.0, 0.0, 1.0, 0.0],
            [-1.0, 0.0, 0.0, 0.5],
            [0.0, 1.0, 0.0, 0.5],
        ],
        [
            [1.0, 0.0, 0.0, 0.25],
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 0.25],
        ],
        [
            [0.0, -1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, -1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.0],
            [1.0, 0.0, 0.0, 0.5],
        ],
        [
            [0.0, 0.0, -1.0, -0.25],
            [0.0, -1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.5],
            [0.0, 0.0, -1.0, 0.0],
            [1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, 1.0, 0.25],
            [0.0, 1.0, 0.0, 0.25],
            [1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0, 0.5],
            [-1.0, 0.0, 0.0, 0.0],
        ],
        [
            [0.0, 0.0, -1.0, -0.25],
            [0.0, 1.0, 0.0, -0.25],
            [-1.0, 0.0, 0.0, 0.25],
        ],
        [
            [0.0, 0.0, 1.0, -0.25],
            [0.0, -1.0, 0.0, 0.25],
            [-1.0, 0.0, 0.0, -0.25],
        ],
    ],
];
