#![cfg(test)]

use crate::str::ParseError::{self, Reject, Indeterminate};
use crate::str::{Parse, Empty, Match};

/// Matches the string "asdf" character by character
fn asdf<'a>(input: &'a str) -> Result<(&'a str, String), ParseError> {
  Match("a")
    .fuse(Match("s"))
    .fuse(Match("d"))
    .fuse(Match("f"))
    .parse(&input)
}

#[test]
fn asdf_test() {
  assert_eq!(asdf("hello world!"), Err(Reject));
  assert_eq!(asdf("asdf world!"), Ok((" world!", "asdf".to_owned())));
  assert_eq!(asdf("asdf"), Ok(("", "asdf".to_owned())));
  assert_eq!(asdf("a"), Err(Indeterminate));
}

/// Matches a number
fn number<'a>(input: &'a str) -> Result<(&'a str, String), ParseError> {
  Match("0")
    .or(Match("1"))
    .or(Match("2"))
    .or(Match("3"))
    .or(Match("4"))
    .or(Match("5"))
    .or(Match("6"))
    .or(Match("7"))
    .or(Match("8"))
    .or(Match("9"))
    .parse(&input)
}

#[test]
fn number_test() {
  assert_eq!(number("t"), Err(Reject));
  assert_eq!(number("F"), Err(Reject));
  assert_eq!(number("1"), Ok(("", "1".to_owned())));
}

/// Matches a uppercase or lowercase letter
fn letter<'a>(input: &'a str) -> Result<(&'a str, String), ParseError> {
  let c = input.chars().next()
    .ok_or(ParseError::Indeterminate)?;

  if 'a' <= c && c <= 'z' || 'A' <= c && c <= 'Z' {
    Ok((&input[c.len_utf8()..], c.to_string()))
  } else {
    Err(ParseError::Reject)
  }
}

#[test]
fn letter_test() {
  assert_eq!(letter("t"), Ok(("", "t".to_owned())));
  assert_eq!(letter("F"), Ok(("", "F".to_owned())));
  assert_eq!(letter("1"), Err(Reject));
}

/// Matches a email in the form "XXXX@YYY.DOMAIN"
fn email<'a>(input: &'a str) -> Result<(&'a str, String), ParseError> {
  letter
    .fuse(letter)
    .fuse(letter)
    .fuse(letter)
    .fuse(Match("@"))
    .fuse(letter)
    .fuse(letter)
    .fuse(letter)
    .fuse(Match("."))
    .fuse(
      Match("com")
        .or(Match("org"))
        .or(Match("co.uk"))
        .or(Match("ca"))
        .or(Match("io"))
    )
    .parse(input)
}

#[test]
fn email_test() {
  assert_eq!(email("user@hey.com"), Ok(("", "user@hey.com".to_owned())));
  assert_eq!(email("root@kek.co.uk"), Ok(("", "root@kek.co.uk".to_owned())));
  assert_eq!(email("name@longdomain.co.uk"), Err(Reject));
}

#[test]
fn dyn_or_test() {
  let mut parser = Empty;
  let mut parser = &parser as &dyn Parse<Output = String, Err = ParseError>;

  for i in 1..=5 {
    parser = parser.or(Match(&i.to_string()));
  }

  assert_eq!(parser.parse("123456789"), Ok(("23456789", "1".to_owned())));
}

#[test]
fn dyn_fuse_test() {
  let mut parser = Empty;
  let mut parser = &parser as &dyn Parse<Output = String, Err = ParseError>;

  for i in 1..=5 {
    parser = parser.fuse(Match(&i.to_string()));
  }

  assert_eq!(parser.parse("123456789"), Ok(("6789", "12345".to_owned())));
}
