//! Static-dispatch based parser combinators.

use std::{error::Error, fmt::Display, iter::{self, Extend}};

/// Denotes the failure to parse a given input
#[derive(Debug, PartialEq, Eq, Clone)]
pub enum ParseError {
  /// General rejection: an invalid input has been rejected for an unspecified reason
  Reject,
  /// The input is insufficiently short to conclusively determine if it matches
  Indeterminate,
}

impl Display for ParseError {
  fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
    match self {
      ParseError::Reject => write!(f, "Input rejected"),
      ParseError::Indeterminate => write!(f, "Insufficent input to conclusively determine validity"),
    }
  }
}

impl Error for ParseError {}


/// Attempt to parse a string into some output.
/// If the string is accepted, `Ok` includes a slice of `input` and the parsed value in Output.
/// Otherwise `Err` describes the error associated with why the input was rejected.
pub trait Parse {
  type Output;
  type Err;

  fn parse<'a>(&self, input: &'a str) -> Result<(&'a str, Self::Output), Self::Err>;

  /// Takes the union of both combinators, accepting input which passes either parser.
  /// This is evaluated lazily: if the input would pass both combinators, the first from `self` is used.
  fn or<O>(self, other: O) -> Or<Self, O>
  where O: Parse<Output = Self::Output, Err = Self::Err>, Self: Sized {
    Or(self, other)
  }

  /// Takes the concatenation of both combinators, accepting input which passes this parser first and then the other.
  /// If the first combinator returns a different input reference, that is passed along to the second combinator.
  fn fuse<O>(self, other: O) -> Fuse<Self, O>
  where O: Parse<Output = Self::Output, Err = Self::Err>,
    Self: Sized,
    Self::Output: Extend<Self::Output> {
    Fuse(self, other)
  }
}


/// The intermediate type for [Parse::or], see method for documentation.
#[derive(Debug, PartialEq, Eq, Clone)]
pub struct Or<L, R>(L, R);

impl<L, R> Parse for Or<L, R>
where L: Parse, R: Parse<Output = L::Output, Err = L::Err> {
  type Output = L::Output;
  type Err = L::Err;

  fn parse<'a>(&self, input: &'a str) -> Result<(&'a str, Self::Output), Self::Err> {
    match self.0.parse(&input) {
      Ok(res) => Ok(res),
      Err(_) => self.1.parse(input),
    }
  }
}


/// The intermediate type for [Parse::fuse], see method for documentation.
#[derive(Debug, PartialEq, Eq, Clone)]
pub struct Fuse<L, R>(L, R);

impl<L, R, E> Parse for Fuse<L, R>
where L: Parse<Output = E>, R: Parse<Err = L::Err>,
  E: Extend<R::Output> {
  type Output = L::Output;
  type Err = L::Err;

  fn parse<'a>(&self, input: &'a str) -> Result<(&'a str, Self::Output), Self::Err> {
    let (next, mut out) = self.0.parse(&input)?;
    let (next, out2) = self.1.parse(next)?;
    out.extend(iter::once(out2));
    Ok((next, out))
  }
}


impl<F, R, E> Parse for F
where F: Fn(&str) -> Result<(&str, R), E> {
  type Output = R;
  type Err = E;

  fn parse<'a>(&self, input: &'a str) -> Result<(&'a str, Self::Output), Self::Err> {
    self(input)
  }
}


/// Accept the empty string for any input.
/// Empty is infallible and always returns `Ok`.
#[derive(Debug, PartialEq, Eq, Clone)]
pub struct Empty;

impl Parse for Empty {
  type Output = String;
  type Err = ParseError;  // Infallible; convert to never type (!) when stable

  fn parse<'a>(&self, input: &'a str) -> Result<(&'a str, Self::Output), Self::Err> {
    Ok((input, String::new()))
  }
}


/// Accept the input if the input matches the contained character.
#[derive(Debug, PartialEq, Eq, Clone)]
pub struct Match<'a>(pub &'a str);

impl<'a> Parse for Match<'a> {
  type Output = String;
  type Err = ParseError;

  fn parse<'b>(&self, input: &'b str) -> Result<(&'b str, Self::Output), Self::Err> {
    if self.0.len() > input.len() {
      Err(ParseError::Indeterminate)
    } else if input.starts_with(self.0) {
      Ok((&input[self.0.len()..], self.0.to_owned()))
    } else {
      Err(ParseError::Reject)
    }
  }
}
