use crate::instructions::utils::ONE_MINUTE;

use {
    crate::{errors::*, state::*},
    anchor_lang::{prelude::*, solana_program::system_program},
    std::mem::size_of,
};

#[derive(Accounts)]
#[instruction(
    process_at: u64,
    index_bump: u8,
    namespace_bump: u8
)]
pub struct CreateTaskNamespace<'info> {
    #[account(
        mut, 
        seeds = [SEED_AUTHORITY], 
        bump = authority.bump,
        owner = crate::ID,
    )]
    pub authority: Account<'info, Authority>,

    pub clock: Sysvar<'info, Clock>,

    #[account(mut)]
    pub index: AccountInfo<'info>,

    #[account(address = index_program::ID)]
    pub index_program: Program<'info, index_program::program::IndexProgram>,

    #[account(
        init,
        seeds = [
            SEED_TASK_NAMESPACE,
            process_at.to_string().as_bytes()
        ],
        bump = namespace_bump,
        payer = payer,
        space = 8 + size_of::<TaskNamespace>(),
    )]
    pub namespace: Account<'info, TaskNamespace>,

    #[account(mut)]
    pub payer: Signer<'info>,

    #[account(address = system_program::ID)]
    pub system_program: Program<'info, System>,
}

pub fn handler(
    ctx: Context<CreateTaskNamespace>,
    process_at: u64,
    index_bump: u8,
    namespace_bump: u8,
) -> ProgramResult {
    // Get accounts.
    let authority = &ctx.accounts.authority;
    let clock = &ctx.accounts.clock;
    let index = &ctx.accounts.index;
    let namespace = &mut ctx.accounts.namespace;
    let payer = &ctx.accounts.payer;
    let index_program = &ctx.accounts.index_program;
    let system_program = &ctx.accounts.system_program;

    // Validate process_at is at the top of the minute.
    require!(
        process_at % ONE_MINUTE == 0,
        ErrorCode::InvalidProcessAtIntraMinute
    );

    // Validate process_at is not in the past.
    require!(
        process_at > clock.unix_timestamp as u64,
        ErrorCode::InvalidProcessAtPast
    );

    // Initialize namespace account.
    namespace.process_at = process_at;
    namespace.bump = namespace_bump;

    // Create an index to lookup payments by (party, role) pairs.
    // (e.g. all the payments where Alice is a creditor or Bob is a debtor)
    index_program::cpi::create_index(
        CpiContext::new_with_signer(
            index_program.to_account_info(),
            index_program::cpi::accounts::CreateIndex {
                index: index.to_account_info(),
                owner: authority.to_account_info(),
                payer: payer.to_account_info(),
                namespace: namespace.to_account_info(),
                system_program: system_program.to_account_info(),
            },
            &[&[SEED_AUTHORITY, &[authority.bump]]],
        ),
        index_bump,
    )
}
