import IndexProgram from "@faktorfi/index-program";
import { BN } from "@project-serum/anchor";
import { TOKEN_PROGRAM_ID } from "@solana/spl-token";
import {
  PublicKey,
  SystemProgram,
  SYSVAR_CLOCK_PUBKEY,
  TransactionInstruction,
} from "@solana/web3.js";
import * as account from "../account";
import program from "../program";
import { Role } from "../enum";

const PERCENTAGE_BASE = 1000000000;

export type CreatePaymentProps = {
  memo: String;
  amountRaw: number;
  amountPercent: number;
  creditor: PublicKey;
  creditorTokens: PublicKey;
  debtor: PublicKey;
  debtorTokens: PublicKey;
  mint: PublicKey;
  recurrenceInterval: number;
  startAt: number;
  endAt: number;
};

export async function createPayment({
  amountRaw,
  amountPercent,
  creditor,
  creditorTokens,
  debtor,
  debtorTokens,
  mint,
  memo,
  recurrenceInterval,
  startAt,
  endAt,
}: CreatePaymentProps): Promise<TransactionInstruction> {
  // Validate request.
  if (amountPercent < 0 || amountPercent > 1) {
    throw new Error("Percentage amount must be between 0 and 1");
  }

  // Find PDAs.
  const authorityPDA = await account.authority.pda();
  const configPDA = await account.config.pda();

  const creditorPaymentNamespacePDA = await account.paymentNamespace.pda(
    creditor,
    Role.Creditor
  );

  const creditorPaymentIndexPDA = await IndexProgram.account.index.pda(
    authorityPDA.address,
    creditorPaymentNamespacePDA.address
  );

  const creditorPaymentIndexData = await IndexProgram.account.index.fetch(
    creditorPaymentIndexPDA.address
  );

  const creditorPaymentPointerPDA = await IndexProgram.account.pointer.pda(
    creditorPaymentIndexPDA.address,
    creditorPaymentIndexData.count.toString()
  );

  const debtorPaymentNamespacePDA = await account.paymentNamespace.pda(
    debtor,
    Role.Debtor
  );

  const debtorPaymentIndexPDA = await IndexProgram.account.index.pda(
    authorityPDA.address,
    debtorPaymentNamespacePDA.address
  );

  const debtorPaymentIndexData = await IndexProgram.account.index.fetch(
    debtorPaymentIndexPDA.address
  );

  const debtorPaymentPointerPDA = await IndexProgram.account.pointer.pda(
    debtorPaymentIndexPDA.address,
    debtorPaymentIndexData.count.toString()
  );

  const paymentPDA = await account.payment.pda(
    debtor,
    debtorPaymentIndexData.count
  );

  const creditorPaymentProofPDA = await IndexProgram.account.proof.pda(
    creditorPaymentIndexPDA.address,
    paymentPDA.address
  );

  const debtorPaymentProofPDA = await IndexProgram.account.proof.pda(
    debtorPaymentIndexPDA.address,
    paymentPDA.address
  );

  const taskPDA = await account.task.pda(paymentPDA.address, startAt);

  const taskIndexNamespacePDA = await account.taskNamespace.pda(startAt);

  const taskIndexPDA = await IndexProgram.account.index.pda(
    authorityPDA.address,
    taskIndexNamespacePDA.address
  );

  const taskIndexData = await IndexProgram.account.index.fetch(
    taskIndexPDA.address
  );

  const taskPointerPDA = await IndexProgram.account.pointer.pda(
    taskIndexPDA.address,
    taskIndexData.count.toString()
  );

  const taskProofPDA = await IndexProgram.account.proof.pda(
    taskIndexPDA.address,
    taskPDA.address
  );

  // Create instruction.
  return program.instruction.createPayment(
    memo,
    new BN(amountRaw),
    new BN(amountPercent * PERCENTAGE_BASE),
    new BN(recurrenceInterval),
    new BN(startAt),
    new BN(endAt),
    creditorPaymentPointerPDA.bump,
    creditorPaymentProofPDA.bump,
    debtorPaymentPointerPDA.bump,
    debtorPaymentProofPDA.bump,
    paymentPDA.bump,
    taskPDA.bump,
    taskPointerPDA.bump,
    taskProofPDA.bump,
    {
      accounts: {
        authority: authorityPDA.address,
        clock: SYSVAR_CLOCK_PUBKEY,
        config: configPDA.address,
        creditor: creditor,
        creditorPaymentIndex: creditorPaymentIndexPDA.address,
        creditorPaymentNamespace: creditorPaymentNamespacePDA.address,
        creditorPaymentPointer: creditorPaymentPointerPDA.address,
        creditorPaymentProof: creditorPaymentProofPDA.address,
        creditorTokens: creditorTokens,
        debtor: debtor,
        debtorPaymentIndex: debtorPaymentIndexPDA.address,
        debtorPaymentNamespace: debtorPaymentNamespacePDA.address,
        debtorPaymentPointer: debtorPaymentPointerPDA.address,
        debtorPaymentProof: debtorPaymentProofPDA.address,
        debtorTokens: debtorTokens,
        indexProgram: IndexProgram.programId,
        mint: mint,
        payment: paymentPDA.address,
        systemProgram: SystemProgram.programId,
        task: taskPDA.address,
        taskIndex: taskIndexPDA.address,
        taskNamespace: taskIndexNamespacePDA.address,
        taskPointer: taskPointerPDA.address,
        taskProof: taskProofPDA.address,
        tokenProgram: TOKEN_PROGRAM_ID,
      },
    }
  );
}
