import assert from "assert";
import * as anchor from "@project-serum/anchor";
import { Provider } from "@project-serum/anchor";
import { Token, TOKEN_PROGRAM_ID } from "@solana/spl-token";
import { Keypair, LAMPORTS_PER_SOL, PublicKey } from "@solana/web3.js";
import {
  dateToSeconds,
  newSigner,
  PDA,
  signAndSubmit,
  sleepUntil,
} from "@faktorfi/utils";
import PaymentProgram from ".";
import IndexProgram from "@faktorfi/index-program";

// Mints
const WSOL_MINT = new PublicKey("So11111111111111111111111111111111111111112");

// Time
const ONE_MINUTE = 60;

describe("Payment Program", () => {
  // Test environment
  const provider = Provider.local();
  anchor.setProvider(provider);

  // Shared test data
  let creditor: Keypair;
  let creditorTokensAddress: PublicKey;
  let debtor: Keypair;
  let debtorTokensAddress: PublicKey;
  let worker: Keypair;
  let paymentPDA_0: PDA;
  let paymentPDA_1: PDA;
  let taskPDA_0: PDA;
  let taskPDA_1: PDA;
  let taskProcessAt: number;
  let authorityPDA: PDA, configPDA: PDA, treasuryPDA: PDA;

  before(async () => {
    creditor = await newSigner(provider.connection);
    debtor = await newSigner(provider.connection);
    worker = await newSigner(provider.connection);
    authorityPDA = await PaymentProgram.account.authority.pda();
    configPDA = await PaymentProgram.account.config.pda();
    treasuryPDA = await PaymentProgram.account.treasury.pda();
  });

  it("initializes the program", async () => {
    // Generate test data.
    const signer = await newSigner(provider.connection);
    const transferFeeDistributor = 1000;
    const transferFeeProgram = 1000;

    // Create instructions.
    const ix = await PaymentProgram.instruction.initializeProgram({
      signer: signer.publicKey,
      transferFeeDistributor,
      transferFeeProgram,
    });

    // Sign and submit transaction.
    await signAndSubmit(PaymentProgram.connection, [ix], signer);

    // Validate authority account data.
    const authorityData = await PaymentProgram.account.authority.fetch(
      authorityPDA.address
    );
    assert.ok(authorityData.bump === authorityPDA.bump);

    // Validate config account data.
    const configData = await PaymentProgram.account.config.fetch(
      configPDA.address
    );
    assert.ok(
      configData.transferFeeDistributor.toNumber() === transferFeeDistributor
    );
    assert.ok(configData.transferFeeProgram.toNumber() === transferFeeProgram);
    assert.ok(configData.bump === configPDA.bump);

    // Validate treasury account data.
    const treasuryData = await PaymentProgram.account.treasury.fetch(
      treasuryPDA.address
    );
    assert.ok(treasuryData.bump === treasuryPDA.bump);
  });

  it("creates a creditor payment index", async () => {
    // Generate test data
    const signer = await newSigner(provider.connection);

    // Generate instruction.
    const ix = await PaymentProgram.instruction.createPaymentNamespace({
      party: creditor.publicKey,
      payer: signer.publicKey,
      role: PaymentProgram.enum.Role.Creditor,
    });

    // Sign and submit transaction.
    await signAndSubmit(PaymentProgram.connection, [ix], signer);

    // Validate index account data.
    const namespacePDA = await PaymentProgram.account.paymentNamespace.pda(
      creditor.publicKey,
      PaymentProgram.enum.Role.Creditor
    );
    const indexPDA = await IndexProgram.account.index.pda(
      authorityPDA.address,
      namespacePDA.address
    );
    const indexData = await IndexProgram.account.index.fetch(indexPDA.address);

    assert.ok(indexData.owner.toString() === authorityPDA.address.toString());
    assert.ok(
      indexData.namespace.toString() === namespacePDA.address.toString()
    );
    assert.ok(indexData.count.toNumber() === 0);
    assert.ok(indexData.bump === indexPDA.bump);

    // Validate namespace account
    const namespaceData = await PaymentProgram.account.paymentNamespace.fetch(
      namespacePDA.address
    );

    assert.ok(namespaceData.party.toString() === creditor.publicKey.toString());
    assert.ok(
      Object.keys(namespaceData.role)[0].toString() ===
        PaymentProgram.enum.Role.toString(
          PaymentProgram.enum.Role.Creditor
        ).toLowerCase()
    );
    assert.ok(namespaceData.bump === namespacePDA.bump);
  });

  it("creates a debtor payment index", async () => {
    // Generate test data
    const signer = await newSigner(provider.connection);

    // Generate instruction.
    const ix = await PaymentProgram.instruction.createPaymentNamespace({
      party: debtor.publicKey,
      payer: signer.publicKey,
      role: PaymentProgram.enum.Role.Debtor,
    });

    const namespacePDA = await PaymentProgram.account.paymentNamespace.pda(
      debtor.publicKey,
      PaymentProgram.enum.Role.Debtor
    );

    // Sign and submit transaction.
    await signAndSubmit(PaymentProgram.connection, [ix], signer);

    // Validate index account data.
    const indexPDA = await IndexProgram.account.index.pda(
      authorityPDA.address,
      namespacePDA.address
    );
    const indexData = await IndexProgram.account.index.fetch(indexPDA.address);
    assert.ok(indexData.owner.toString() === authorityPDA.address.toString());
    assert.ok(
      indexData.namespace.toString() === namespacePDA.address.toString()
    );
    assert.ok(indexData.count.toNumber() === 0);
    assert.ok(indexData.bump === indexPDA.bump);

    // Validate namespace account
    const namespaceData = await PaymentProgram.account.paymentNamespace.fetch(
      namespacePDA.address
    );
    assert.ok(namespaceData.party.toString() === debtor.publicKey.toString());
    assert.ok(
      Object.keys(namespaceData.role)[0].toString() ===
        PaymentProgram.enum.Role.toString(
          PaymentProgram.enum.Role.Debtor
        ).toLowerCase()
    );
    assert.ok(namespaceData.bump === namespacePDA.bump);
  });

  it("creates a task index", async () => {
    // Generate test data.
    const signer = await newSigner(provider.connection);
    const thisMinute = new Date(new Date().setSeconds(0, 0));
    const nextMinute = new Date(thisMinute.getTime() + ONE_MINUTE * 1000);
    taskProcessAt = dateToSeconds(nextMinute);

    // Generate instruction.
    const namespacePDA = await PaymentProgram.account.taskNamespace.pda(
      taskProcessAt
    );
    const ix = await PaymentProgram.instruction.createTaskNamespace({
      payer: signer.publicKey,
      processAt: nextMinute,
    });

    // Sign and submit transaction.
    await signAndSubmit(PaymentProgram.connection, [ix], signer);

    // Validate index account data.
    const indexPDA = await IndexProgram.account.index.pda(
      authorityPDA.address,
      namespacePDA.address
    );
    const indexData = await IndexProgram.account.index.fetch(indexPDA.address);
    assert.ok(indexData.owner.toString() === authorityPDA.address.toString());
    assert.ok(
      indexData.namespace.toString() === namespacePDA.address.toString()
    );
    assert.ok(indexData.count.toNumber() === 0);
    assert.ok(indexData.bump === indexPDA.bump);

    // Validate namespace account
    const namespaceData = await PaymentProgram.account.taskNamespace.fetch(
      namespacePDA.address
    );

    assert.ok(namespaceData.processAt.toNumber() === dateToSeconds(nextMinute));
    assert.ok(namespaceData.bump === namespacePDA.bump);
  });

  it("creates a one-time raw payment", async () => {
    // Generate test data
    const memo = "Test";
    const amountRaw = 0.2 * LAMPORTS_PER_SOL;
    const amountPercent = 0;
    const recurrenceInterval = 0;
    const startAt = taskProcessAt;
    const endAt = startAt;

    // Generate token accounts.
    creditorTokensAddress = await Token.createWrappedNativeAccount(
      provider.connection,
      TOKEN_PROGRAM_ID,
      creditor.publicKey,
      creditor,
      0
    );
    debtorTokensAddress = await Token.createWrappedNativeAccount(
      provider.connection,
      TOKEN_PROGRAM_ID,
      debtor.publicKey,
      debtor,
      LAMPORTS_PER_SOL
    );

    // Generate instruction.
    const ix = await PaymentProgram.instruction.createPayment({
      amountRaw: amountRaw,
      amountPercent: amountPercent,
      memo: memo,
      creditor: creditor.publicKey,
      creditorTokens: creditorTokensAddress,
      debtor: debtor.publicKey,
      debtorTokens: debtorTokensAddress,
      mint: WSOL_MINT,
      recurrenceInterval: recurrenceInterval,
      startAt: startAt,
      endAt: endAt,
    });

    // Calculate PDAs
    paymentPDA_0 = await PaymentProgram.account.payment.pda(
      debtor.publicKey,
      new anchor.BN(0)
    );
    taskPDA_0 = await PaymentProgram.account.task.pda(
      paymentPDA_0.address,
      startAt
    );

    // Sign and submit transaction.
    await signAndSubmit(PaymentProgram.connection, [ix], debtor);

    // Validate payment data.
    const payment = await PaymentProgram.account.payment.fetch(
      paymentPDA_0.address
    );
    assert.ok(payment.memo === memo);
    assert.ok(payment.debtor.toString() === debtor.publicKey.toString());
    assert.ok(
      payment.debtorTokens.toString() === debtorTokensAddress.toString()
    );
    assert.ok(payment.creditor.toString() === creditor.publicKey.toString());
    assert.ok(
      payment.creditorTokens.toString() === creditorTokensAddress.toString()
    );
    assert.ok(payment.mint.toString() === WSOL_MINT.toString());
    assert.ok(payment.amountRaw.toNumber() === amountRaw);
    assert.ok(payment.amountPercent.toNumber() === amountPercent);
    assert.ok(payment.recurrenceInterval.toNumber() === recurrenceInterval);
    assert.ok(payment.startAt.toNumber() === startAt);
    assert.ok(payment.endAt.toNumber() === endAt);
    assert.ok(payment.bump === paymentPDA_0.bump);

    // Validate task data.
    const task = await PaymentProgram.account.task.fetch(taskPDA_0.address);
    assert.ok(task.payment.toString() === paymentPDA_0.address.toString());
    assert.ok(Object.keys(task.status)[0].toString() === "pending");
    assert.ok(task.processAt.toNumber() === startAt);
    assert.ok(task.bump === taskPDA_0.bump);
  });

  it("creates a one-time percentage payment", async () => {
    // Generate test data
    const memo = "Test";
    const amountRaw = 0;
    const amountPercent = 0.5;
    const recurrenceInterval = 0;
    const startAt = taskProcessAt;
    const endAt = startAt;

    // Generate instruction.
    const ix = await PaymentProgram.instruction.createPayment({
      amountRaw: amountRaw,
      amountPercent: amountPercent,
      memo: memo,
      creditor: creditor.publicKey,
      creditorTokens: creditorTokensAddress,
      debtor: debtor.publicKey,
      debtorTokens: debtorTokensAddress,
      mint: WSOL_MINT,
      recurrenceInterval: recurrenceInterval,
      startAt: startAt,
      endAt: endAt,
    });
    paymentPDA_1 = await PaymentProgram.account.payment.pda(
      debtor.publicKey,
      new anchor.BN(1)
    );
    taskPDA_1 = await PaymentProgram.account.task.pda(
      paymentPDA_1.address,
      startAt
    );

    // Sign and submit transaction.
    await signAndSubmit(PaymentProgram.connection, [ix], debtor);

    // Validate payment data.
    const payment = await PaymentProgram.account.payment.fetch(
      paymentPDA_1.address
    );
    assert.ok(payment.memo === memo);
    assert.ok(payment.debtor.toString() === debtor.publicKey.toString());
    assert.ok(
      payment.debtorTokens.toString() === debtorTokensAddress.toString()
    );
    assert.ok(payment.creditor.toString() === creditor.publicKey.toString());
    assert.ok(
      payment.creditorTokens.toString() === creditorTokensAddress.toString()
    );
    assert.ok(payment.mint.toString() === WSOL_MINT.toString());
    assert.ok(payment.amountRaw.toNumber() === amountRaw);
    assert.ok(payment.amountPercent.toNumber() === amountPercent * 1000000000);
    assert.ok(payment.recurrenceInterval.toNumber() === recurrenceInterval);
    assert.ok(payment.startAt.toNumber() === startAt);
    assert.ok(payment.endAt.toNumber() === endAt);
    assert.ok(payment.bump === paymentPDA_1.bump);

    // Validate task data.
    const task = await PaymentProgram.account.task.fetch(taskPDA_1.address);
    assert.ok(task.payment.toString() === paymentPDA_1.address.toString());
    assert.ok(Object.keys(task.status)[0].toString() === "pending");
    assert.ok(task.processAt.toNumber() === startAt);
    assert.ok(task.bump === taskPDA_1.bump);
  });

  it("processes a one-time raw payment task", async () => {
    // Sleep until the task is due for processing.
    await sleepUntil(new Date((taskProcessAt + 2) * 1000));

    // Generate instruction.
    const ix = await PaymentProgram.instruction.processTask({
      signer: worker.publicKey,
      task: taskPDA_0.address,
    });

    // Sign and submit transaction.
    await signAndSubmit(PaymentProgram.connection, [ix], worker);

    // Validate task data.
    const taskData = await PaymentProgram.account.task.fetch(taskPDA_0.address);
    assert.ok(taskData.payment.toString() === paymentPDA_0.address.toString());
    assert.ok(Object.keys(taskData.status)[0].toString() === "done");
    assert.ok(
      Object.keys(taskData.transferStatus)[0].toString() === "succeeded"
    );
    assert.ok(taskData.processAt.toNumber() === taskProcessAt);
    assert.ok(taskData.bump === taskPDA_0.bump);

    // Validate debtor token balance.
    const token = new Token(
      PaymentProgram.connection,
      WSOL_MINT,
      TOKEN_PROGRAM_ID,
      worker
    );
    const debtorTokenData = await token.getAccountInfo(debtorTokensAddress);
    assert.ok(debtorTokenData.amount.toNumber() === 0.8 * LAMPORTS_PER_SOL);

    // Validate creditor token balance.
    const creditorTokenData = await token.getAccountInfo(creditorTokensAddress);
    assert.ok(creditorTokenData.amount.toNumber() === 0.2 * LAMPORTS_PER_SOL);
  });

  it("processes a one-time percentage payment task", async () => {
    // Sleep until the task is due for processing.
    await sleepUntil(new Date((taskProcessAt + 2) * 1000));

    // Generate instruction.
    const ix = await PaymentProgram.instruction.processTask({
      signer: worker.publicKey,
      task: taskPDA_1.address,
    });

    // Sign and submit transaction.
    await signAndSubmit(PaymentProgram.connection, [ix], worker);

    // Validate task data.
    const taskData = await PaymentProgram.account.task.fetch(taskPDA_1.address);
    assert.ok(taskData.payment.toString() === paymentPDA_1.address.toString());
    assert.ok(Object.keys(taskData.status)[0].toString() === "done");
    assert.ok(
      Object.keys(taskData.transferStatus)[0].toString() === "succeeded"
    );
    assert.ok(taskData.processAt.toNumber() === taskProcessAt);
    assert.ok(taskData.bump === taskPDA_1.bump);

    // Validate debtor token balance.
    const token = new Token(
      PaymentProgram.connection,
      WSOL_MINT,
      TOKEN_PROGRAM_ID,
      worker
    );
    const debtorTokenData = await token.getAccountInfo(debtorTokensAddress);
    assert.ok(debtorTokenData.amount.toNumber() === 0.4 * LAMPORTS_PER_SOL);

    // Validate creditor token balance.
    const creditorTokenData = await token.getAccountInfo(creditorTokensAddress);
    assert.ok(creditorTokenData.amount.toNumber() === 0.6 * LAMPORTS_PER_SOL);
  });

  it("processes a payment task where debtor's delegate token authorization is revoked", async () => {
    // Generate test data
    const memo = "Test";
    const amountRaw = 0.2 * LAMPORTS_PER_SOL;
    const amountPercent = 0;
    const recurrenceInterval = 0;
    const startAt = taskProcessAt;
    const endAt = startAt;

    // Generate instruction.
    const paymentPDA = await PaymentProgram.account.payment.pda(
      debtor.publicKey,
      new anchor.BN(2)
    );
    const ixA = await PaymentProgram.instruction.createPayment({
      amountRaw: amountRaw,
      amountPercent: amountPercent,
      memo: memo,
      creditor: creditor.publicKey,
      creditorTokens: creditorTokensAddress,
      debtor: debtor.publicKey,
      debtorTokens: debtorTokensAddress,
      mint: WSOL_MINT,
      recurrenceInterval: recurrenceInterval,
      startAt: startAt,
      endAt: endAt,
    });

    // Sign and submit transaction.
    await signAndSubmit(PaymentProgram.connection, [ixA], debtor);

    // Revoke delegate authorization.
    const token = new Token(
      PaymentProgram.connection,
      WSOL_MINT,
      TOKEN_PROGRAM_ID,
      worker
    );
    await token.revoke(debtorTokensAddress, debtor.publicKey, [debtor]);

    // Generate instruction.
    const taskPDA = await PaymentProgram.account.task.pda(
      paymentPDA.address,
      startAt
    );
    const ixB = await PaymentProgram.instruction.processTask({
      signer: worker.publicKey,
      task: taskPDA.address,
    });

    // Sign and submit transaction.
    await signAndSubmit(PaymentProgram.connection, [ixB], worker);

    // Validate task data.
    const taskData = await PaymentProgram.account.task.fetch(taskPDA.address);
    assert.ok(taskData.payment.toString() === paymentPDA.address.toString());
    assert.ok(Object.keys(taskData.status)[0].toString() === "done");
    assert.ok(
      Object.keys(taskData.transferStatus)[0].toString() ===
        "failedNotAuthorized"
    );
    assert.ok(taskData.processAt.toNumber() === taskProcessAt);
    assert.ok(taskData.bump === taskPDA.bump);

    // Validate debtor token balance.
    const debtorTokenData = await token.getAccountInfo(debtorTokensAddress);
    assert.ok(debtorTokenData.amount.toNumber() === 0.4 * LAMPORTS_PER_SOL);

    // Validate creditor token balance.
    const creditorTokenData = await token.getAccountInfo(creditorTokensAddress);
    assert.ok(creditorTokenData.amount.toNumber() === 0.6 * LAMPORTS_PER_SOL);
  });

  it("processes a payment task where debtor's delegate token balance is insufficient", async () => {
    // Generate test data
    const memo = "Test";
    const amountRaw = 0.2 * LAMPORTS_PER_SOL;
    const amountPercent = 0;
    const recurrenceInterval = 0;
    const startAt = taskProcessAt;
    const endAt = startAt;

    // Generate instruction.
    const paymentPDA = await PaymentProgram.account.payment.pda(
      debtor.publicKey,
      new anchor.BN(3)
    );
    const ixA = await PaymentProgram.instruction.createPayment({
      amountRaw: amountRaw,
      amountPercent: amountPercent,
      memo: memo,
      creditor: creditor.publicKey,
      creditorTokens: creditorTokensAddress,
      debtor: debtor.publicKey,
      debtorTokens: debtorTokensAddress,
      mint: WSOL_MINT,
      recurrenceInterval: recurrenceInterval,
      startAt: startAt,
      endAt: endAt,
    });

    // Sign and submit transaction.
    await signAndSubmit(PaymentProgram.connection, [ixA], debtor);

    // Set delegate token balance to be insufficient.
    const token = new Token(
      PaymentProgram.connection,
      WSOL_MINT,
      TOKEN_PROGRAM_ID,
      worker
    );
    await token.approve(
      debtorTokensAddress,
      authorityPDA.address,
      debtor.publicKey,
      [debtor],
      1
    );

    // Generate instruction.
    const taskPDA = await PaymentProgram.account.task.pda(
      paymentPDA.address,
      startAt
    );
    const ixB = await PaymentProgram.instruction.processTask({
      signer: worker.publicKey,
      task: taskPDA.address,
    });

    // Sign and submit transaction.
    await signAndSubmit(PaymentProgram.connection, [ixB], worker);

    // Validate task data.
    const taskData = await PaymentProgram.account.task.fetch(taskPDA.address);
    assert.ok(taskData.payment.toString() === paymentPDA.address.toString());
    assert.ok(Object.keys(taskData.status)[0].toString() === "done");
    assert.ok(
      Object.keys(taskData.transferStatus)[0].toString() ===
        "failedInsufficientDelegateBalance"
    );
    assert.ok(taskData.processAt.toNumber() === taskProcessAt);
    assert.ok(taskData.bump === taskPDA.bump);

    // Validate debtor token balance.
    const debtorTokenData = await token.getAccountInfo(debtorTokensAddress);
    assert.ok(debtorTokenData.amount.toNumber() === 0.4 * LAMPORTS_PER_SOL);

    // Validate creditor token balance.
    const creditorTokenData = await token.getAccountInfo(creditorTokensAddress);
    assert.ok(creditorTokenData.amount.toNumber() === 0.6 * LAMPORTS_PER_SOL);
  });

  it("processes a payment task where debtor's token balance is insufficient", async () => {
    // Generate test data
    const memo = "Test";
    const amountRaw = 10 * LAMPORTS_PER_SOL;
    const amountPercent = 0;
    const recurrenceInterval = 0;
    const startAt = taskProcessAt;
    const endAt = startAt;

    // Generate instruction.
    const paymentPDA = await PaymentProgram.account.payment.pda(
      debtor.publicKey,
      new anchor.BN(4)
    );
    const taskPDA = await PaymentProgram.account.task.pda(
      paymentPDA.address,
      startAt
    );
    const ixA = await PaymentProgram.instruction.createPayment({
      amountRaw: amountRaw,
      amountPercent: amountPercent,
      memo: memo,
      creditor: creditor.publicKey,
      creditorTokens: creditorTokensAddress,
      debtor: debtor.publicKey,
      debtorTokens: debtorTokensAddress,
      mint: WSOL_MINT,
      recurrenceInterval: recurrenceInterval,
      startAt: startAt,
      endAt: endAt,
    });

    // Sign and submit transaction.
    await signAndSubmit(PaymentProgram.connection, [ixA], debtor);

    // Generate instruction.
    const ixB = await PaymentProgram.instruction.processTask({
      signer: worker.publicKey,
      task: taskPDA.address,
    });

    // Sign and submit transaction.
    await signAndSubmit(PaymentProgram.connection, [ixB], worker);

    // Validate task data.
    const taskData = await PaymentProgram.account.task.fetch(taskPDA.address);
    assert.ok(taskData.payment.toString() === paymentPDA.address.toString());
    assert.ok(Object.keys(taskData.status)[0].toString() === "done");
    assert.ok(
      Object.keys(taskData.transferStatus)[0].toString() ===
        "failedInsufficientBalance"
    );
    assert.ok(taskData.processAt.toNumber() === taskProcessAt);
    assert.ok(taskData.bump === taskPDA.bump);

    // Validate debtor token balance.
    const token = new Token(
      PaymentProgram.connection,
      WSOL_MINT,
      TOKEN_PROGRAM_ID,
      worker
    );
    const debtorTokenData = await token.getAccountInfo(debtorTokensAddress);
    assert.ok(debtorTokenData.amount.toNumber() === 0.4 * LAMPORTS_PER_SOL);

    // Validate creditor token balance.
    const creditorTokenData = await token.getAccountInfo(creditorTokensAddress);
    assert.ok(creditorTokenData.amount.toNumber() === 0.6 * LAMPORTS_PER_SOL);
  });
});
