use crate::{OrderId, Refund, Status};

#[derive(serde::Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct CreateOrder {
    /// Http status as a text
    pub status: Status,
    /// Client should be redirected to this URI
    pub redirect_uri: String,
    /// This should be connected to your own order
    pub order_id: OrderId,
    /// This is YOUR_EXT_ORDER_ID
    pub ext_order_id: Option<String>,
}

#[derive(serde::Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct RefundDetails {
    pub order_id: Option<String>,
    pub refund: Option<Refund>,
    pub status: Status,
}

#[derive(serde::Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Refunds {
    pub refunds: Vec<Refund>,
}

#[derive(serde::Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct TransactionPayMethod {
    pub value: String,
}

#[derive(serde::Deserialize, Debug)]
#[serde(rename_all = "SCREAMING_SNAKE_CASE")]
pub enum CardProfile {
    Consumer,
    Business,
}

#[derive(serde::Deserialize, Debug)]
#[serde(rename_all = "SCREAMING_SNAKE_CASE")]
pub enum CardClassification {
    Debit,
    Credit,
}

#[derive(serde::Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct TransactionCartData {
    /// // "543402******4014",
    pub card_number_masked: String,
    /// MC (MasterCard/Maestro), VS (Visa)
    /// Example; "MC"
    pub card_scheme: String,
    pub card_profile: CardProfile,
    pub card_classification: CardClassification,
    ///  Example: "000"
    pub card_response_code: String,
    ///  Example: "000 - OK"
    pub card_response_code_desc: String,
    ///  Example: "5"
    pub card_eci_code: String,
    ///  Example: "AY",
    pub card3ds_status: String,
    ///  Example: "PL",
    pub card_bin_country: String,
    ///  Example: "MCC0111LL1121"
    pub first_transaction_id: String,
}

/// > Installment proposal on the Sandbox environment is not related to the
/// > order amount and always returns data for 480 PLN.
#[derive(serde::Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct TransactionCardInstallmentProposal {
    /// Example: "5aff3ba8-0c37-4da1-ba4a-4ff24bcc2eed"
    pub proposal_id: String,
}

#[derive(serde::Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct TransactionCart {
    pub cart_data: TransactionCartData,
    pub card_installment_proposal: TransactionCardInstallmentProposal,
}

#[derive(serde::Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Transaction {
    pub pay_method: TransactionPayMethod,
    pub payment_flow: String,
}

#[derive(serde::Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Transactions {
    pub transactions: Vec<Transaction>,
}

#[derive(serde::Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Order {
    /// Example: "{orderId}",
    pub order_id: super::OrderId,
    /// Example:  "358766",
    pub ext_order_id: Option<String>,
    /// Example: "2014-10-27T14:58:17.443+01:00",
    pub order_create_date: String,
    /// Example: "http://localhost/OrderNotify/",
    pub notify_url: Option<String>,
    /// Example: "127.0.0.1",
    pub customer_ip: String,
    /// Example: "145227",
    pub merchant_pos_id: String,
    /// Example: "New order",
    pub description: String,
    /// Example: "PLN",
    pub currency_code: String,
    /// Example: "3200",
    pub total_amount: String,
    /// Example: "NEW",
    pub status: String,
    /// Example: `[{"name":"Product1","unitPrice":"1000","quantity":"1"}]`
    pub products: Vec<super::Product>,
}

#[derive(serde::Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct OrdersInfo {
    pub orders: Vec<Order>,
    pub status: Status,
    pub properties: Option<Vec<crate::Prop>>,
}

#[derive(serde::Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct OrderInfo {
    pub order: Order,
    pub status: Status,
    pub properties: Option<Vec<crate::Prop>>,
}
