//! This module allow to create credit request during create order request

use serde::{Deserialize, Serialize};

use crate::{Address, ShoppingCart};

/// Describe customer during credit request
#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Applicant {
    /// Applicant’s email address
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    email: Option<String>,
    /// Applicant’s phone number
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    phone: Option<String>,
    /// Applicant’s first name
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    first_name: Option<String>,
    /// Applicant’s last name
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    last_name: Option<String>,
    /// Language code, ISO-639-1 compliant. Denotes the language version of
    /// PayU hosted payment page and of e-mail messages sent from PayU to the
    /// payer (supported values are here).
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    language: Option<String>,
    /// National Identification Number
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    nin: Option<String>,
    /// Section containing data about applicant’s address.
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    address: Option<Address>,
    /// Additional information about person applying for credit.
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    additional_info: Option<ApplicantAdditionalInfo>,
}

/// Allow to create credit request
#[derive(Serialize, Deserialize, Debug, Default)]
#[serde(rename_all = "camelCase")]
pub struct Credit {
    /// Section containing data of the ordered products
    #[serde(skip_serializing_if = "Option::is_none")]
    shopping_carts: Option<Vec<ShoppingCart>>,
    /// Section containing data of person applying for a credit
    #[serde(skip_serializing_if = "Option::is_none")]
    applicant: Option<Applicant>,
}

impl Credit {
    pub fn with_shopping_carts<ShoppingCarts>(mut self, shopping_carts: ShoppingCarts) -> Self
    where
        ShoppingCarts: Iterator<Item = ShoppingCart>,
    {
        self.shopping_carts = Some(shopping_carts.collect());
        self
    }
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct ApplicantAdditionalInfo {
    /// Information whether there were previous, successfully completed orders
    /// for applicant.
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    has_successfully_finished_order_in_shop: Option<String>,
}
