# Unofficial Payu client

> Since 0.1.9
>
> Stabilizing, breaking changes may occur but no so often

<img src="https://poland.payu.com/wp-content/themes/global-website/assets/src/images/payu-logo.svg">

This client support integration with the REST API 2.1 protocol.
It presents various methods of implementing online payments via different PayU services and is dedicated primarily to developers wanting to implement the PayU payment services.

## Install

```bash
cargo add pay_u
```

## Usage

```rust
async fn usage() {
    use pay_u::*;
    
    let client_id = ClientId::new(std::env::var("PAYU_CLIENT_ID").unwrap());
    let client_secret = ClientSecret::new(std::env::var("PAYU_CLIENT_SECRET").unwrap());
    let merchant_id = std::env::var("PAYU_CLIENT_MERCHANT_ID").unwrap().parse::<i32>().map(MerchantPosId::from).unwrap();
    let mut client = Client::new(client_id, client_secret, merchant_id);
    client.authorize().await.expect("Invalid credentials");

    let _res = client.create_order(
        req::OrderCreate::build(
            Buyer::new("john.doe@example.com", "654111654", "John", "Doe", "pl"),
            "127.0.0.1",
            "PLN",
            "Some description"
        )
            .expect("All required fields must be valid")
            // Endpoint which will be requested by PayU with payment status update
            .with_notify_url("https://your.eshop.com/notify")
            // payment description (MANDATORY)
            .with_description("RTV market")
            // add list of products
            .with_products(
                [
                    Product::new("Wireless Mouse for Laptop", 15000, 1),
                    Product::new("HDMI cable", 6000, 1),
                ]
                    .into_iter(),
            )
            // add additional product
            .with_product(Product::new("HDMI cable", 6000, 1)),
    )
        .await;

    // partial refund
    let _res = client
        .refund(
            OrderId::new("H9LL64F37H160126GUEST000P01"),
            RefundRequest::new("Refund", Some(1000)),
        )
        .await;
    // Full refund
    let _res = client
        .refund(
            OrderId::new("H9LL64F37H160126GUEST000P01"),
            RefundRequest::new("Refund", None),
        )
        .await;
    
    // Order details
    let _res = client.order_details(OrderId::new("H9LL64F37H160126GUEST000P01")).await;
    
    // Transactions
    let _res = client.order_transactions(OrderId::new("H9LL64F37H160126GUEST000P01")).await;
}
```

### Actix integration

```rust
use actix_web::{*, web::*};

#[post("/checkout")]
async fn checkout(session: Data<Session>, db: Data<Database>, payu: Data<Arc<Mutex<pay_u::Client>>>) -> HttpResponse {
    let user_id = session.user_required()?;
    let payu = payu.into_inner();
    let shopping_cart = db.send(LoadShoppingCart { user_id }).await??;
    let shopping_cart_id = shopping_cart.id;
    let create_order_req: pay_u::req::OrderCreate = shopping_cart.into();
    let pay_u::res::CreateOrder { redirect_uri, order_id, .. } = payu.create_order(create_order_req).await?;
    db.send(database::CreateOrder { shopping_cart_id, order_id }).await??;
    HttpResponse::SeeOther().append_header((actix_web::http::header::LOCATION, redirect_uri)).body("")
}

#[post("/pay_u/{own_order_id}/notify")]
async fn handle_notification(
    path: Path<i32>,
    Json(notify): Json<pay_u::notify::StatusUpdate>,
    payment: Data<Addr<PaymentManager>>
) -> HttpResponse {
    let status = notify.status();
    // Create additional field which will always be unique like UUID
    // Do not use record primary key!
    let ext_order_id = String::from(notify.ext_order_id());
    
    let order_id = path.into_inner();
    payment.do_send(payment_manager::Update {
        status,
        ext_order_id
    });
    HttpResponse::Ok().body("")
}
```

### Releases

0.1.9 - Stabilizing, breaking changes may occur but no so often
0.1.8 - Additional documentation, move requests to module
0.1.7 - Added credit and more create order request options like additional description, visible description.

## Bugs

Please report bugs here: https://todo.sr.ht/~tsumanu/payu-rs

