mod deserialize;
mod serialize;

use std::sync::Arc;

use reqwest::redirect;
use serde::{Deserialize, Serialize};

use crate::res::OrdersInfo;

macro_rules! get_client {
    ($self:expr) => {{
        #[cfg(feature = "single-client")]
        {
            $self.client.clone()
        }
        #[cfg(not(feature = "single-client"))]
        {
            Client::build_client()
        }
    }};
}

pub static SUCCESS: &str = "SUCCESS";

#[derive(Debug, thiserror::Error)]
pub enum Error {
    #[error("Client is not authorized. No bearer token available")]
    NoToken,
    #[error("Invalid customer ip. IP 0.0.0.0 is not acceptable")]
    CustomerIp,
    #[error("{0}")]
    Io(#[from] std::io::Error),
    #[error("Total value is not sum of products price")]
    IncorrectTotal,
    #[error("{0}")]
    Reqwest(#[from] reqwest::Error),
    #[error("Buyer is required to place an order")]
    NoBuyer,
    #[error("Description is required to place an order")]
    NoDescription,
    #[error("Client is not authorized")]
    Unauthorized,
    #[error("Refund returned invalid response")]
    Refund,
    #[error("Create order returned invalid response")]
    CreateOrder,
    #[error("Failed to fetch order transactions")]
    OrderTransactions,
    #[error("Failed to fetch order details")]
    OrderDetails,
    #[error("Failed to fetch order refunds")]
    OrderRefunds,
    #[error("PayU rejected to create order with status {status_code:?}")]
    CreateFailed {
        status_code: String,
        status_desc: Option<String>,
        code: Option<String>,
        severity: Option<String>,
        code_literal: Option<CodeLiteral>,
    },
    #[error("PayU rejected to perform refund with status {status_code:?}")]
    RefundFailed {
        status_code: String,
        status_desc: Option<String>,
        code: Option<String>,
        severity: Option<String>,
        code_literal: Option<CodeLiteral>,
    },
    #[error("PayU rejected order details request with status {status_code:?}")]
    OrderDetailsFailed {
        status_code: String,
        status_desc: Option<String>,
        code: Option<String>,
        severity: Option<String>,
        code_literal: Option<CodeLiteral>,
    },
    #[error("PayU rejected order transactions details request with status {status_code:?}")]
    OrderTransactionsFailed {
        status_code: String,
        status_desc: Option<String>,
        code: Option<String>,
        severity: Option<String>,
        code_literal: Option<CodeLiteral>,
    },
    #[error("PayU returned order details but without any order")]
    NoOrderInDetails,
}

pub type Result<T> = std::result::Result<T, Error>;

/// PayU internal order id
#[derive(
    Debug,
    Clone,
    serde::Deserialize,
    serde::Serialize,
    derive_more::Display,
    derive_more::From,
    derive_more::Deref,
)]
#[serde(transparent)]
pub struct OrderId(pub String);

impl OrderId {
    pub fn new<S: Into<String>>(id: S) -> Self {
        Self(id.into())
    }
}

/// PayU internal order id
#[derive(
    Debug,
    serde::Deserialize,
    serde::Serialize,
    Copy,
    Clone,
    derive_more::Display,
    derive_more::From,
    derive_more::Deref,
    derive_more::Constructor,
)]
#[serde(transparent)]
pub struct MerchantPosId(pub i32);

#[derive(
    Debug,
    Clone,
    serde::Deserialize,
    serde::Serialize,
    derive_more::Display,
    derive_more::From,
    derive_more::Deref,
)]
#[serde(transparent)]
pub struct ClientId(pub String);

impl ClientId {
    pub fn new<S: Into<String>>(id: S) -> Self {
        Self(id.into())
    }
}

#[derive(
    Debug,
    Clone,
    serde::Deserialize,
    serde::Serialize,
    derive_more::Display,
    derive_more::From,
    derive_more::Deref,
)]
#[serde(transparent)]
pub struct ClientSecret(pub String);

impl ClientSecret {
    pub fn new<S: Into<String>>(id: S) -> Self {
        Self(id.into())
    }
}

/// PayU payment status.
///
/// Each payment is initially Pending and can change according to following
/// graph:
///
/// <img src="https://developers.payu.com/images/order_statusesV2-en.png">
#[derive(Serialize, Deserialize, Copy, Clone, Debug)]
#[serde(rename_all = "SCREAMING_SNAKE_CASE")]
pub enum PaymentStatus {
    /// Payment is currently being processed.
    Pending,
    /// PayU is currently waiting for the merchant system to receive (capture)
    /// the payment. This status is set if auto-receive is disabled on the
    /// merchant system.
    WaitingForConfirmation,
    /// Payment has been accepted. PayU will pay out the funds shortly.
    Completed,
    /// Payment has been cancelled and the buyer has not been charged (no money
    /// was taken from buyer's account).
    Canceled,
}

#[derive(Serialize, Deserialize, Copy, Clone, Debug)]
#[serde(rename_all = "SCREAMING_SNAKE_CASE")]
pub enum RefundStatus {
    /// refund was completed successfully
    Finalized,
    /// refund was cancelled
    Canceled,
    /// refund in progress
    Pending,
    /// PayU is currently waiting for the merchant system to receive (capture)
    /// the payment. This status is set if auto-receive is disabled on the
    /// merchant system.
    WaitingForConfirmation,
    /// Payment has been accepted. PayU will pay out the funds shortly.
    Completed,
}

#[derive(Serialize, Deserialize, Debug, Default)]
#[serde(rename_all = "camelCase")]
pub struct Delivery {
    #[serde(skip_serializing_if = "Option::is_none")]
    /// Street name
    street: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    /// Postal box number
    postal_box: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    /// Postal code
    postal_code: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    /// City
    city: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    /// Province
    state: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    /// Two-letter country code compliant with ISO-3166.
    country_code: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    /// Address description
    name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    /// Recipient’s name
    recipient_name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    /// Recipient’s e-mail address
    recipient_email: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    /// Recipient’s phone number
    recipient_phone: Option<String>,
}

#[derive(Serialize, Deserialize, Debug, Default)]
#[serde(rename_all = "camelCase")]
pub struct BuyerShippingAddress {
    #[serde(skip_serializing_if = "Option::is_none")]
    /// stores the shipping address
    delivery: Option<Delivery>,
}

impl BuyerShippingAddress {
    pub fn new_with_delivery(delivery: Delivery) -> Self {
        Self {
            delivery: Some(delivery),
        }
    }
}

#[derive(Serialize, Deserialize, Debug, Default)]
#[serde(rename_all = "camelCase")]
pub struct Buyer {
    /// Required customer e-mail
    #[serde(skip_serializing_if = "Option::is_none")]
    email: Option<String>,
    /// Required customer phone number
    #[serde(skip_serializing_if = "Option::is_none")]
    phone: Option<String>,
    /// Required customer first name
    #[serde(skip_serializing_if = "Option::is_none")]
    first_name: Option<String>,
    /// Required customer last name
    #[serde(skip_serializing_if = "Option::is_none")]
    last_name: Option<String>,
    /// Required customer language
    #[serde(skip_serializing_if = "Option::is_none")]
    language: Option<String>,
    #[serde(skip_serializing_if = "Option::is_none")]
    delivery: Option<BuyerShippingAddress>,
}

impl Buyer {
    pub fn new<Email, Phone, FirstName, LastName, Language>(
        email: Email,
        phone: Phone,
        first_name: FirstName,
        last_name: LastName,
        lang: Language,
    ) -> Self
    where
        Email: Into<String>,
        Phone: Into<String>,
        FirstName: Into<String>,
        LastName: Into<String>,
        Language: Into<String>,
    {
        Self {
            email: Some(email.into()),
            phone: Some(phone.into()),
            first_name: Some(first_name.into()),
            last_name: Some(last_name.into()),
            language: Some(lang.into()),
            delivery: None,
        }
    }

    pub fn email(&self) -> &str {
        self.email.as_deref().unwrap_or_default()
    }
    pub fn with_email<S>(mut self, email: S) -> Self
    where
        S: Into<String>,
    {
        self.email = Some(email.into());
        self
    }
    pub fn phone(&self) -> &str {
        self.phone.as_deref().unwrap_or_default()
    }
    pub fn with_phone<S>(mut self, phone: S) -> Self
    where
        S: Into<String>,
    {
        self.phone = Some(phone.into());
        self
    }
    pub fn first_name(&self) -> &str {
        self.first_name.as_deref().unwrap_or_default()
    }
    pub fn with_first_name<S>(mut self, first_name: S) -> Self
    where
        S: Into<String>,
    {
        self.first_name = Some(first_name.into());
        self
    }
    pub fn last_name(&self) -> &str {
        self.last_name.as_deref().unwrap_or_default()
    }
    pub fn with_last_name<S>(mut self, last_name: S) -> Self
    where
        S: Into<String>,
    {
        self.last_name = Some(last_name.into());
        self
    }
    pub fn language(&self) -> &str {
        self.language.as_deref().unwrap_or_default()
    }
    pub fn with_language<S>(mut self, language: S) -> Self
    where
        S: Into<String>,
    {
        self.language = Some(language.into());
        self
    }

    pub fn with_delivery(mut self, delivery: Delivery) -> Self {
        self.delivery = Some(BuyerShippingAddress::new_with_delivery(delivery));
        self
    }
}

pub type Price = i32;
pub type Quantity = u32;

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Product {
    pub name: String,
    #[serde(
        serialize_with = "serialize::serialize_i32",
        deserialize_with = "deserialize::deserialize_i32"
    )]
    pub unit_price: Price,
    #[serde(
        serialize_with = "serialize::serialize_u32",
        deserialize_with = "deserialize::deserialize_u32"
    )]
    pub quantity: Quantity,
    /// Product type, which can be virtual or material; (possible values true or
    /// false).
    #[serde(rename = "virtual", skip_serializing_if = "Option::is_none")]
    pub virtual_product: Option<bool>,
    /// Marketplace date from which the product (or offer) is available, ISO
    /// format applies, e.g. "2019-03-27T10:57:59.000+01:00".
    #[serde(skip_serializing_if = "Option::is_none")]
    pub listing_date: Option<chrono::NaiveDateTime>,
}

impl Product {
    pub fn new<Name: Into<String>>(name: Name, unit_price: Price, quantity: Quantity) -> Self {
        Self {
            name: name.into(),
            unit_price,
            quantity,
            virtual_product: None,
            listing_date: None,
        }
    }

    /// Product type, which can be virtual or material; (possible values true or
    /// false).
    pub fn into_virtual(mut self) -> Self {
        self.virtual_product = Some(true);
        self
    }

    /// Product type, which can be virtual or material; (possible values true or
    /// false).
    pub fn non_virtual(mut self) -> Self {
        self.virtual_product = Some(false);
        self
    }

    /// Marketplace date from which the product (or offer) is available, ISO
    /// format applies, e.g. "2019-03-27T10:57:59.000+01:00".
    pub fn with_listing_date(mut self, listing_date: chrono::NaiveDateTime) -> Self {
        self.listing_date = Some(listing_date);
        self
    }

    fn erase_listing_date(mut self) -> Self {
        self.listing_date = None;
        self
    }
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct ShoppingCart {
    /// Section containing data of shipping method.
    #[serde(skip_serializing_if = "Option::is_none")]
    shopping_method: Option<ShoppingMethod>,
    /// Section containing data about ordered products.
    /// > Note: product objects in the <shoppingCart.products> section do not
    /// > have a listingDate field
    #[serde(skip_serializing_if = "Option::is_none")]
    products: Option<Vec<Product>>,
    /// Submerchant identifier. This field should be consistent with field
    /// extCustomerId in shoppingCarts section when order is placed in
    /// marketplace.
    ext_customer_id: String,
}

impl ShoppingCart {
    pub fn new<ExtCustomerId, Products>(ext_customer_id: ExtCustomerId) -> Self
    where
        ExtCustomerId: Into<String>,
    {
        Self {
            shopping_method: None,
            ext_customer_id: ext_customer_id.into(),
            products: None,
        }
    }

    pub fn new_with_products<ExtCustomerId, Products>(
        ext_customer_id: ExtCustomerId,
        products: Products,
    ) -> Self
    where
        ExtCustomerId: Into<String>,
        Products: Iterator<Item = Product>,
    {
        Self {
            shopping_method: None,
            ext_customer_id: ext_customer_id.into(),
            products: Some(products.map(Product::erase_listing_date).collect()),
        }
    }

    pub fn with_products<Products>(mut self, products: Products) -> Self
    where
        Products: Iterator<Item = Product>,
    {
        self.products = Some(products.map(Product::erase_listing_date).collect());
        self
    }

    /// Section containing data of shipping method.
    pub fn shopping_method(&self) -> &Option<ShoppingMethod> {
        &self.shopping_method
    }

    /// Section containing data about ordered products.
    /// > Note: product objects in the <shoppingCart.products> section do not
    /// > have a listingDate field
    pub fn products(&self) -> &Option<Vec<Product>> {
        &self.products
    }

    /// Submerchant identifier. This field should be consistent with field
    /// extCustomerId in shoppingCarts section when order is placed in
    /// marketplace.
    pub fn ext_customer_id(&self) -> &String {
        &self.ext_customer_id
    }
}

/// Type of shipment
#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "SCREAMING_SNAKE_CASE")]
pub enum ShoppingMethodType {
    Courier,
    CollectionPointPickup,
    ParcelLocker,
    StorePickup,
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Address {
    /// The full name of the pickup point, including its unique identifier, e.g.
    /// „Parcel locker POZ29A”.
    #[serde(skip_serializing_if = "Option::is_none")]
    pub point_id: Option<String>,
    /// Street name, possibly including house and flat number.
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    pub street: Option<String>,
    /// Street number
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    pub street_no: Option<String>,
    /// Flat number
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    pub flat_no: Option<String>,
    /// Postal code
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    pub postal_code: Option<String>,
    /// City
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    pub city: Option<String>,
    /// Two-letter country code compliant with ISO-3166
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    pub country_code: Option<String>,
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct ShoppingMethod {
    /// Shipping type
    /// Recommended
    #[serde(rename = "type", skip_serializing_if = "Option::is_none")]
    pub shopping_type: Option<ShoppingMethodType>,
    /// Shipping cost
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    pub price: Option<String>,
    /// Section containing data about shipping address.
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    pub address: Option<Address>,
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct ApplicantAdditionalInfo {
    /// Information whether there were previous, successfully completed orders
    /// for applicant.
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    has_successfully_finished_order_in_shop: Option<String>,
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Applicant {
    /// Applicant’s email address
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    email: Option<String>,
    /// Applicant’s phone number
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    phone: Option<String>,
    /// Applicant’s first name
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    first_name: Option<String>,
    /// Applicant’s last name
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    last_name: Option<String>,
    /// Language code, ISO-639-1 compliant. Denotes the language version of
    /// PayU hosted payment page and of e-mail messages sent from PayU to the
    /// payer (supported values are here).
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    language: Option<String>,
    /// National Identification Number
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    nin: Option<String>,
    /// Section containing data about applicant’s address.
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    address: Option<Address>,
    /// Additional information about person applying for credit.
    /// Recommended
    #[serde(skip_serializing_if = "Option::is_none")]
    additional_info: Option<ApplicantAdditionalInfo>,
}

#[derive(Serialize, Deserialize, Debug, Default)]
#[serde(rename_all = "camelCase")]
pub struct Credit {
    /// Section containing data of the ordered products
    #[serde(skip_serializing_if = "Option::is_none")]
    shopping_carts: Option<Vec<ShoppingCart>>,
    /// Section containing data of person applying for a credit
    #[serde(skip_serializing_if = "Option::is_none")]
    applicant: Option<Applicant>,
}

impl Credit {
    pub fn with_shopping_carts<ShoppingCarts>(mut self, shopping_carts: ShoppingCarts) -> Self
    where
        ShoppingCarts: Iterator<Item = ShoppingCart>,
    {
        self.shopping_carts = Some(shopping_carts.collect());
        self
    }
}

/// MultiUseCartToken
pub mod muct {
    use serde::{Deserialize, Serialize};

    #[derive(Serialize, Deserialize, Debug)]
    #[serde(rename_all = "SCREAMING_SNAKE_CASE")]
    pub enum CardOnFile {
        /// Payment initialized by the card owner who agreed to save card for
        /// future use. You can expect full authentication (3D Secure
        /// and/or CVV). If you want to use multi-use token (TOKC_)
        /// later, you have to be confident, that first payment was
        /// successful. Default value for single-use token (TOK_).
        ///
        /// In case of plain card data payments you should retrieve transaction
        /// data to obtain first TransactionId. It should be passed in
        /// payMethods.payMethod.card section for transactions marked as
        /// STANDARD, STANDARD_CARDHOLDER and STANDARD_MERCHANT;
        /// STANDARD_CARDHOLDER - payment with already saved card,
        /// initialized by the card owner. This transaction has
        /// multi-use token (TOKC_). Depending of payment parameters
        /// (e.g. high transaction amount) strong authentication can be
        /// expected (3D Secure and/or CVV). Default value for multi-use token
        /// (TOKC_);
        First,
        /// Payment with already saved card, initialized by the shop without the
        /// card owner participation. This transaction has multi-use token
        /// (TOKC_). By the definition, this payment type does not
        /// require strong authentication. You cannot use it if FIRST
        /// card-on-file payment failed.
        StandardMerchant,
    }

    #[derive(Serialize, Deserialize, Debug)]
    #[serde(rename_all = "SCREAMING_SNAKE_CASE")]
    pub enum Recurring {
        /// Payment initialized by the card owner who agreed to save card for
        /// future use in recurring plan. You can expect full authentication (3D
        /// Secure and/or CVV). If you want to use multi-use token (TOKC_)
        /// later, you have to be confident, that   first recurring
        /// payment was successful.
        First,
        /// Subsequent recurring payment (user is not present). This transaction
        /// has multi use token (TOKC_). You cannot use it if FIRST recurring
        /// payment failed.
        Standard,
    }

    #[derive(Serialize, Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct MultiUseCartToken {
        /// Information about party initializing order:
        ///
        /// * `FIRST` - payment initialized by the card owner who agreed to save
        ///   card for future use. You can expect full authentication (3D Secure
        ///   and/or CVV). If you want to use multi-use token (TOKC_) later, you
        ///   have to be confident, that first payment was successful. Default
        ///   value for single-use token (TOK_).
        ///
        ///   In case of plain card data payments you should retrieve
        /// transaction   data to obtain first TransactionId. It should
        /// be passed in   payMethods.payMethod.card section for
        /// transactions marked as STANDARD,   STANDARD_CARDHOLDER and
        /// STANDARD_MERCHANT; STANDARD_CARDHOLDER -   payment with
        /// already saved card, initialized by the card owner. This
        ///   transaction has multi-use token (TOKC_). Depending of payment
        ///   parameters (e.g. high transaction amount) strong authentication
        /// can be   expected (3D Secure and/or CVV). Default value for
        /// multi-use token   (TOKC_);
        /// * `STANDARD_MERCHANT` - payment with already saved card, initialized
        ///   by the shop without the card owner participation. This transaction
        ///   has multi-use token (TOKC_). By the definition, this payment type
        ///   does not require strong authentication. You cannot use it if FIRST
        ///   card-on-file payment failed.
        ///
        /// `cardOnFile` parameter cannot be used with recurring parameter.
        pub card_on_file: CardOnFile,
        /// Marks the order as recurring payment.
        ///
        /// * `FIRST` - payment initialized by the card owner who agreed to save
        ///   card for future use in recurring plan. You can expect full
        ///   authentication (3D Secure and/or CVV). If you want to use
        ///   multi-use token (TOKC_) later, you have to be confident, that
        ///   first recurring payment was successful.
        /// * `STANDARD` - subsequent recurring payment (user is not present).
        ///   This transaction has multi use token (TOKC_). You cannot use it if
        ///   FIRST recurring payment failed.
        ///
        /// `recurring` parameter cannot be used with cardOnFile parameter.
        pub recurring: Recurring,
    }
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct OrderCreateRequest {
    /// ID of an order used in merchant system. Order identifier assigned by the
    /// merchant. It enables merchants to find a specific order in their system.
    /// This value must be unique within a single POS.
    ext_order_id: Option<String>,
    /// URL to which web hook will be send. It's important to return 200 to all
    /// notifications.
    ///
    /// All notifications are send as POST with JSON payload
    ///
    /// Notifications are sent immediately after a payment status changes. If
    /// the notification is not received by the Shop application, it will be
    /// sent again in accordance with the table below:
    ///
    /// | Attempt | Time |
    /// |---------|------|
    /// | 1 | immediately |
    /// | 2 | 1 minute |
    /// | 3 | 2 minutes |
    /// | 4 | 5 minutes |
    /// | 5 | 10 minutes |
    /// | 6 | 30 minutes |
    /// | 7 | 1 hour |
    /// | 8 | 2 hours |
    /// | 9 | 3 hours |
    /// | 10| 6 hours |
    /// | 11| 9 hours |
    /// | 12| 12 hours |
    /// | 13| 15 hours |
    /// | 14| 18 hours |
    /// | 15| 21 hours |
    /// | 16| 24 hours |
    /// | 17| 36 hours |
    /// | 18| 48 hours |
    /// | 19| 60 hours |
    /// | 20| 72 hours |
    #[serde(skip_serializing_if = "Option::is_none")]
    notify_url: Option<String>,
    /// Address for redirecting the customer after payment is commenced. If the
    /// payment has not been authorized, error=501 parameter will be added.
    /// Please note that no decision regarding payment status should be made
    /// depending on the presence or lack of this parameter (to get payment
    /// status, wait for notification or retrieve order details).
    ///
    /// IMPORTANT: the address must be compliant with the structure below:
    /// <img src="https://developers.payu.com/images/continueUrlStructure_en.png" />
    ///
    /// Please keep in mind:
    /// * accepted schemas are http and https,
    /// * such elements as port, path, query and fragment are optional,
    /// * query values must be encoded.
    #[serde(skip_serializing_if = "Option::is_none")]
    continue_url: Option<String>,
    /// Payer’s IP address, e.g. 123.123.123.123. Note: 0.0.0.0 is not accepted.
    customer_ip: String,
    /// Secret pos ip. This is connected to PayU account
    #[serde(
        serialize_with = "serialize::serialize_newtype",
        deserialize_with = "deserialize::deserialize_i32_newtype"
    )]
    merchant_pos_id: MerchantPosId,
    /// Transaction description
    description: String,
    /// 3 characters currency identifier, ex. PLN
    currency_code: String,
    /// Total price of the order in pennies (e.g. 1000 is 10.00 EUR). Applies
    /// also to currencies without subunits (e.g. 1000 is 10 HUF).
    #[serde(
        serialize_with = "serialize::serialize_i32",
        deserialize_with = "deserialize::deserialize_i32"
    )]
    total_amount: Price,
    /// @see [Buyer]
    buyer: Option<Buyer>,
    /// List of products
    products: Vec<Product>,
    #[serde(skip_serializing)]
    order_create_date: Option<String>,
    /// Duration for the validity of an order (in seconds), during which time
    /// payment must be made. Default value 86400.
    #[serde(skip_serializing_if = "Option::is_none")]
    validity_time: Option<u16>,
    ///  Additional description of the order.
    #[serde(skip_serializing_if = "Option::is_none")]
    additional_description: Option<String>,
    /// Text visible on the PayU payment page (max. 80 chars).
    #[serde(skip_serializing_if = "Option::is_none")]
    visible_description: Option<String>,
    /// Payment recipient name followed by payment description (order ID, ticket
    /// number etc) visible on card statement (max. 22 chars). The name should
    /// be easy to recognize by the cardholder (e.g "shop.com 124343"). If field
    /// is not provided, static name configured by PayU will be used.
    #[serde(skip_serializing_if = "Option::is_none")]
    statement_description: Option<String>,
    #[serde(flatten, skip_serializing_if = "Option::is_none")]
    muct: Option<muct::MultiUseCartToken>,
    #[serde(skip_serializing_if = "Option::is_none")]
    credit: Option<Credit>,
}

impl OrderCreateRequest {
    pub fn build<CustomerIp, Currency, Description>(
        buyer: Buyer,
        customer_ip: CustomerIp,
        currency: Currency,
        description: Description,
    ) -> Result<Self>
    where
        CustomerIp: Into<String>,
        Currency: Into<String>,
        Description: Into<String>,
    {
        let customer_ip = customer_ip.into();
        if &customer_ip == "0.0.0.0" {
            return Err(Error::CustomerIp);
        }
        Ok(Self {
            ext_order_id: None,
            notify_url: None,
            continue_url: None,
            customer_ip,
            merchant_pos_id: 0.into(),
            description: description.into(),
            currency_code: currency.into(),
            total_amount: 0,
            buyer: Some(buyer),
            products: Vec::new(),
            order_create_date: None,
            validity_time: None,
            additional_description: None,
            visible_description: None,
            statement_description: None,
            muct: None,
            credit: None,
        })
    }

    /// ID of an order used in merchant system. Order identifier assigned by the
    /// merchant. It enables merchants to find a specific order in their system.
    /// This value must be unique within a single POS.
    pub fn with_ext_order_id<S: Into<String>>(mut self, ext_order_id: S) -> Self {
        self.ext_order_id = Some(ext_order_id.into());
        self
    }

    /// Duration for the validity of an order (in seconds), during which time
    /// payment must be made. Default value 86400.
    pub fn with_validity_time(mut self, validity_time: u16) -> Self {
        self.validity_time = Some(validity_time);
        self
    }

    pub fn with_multi_use_token(
        mut self,
        recurring: muct::Recurring,
        card_on_file: muct::CardOnFile,
    ) -> Self {
        self.muct = Some(muct::MultiUseCartToken {
            recurring,
            card_on_file,
        });
        self
    }

    pub fn with_products<Products>(mut self, products: Products) -> Self
    where
        Products: Iterator<Item = Product>,
    {
        self.products.extend(products);
        self.total_amount = self
            .products
            .iter()
            .fold(0, |agg, p| agg + (p.quantity as i32 * p.unit_price as i32));
        self
    }

    pub fn with_product(mut self, product: Product) -> Self {
        self.products.push(product);
        self.total_amount = self
            .products
            .iter()
            .fold(0, |agg, p| agg + (p.quantity as i32 * p.unit_price as i32));
        self
    }

    /// Description of the an order.
    ///
    /// > This method will override initial description!
    pub fn with_description<Description>(mut self, desc: Description) -> Self
    where
        Description: Into<String>,
    {
        self.description = String::from(desc.into().trim());
        self
    }

    /// Additional description of the order.
    pub fn with_additional_description<S: Into<String>>(
        mut self,
        additional_description: S,
    ) -> Self {
        self.additional_description = Some(additional_description.into());
        self
    }

    /// Text visible on the PayU payment page (max. 80 chars).
    pub fn with_visible_description(mut self, visible_description: &str) -> Self {
        let visible_description = if visible_description.len() > 60 {
            &visible_description[..60]
        } else {
            visible_description
        };
        self.visible_description = Some(String::from(visible_description));
        self
    }

    /// Payment recipient name followed by payment description (order ID, ticket
    /// number etc) visible on card statement (max. 22 chars). The name should
    /// be easy to recognize by the cardholder (e.g "shop.com 124343"). If field
    /// is not provided, static name configured by PayU will be used.
    pub fn with_statement_description<Description>(mut self, desc: Description) -> Self
    where
        Description: Into<String>,
    {
        self.statement_description = Some(String::from(desc.into().trim()));
        self
    }

    /// Add url to which PayU will be able to send http request with payment
    /// status updates
    ///
    /// All requests from PayU should receive 200 response!
    ///
    /// See more [Order::notify_url]
    pub fn with_notify_url<NotifyUrl>(mut self, notify_url: NotifyUrl) -> Self
    where
        NotifyUrl: Into<String>,
    {
        self.notify_url = Some(notify_url.into());
        self
    }

    /// Address for redirecting the customer after payment is commenced. If the
    /// payment has not been authorized, error=501 parameter will be added.
    /// Please note that no decision regarding payment status should be made
    /// depending on the presence or lack of this parameter (to get payment
    /// status, wait for notification or retrieve order details).
    pub fn with_continue_url<ContinueUrl>(mut self, continue_url: ContinueUrl) -> Self
    where
        ContinueUrl: Into<String>,
    {
        self.continue_url = Some(continue_url.into());
        self
    }

    /// Section containing credit data. This information is not required, but it
    /// is strongly recommended to include it. Otherwise the buyer will be
    /// prompted to provide missing data on provider page when payment by
    /// Installments or Pay later.
    pub fn with_credit(mut self, credit: Credit) -> Self {
        self.credit = Some(credit);
        self
    }

    /// URL to which web hook will be send. It's important to return 200 to all
    /// notifications.
    ///
    /// All notifications are send as POST with JSON payload
    ///
    /// Notifications are sent immediately after a payment status changes. If
    /// the notification is not received by the Shop application, it will be
    /// sent again in accordance with the table below:
    ///
    /// | Attempt | Time |
    /// |---------|------|
    /// | 1 | immediately |
    /// | 2 | 1 minute |
    /// | 3 | 2 minutes |
    /// | 4 | 5 minutes |
    /// | 5 | 10 minutes |
    /// | 6 | 30 minutes |
    /// | 7 | 1 hour |
    /// | 8 | 2 hours |
    /// | 9 | 3 hours |
    /// | 10| 6 hours |
    /// | 11| 9 hours |
    /// | 12| 12 hours |
    /// | 13| 15 hours |
    /// | 14| 18 hours |
    /// | 15| 21 hours |
    /// | 16| 24 hours |
    /// | 17| 36 hours |
    /// | 18| 48 hours |
    /// | 19| 60 hours |
    /// | 20| 72 hours |
    pub fn notify_url(&self) -> &Option<String> {
        &self.notify_url
    }

    /// Customer IP address from http request received from client
    pub fn customer_ip(&self) -> &String {
        &self.customer_ip
    }

    pub fn merchant_pos_id(&self) -> MerchantPosId {
        self.merchant_pos_id
    }

    pub fn description(&self) -> &String {
        &self.description
    }

    pub fn currency_code(&self) -> &String {
        &self.currency_code
    }

    pub fn total_amount(&self) -> &Price {
        &self.total_amount
    }

    pub fn buyer(&self) -> &Option<Buyer> {
        &self.buyer
    }

    pub fn products(&self) -> &[Product] {
        &self.products
    }

    pub fn order_create_date(&self) -> &Option<String> {
        &self.order_create_date
    }

    pub(crate) fn with_merchant_pos_id(mut self, merchant_pos_id: MerchantPosId) -> Self {
        self.merchant_pos_id = merchant_pos_id;
        self
    }
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "SCREAMING_SNAKE_CASE")]
pub enum PaymentType {
    Pbl,
    CardToken,
    Installments,
}

/// Wrapper around pay method. It's used only for deserializing notifications
///
/// # Examples
///
/// ```
/// # use pay_u::PayMethod;
/// let method: PayMethod = serde_json::from_str(r#"
///     {
///         "type": "INSTALLMENTS"
///     }
/// "#).unwrap();
/// ```
#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct PayMethod {
    #[serde(rename = "type")]
    pub payment_type: PaymentType,
}

#[derive(Deserialize, Serialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Status {
    status_code: String,
    status_desc: Option<String>,
    code: Option<String>,
    severity: Option<String>,
    code_literal: Option<CodeLiteral>,
}

#[derive(Serialize, Deserialize, Copy, Clone, Debug)]
#[serde(rename_all = "SCREAMING_SNAKE_CASE")]
pub enum StatusCode {
    ErrorValueMissing,
    OpenpayuBusinessError,
    OpenpayuErrorValueInvalid,
    OpenpayuErrorInternal,
}

#[derive(Serialize, Deserialize, Clone, Debug)]
#[serde(rename_all = "SCREAMING_SNAKE_CASE")]
pub enum CodeLiteral {
    /// Request lacks "refund" object.
    MissingRefundSection,

    /// Transaction has not been finalized
    TransNotEnded,

    /// Lack of funds in account
    NoBalance,

    /// Refund amount exceeds transaction amount
    AmountToBig,

    /// Refund value is too small
    AmountToSmall,

    /// Refunds have been disabled
    RefundDisabled,

    /// Too many refund attempts have been made
    RefundToOften,

    /// Refund was already created
    Paid,

    /// Unknown error
    UnknownError,

    /// extRefundId was re-used and other params do not match the values
    /// sent during the first call.
    RefundIdempotencyMismatch,

    /// Shop billing has not yet been completed
    TransBillingEntriesNotCompleted,

    /// The available time for refund has passed.
    TransTooOld,

    /// Transaction amount that remains after refund creation will be too
    /// small to make another refund.
    RemainingTransAmountTooSmall,

    #[serde(other)]
    /// Implementation changed
    Unknown,
}

impl Status {
    /// Check if http request was successful
    ///
    /// # Examples
    ///
    /// ```
    /// # use pay_u::Status;
    /// let status: Status = serde_json::from_str("{\"statusCode\":\"SUCCESS\"}").unwrap();
    /// assert_eq!(status.is_success(), true);
    /// ```
    pub fn is_success(&self) -> bool {
        self.status_code.as_str() == SUCCESS
    }

    /// Returns http status
    ///
    /// # Examples
    ///
    /// ```
    /// # use pay_u::Status;
    /// let status: Status = serde_json::from_str("{\"statusCode\":\"SUCCESS\"}").unwrap();
    /// assert_eq!(status.status_code(), "SUCCESS");
    /// ```
    pub fn status_code(&self) -> &str {
        &self.status_code
    }

    pub fn status_desc(&self) -> Option<&str> {
        self.status_desc.as_deref()
    }
}

#[derive(Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Prop {
    pub name: String,
    pub value: String,
}

pub mod res {
    use crate::{OrderId, Refund, Status};

    #[derive(serde::Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct CreateOrder {
        /// Http status as a text
        pub status: Status,
        /// Client should be redirected to this URI
        pub redirect_uri: String,
        /// This should be connected to your own order
        pub order_id: OrderId,
        /// This is YOUR_EXT_ORDER_ID
        pub ext_order_id: Option<String>,
    }

    #[derive(serde::Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct RefundDetails {
        pub order_id: Option<String>,
        pub refund: Option<Refund>,
        pub status: Status,
    }

    #[derive(serde::Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct Refunds {
        pub refunds: Vec<Refund>,
    }

    #[derive(serde::Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct TransactionPayMethod {
        pub value: String,
    }

    #[derive(serde::Deserialize, Debug)]
    #[serde(rename_all = "SCREAMING_SNAKE_CASE")]
    pub enum CardProfile {
        Consumer,
        Business,
    }

    #[derive(serde::Deserialize, Debug)]
    #[serde(rename_all = "SCREAMING_SNAKE_CASE")]
    pub enum CardClassification {
        Debit,
        Credit,
    }

    #[derive(serde::Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct TransactionCartData {
        /// // "543402******4014",
        pub card_number_masked: String,
        /// MC (MasterCard/Maestro), VS (Visa)
        /// Example; "MC"
        pub card_scheme: String,
        pub card_profile: CardProfile,
        pub card_classification: CardClassification,
        ///  Example: "000"
        pub card_response_code: String,
        ///  Example: "000 - OK"
        pub card_response_code_desc: String,
        ///  Example: "5"
        pub card_eci_code: String,
        ///  Example: "AY",
        pub card3ds_status: String,
        ///  Example: "PL",
        pub card_bin_country: String,
        ///  Example: "MCC0111LL1121"
        pub first_transaction_id: String,
    }

    /// > Installment proposal on the Sandbox environment is not related to the
    /// > order amount and always returns data for 480 PLN.
    #[derive(serde::Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct TransactionCardInstallmentProposal {
        /// Example: "5aff3ba8-0c37-4da1-ba4a-4ff24bcc2eed"
        pub proposal_id: String,
    }

    #[derive(serde::Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct TransactionCart {
        pub cart_data: TransactionCartData,
        pub card_installment_proposal: TransactionCardInstallmentProposal,
    }

    #[derive(serde::Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct Transaction {
        pub pay_method: TransactionPayMethod,
        pub payment_flow: String,
    }

    #[derive(serde::Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct Transactions {
        pub transactions: Vec<Transaction>,
    }

    #[derive(serde::Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct Order {
        /// Example: "{orderId}",
        pub order_id: super::OrderId,
        /// Example:  "358766",
        pub ext_order_id: Option<String>,
        /// Example: "2014-10-27T14:58:17.443+01:00",
        pub order_create_date: String,
        /// Example: "http://localhost/OrderNotify/",
        pub notify_url: Option<String>,
        /// Example: "127.0.0.1",
        pub customer_ip: String,
        /// Example: "145227",
        pub merchant_pos_id: String,
        /// Example: "New order",
        pub description: String,
        /// Example: "PLN",
        pub currency_code: String,
        /// Example: "3200",
        pub total_amount: String,
        /// Example: "NEW",
        pub status: String,
        /// Example: `[{"name":"Product1","unitPrice":"1000","quantity":"1"}]`
        pub products: Vec<super::Product>,
    }

    #[derive(serde::Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct OrdersInfo {
        pub orders: Vec<Order>,
        pub status: super::Status,
        pub properties: Option<Vec<crate::Prop>>,
    }

    #[derive(serde::Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct OrderInfo {
        pub order: Order,
        pub status: super::Status,
        pub properties: Option<Vec<crate::Prop>>,
    }
}

#[derive(Deserialize, Serialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct RefundRequest {
    description: String,
    #[serde(skip_serializing_if = "Option::is_none")]
    amount: Option<Price>,
}

impl RefundRequest {
    pub fn new<Description>(description: Description, amount: Option<Price>) -> Self
    where
        Description: Into<String>,
    {
        Self {
            description: description.into(),
            amount,
        }
    }

    pub fn description(&self) -> &str {
        &self.description
    }

    pub fn amount(&self) -> Option<Price> {
        self.amount
    }
}

#[derive(Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Refund {
    pub refund_id: String,
    pub ext_refund_id: Option<String>,
    pub amount: String,
    pub currency_code: String,
    pub description: String,
    pub creation_date_time: String,
    pub status: String,
    pub status_date_time: String,
}

pub mod notify {
    use serde::Deserialize;

    use super::deserialize;
    use crate::OrderId;

    /// Payment notification object received on [super::Order].[notify_url]
    #[derive(Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct StatusUpdate {
        pub order: Order,
        pub local_receipt_date_time: Option<String>,
        pub properties: Option<Vec<super::Prop>>,
        pub status: Option<super::Status>,
    }

    impl StatusUpdate {
        pub fn status(&self) -> super::PaymentStatus {
            self.order.status
        }
    }

    /// Refund notification object
    #[derive(Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct RefundUpdate {
        pub ext_order_id: String,
        pub order_id: OrderId,
        pub refund: Refund,
    }

    #[derive(Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct Refund {
        pub refund_id: String,
        pub amount: String,
        pub currency_code: String,
        pub status: super::RefundStatus,
        pub status_date_time: String,
        pub reason: String,
        pub reason_description: String,
        pub refund_date: String,
    }

    #[derive(Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct Order {
        pub notify_url: Option<String>,
        /// Customer client IP address
        pub customer_ip: String,
        /// Secret pos ip. This is connected to PayU account
        #[serde(deserialize_with = "deserialize::deserialize_i32_newtype")]
        pub merchant_pos_id: super::MerchantPosId,
        /// Transaction description
        pub description: String,
        /// 3 characters currency identifier, ex. PLN
        pub currency_code: String,
        /// Total price of the order in pennies (e.g. 1000 is 10.00 EUR).
        /// Applies also to currencies without subunits (e.g. 1000 is 10
        /// HUF).
        #[serde(deserialize_with = "deserialize::deserialize_i32")]
        pub total_amount: super::Price,
        /// @see [Buyer]
        pub buyer: Option<super::Buyer>,
        /// List of products
        pub products: Vec<super::Product>,
        #[serde(skip_serializing)]
        pub order_create_date: Option<String>,
        pub pay_method: Option<super::PayMethod>,
        pub status: super::PaymentStatus,
    }
}

pub struct Client {
    sandbox: bool,
    merchant_pos_id: MerchantPosId,
    client_id: ClientId,
    client_secret: ClientSecret,
    bearer: Option<String>,
    bearer_expires_at: chrono::DateTime<chrono::Utc>,
    #[cfg(feature = "single-client")]
    client: Arc<reqwest::Client>,
}

impl Client {
    /// Create new PayU client
    pub fn new(
        client_id: ClientId,
        client_secret: ClientSecret,
        merchant_pos_id: MerchantPosId,
    ) -> Self {
        #[cfg(feature = "single-client")]
        {
            Self {
                bearer: None,
                sandbox: false,
                merchant_pos_id,
                client_id,
                client_secret,
                bearer_expires_at: chrono::Utc::now(),
                client: Arc::new(Self::build_client()),
            }
        }
        #[cfg(not(feature = "single-client"))]
        {
            Self {
                bearer: None,
                sandbox: false,
                merchant_pos_id,
                client_id,
                client_secret,
                bearer_expires_at: chrono::Utc::now(),
            }
        }
    }

    /// All operation will be performed in sandbox PayU environment
    pub fn sandbox(mut self) -> Self {
        self.sandbox = true;
        self
    }

    /// Set your own bearer key
    pub fn with_bearer<Bearer: Into<String>>(mut self, bearer: Bearer, expires_in: i64) -> Self {
        self.bearer = Some(bearer.into());
        self.bearer_expires_at = chrono::Utc::now() + chrono::Duration::seconds(expires_in);
        self
    }

    /// Create new order in PayU
    ///
    /// ### IMPORTANT:
    /// Do not follow redirect for any reason. Location points to
    /// payment page which is useless in this context
    ///
    /// ### IMPORTANT:
    /// Do not use rustls. It'll freeze application!
    ///
    /// # Examples
    ///
    /// ```rust
    /// # use pay_u::*;
    /// async fn pay() {
    ///     let mut client = Client::new(ClientId::new("145227"), ClientSecret::new("12f071174cb7eb79d4aac5bc2f07563f"), MerchantPosId::new(300746))
    ///         .sandbox()
    ///         .with_bearer("d9a4536e-62ba-4f60-8017-6053211d3f47", 2000);
    ///     let res = client
    ///         .create_order(
    ///                 OrderCreateRequest::new(
    ///                     Buyer::new("john.doe@example.com", "654111654", "John", "Doe", "pl"),
    ///                     "127.0.0.1",
    ///                     "PLN",
    ///                 )
    ///                 .with_notify_url("https://your.eshop.com/notify")
    ///                 .with_description("RTV market")
    ///                 .with_products([
    ///                     Product::new("Wireless Mouse for Laptop", 15000, 1),
    ///                     Product::new("HDMI cable", 6000, 1),
    ///                 ].into_iter()),
    ///             )
    ///             .await;
    /// }
    /// ```
    pub async fn create_order(&mut self, order: OrderCreateRequest) -> Result<res::CreateOrder> {
        self.authorize().await?;
        if order.total_amount
            != order
                .products
                .iter()
                .fold(0, |memo, p| memo + (p.unit_price * p.quantity as i32))
        {
            return Err(Error::IncorrectTotal);
        }

        if order.buyer().is_none() {
            return Err(Error::NoBuyer);
        }

        if order.description().trim().is_empty() {
            return Err(Error::NoDescription);
        }

        let bearer = self.bearer.as_ref().cloned().unwrap_or_default();
        let path = format!("{}/orders", self.base_url());
        let client = get_client!(self);
        let text = client
            .post(path)
            .bearer_auth(bearer)
            .json(&order.with_merchant_pos_id(self.merchant_pos_id))
            .send()
            .await?
            .text()
            .await?;
        log::trace!("Response: {}", text);
        let res: res::CreateOrder = serde_json::from_str(&text).map_err(|e| {
            log::error!("{e:?}");
            Error::CreateOrder
        })?;
        if !res.status.is_success() {
            let Status {
                status_code,
                status_desc,
                code,
                severity,
                code_literal,
            } = res.status;
            return Err(Error::CreateFailed {
                status_desc,
                status_code,
                code,
                severity,
                code_literal,
            });
        }
        Ok(res)
    }

    /// The PayU system fully supports refunds for processed payments, the
    /// balance of which is transferred directly to the buyer’s account.
    ///
    /// > For „PayU | Pay later” payment method funds are transferred to a
    /// > credit provider.
    ///
    /// You can process refund requests as either full or partial. For partial
    /// refunds, always specify the amount in the lowest unit of a given
    /// currency, which must be the same currency as the initial order (np. for
    /// Poland lowest currency unit is “grosz” so 10 pln should be given as
    /// 1000).
    ///
    /// You can send several partial refund requests for each single order. The
    /// total value of the requests must not exceed the order value.
    ///
    /// The payu system allows multiple partial refunds to be executed at the
    /// same time. To do so, the extRefundId parameter must be sent in the
    /// request. In situations where partial refunds will not be executed more
    /// than once per second, the extRefundId parameter is not required.
    ///
    /// # Examples
    ///
    /// ```
    /// # use pay_u::*;
    /// async fn partial_refund() {
    ///     let mut client = Client::new(ClientId::new("145227"), ClientSecret::new("12f071174cb7eb79d4aac5bc2f07563f"), MerchantPosId::new(300746))
    ///         .with_bearer("d9a4536e-62ba-4f60-8017-6053211d3f47", 2000)
    ///         .sandbox();
    ///     let res = client
    ///         .refund(
    ///             OrderId::new("H9LL64F37H160126GUEST000P01"),
    ///             RefundRequest::new("Refund", Some(1000)),
    ///         )
    ///         .await;
    /// }
    /// async fn full_refund() {
    ///     let mut client = Client::new(ClientId::new("145227"), ClientSecret::new("12f071174cb7eb79d4aac5bc2f07563f"), MerchantPosId::new(300746))
    ///         .with_bearer("d9a4536e-62ba-4f60-8017-6053211d3f47", 2000)
    ///         .sandbox();
    ///     let res = client
    ///         .refund(
    ///             OrderId::new("H9LL64F37H160126GUEST000P01"),
    ///             RefundRequest::new("Refund", None),
    ///         )
    ///         .await;
    /// }
    /// ```
    pub async fn refund(
        &mut self,
        order_id: OrderId,
        refund: RefundRequest,
    ) -> Result<res::RefundDetails> {
        #[derive(Serialize, Debug)]
        #[serde(rename_all = "camelCase")]
        struct RefundWrapper {
            refund: RefundRequest,
        }

        self.authorize().await?;
        if refund.description().trim().is_empty() {
            return Err(Error::NoDescription);
        }

        let bearer = self.bearer.as_ref().cloned().unwrap_or_default();
        let path = format!("{}/orders/{}/refunds", self.base_url(), order_id);
        let client = get_client!(self);
        let text = client
            .post(path)
            .bearer_auth(bearer)
            .json(&RefundWrapper { refund })
            .send()
            .await?
            .text()
            .await?;
        log::trace!("Response: {}", text);
        let res: res::RefundDetails = serde_json::from_str(&text).map_err(|e| {
            log::error!("Invalid PayU response {e:?}");
            Error::Refund
        })?;
        if !res.status.is_success() {
            let Status {
                status_code,
                status_desc,
                code,
                severity,
                code_literal,
            } = res.status;
            return Err(Error::RefundFailed {
                status_desc,
                status_code,
                code,
                severity,
                code_literal,
            });
        }
        Ok(res)
    }

    /// Order details request. You may use it to completely remove `Order`
    /// persistence and use extOrderId to connect your data with PayU data.
    ///
    /// # Examples
    ///
    /// ```
    /// # use pay_u::*;
    /// async fn order_details() {
    ///     let mut client = Client::new(ClientId::new("145227"), ClientSecret::new("12f071174cb7eb79d4aac5bc2f07563f"), MerchantPosId::new(300746))
    ///         .with_bearer("d9a4536e-62ba-4f60-8017-6053211d3f47", 2000)
    ///         .sandbox();
    ///     let res = client
    ///         .order_details(OrderId::new("H9LL64F37H160126GUEST000P01"))
    ///         .await;
    /// }
    /// ```
    pub async fn order_details(&mut self, order_id: OrderId) -> Result<res::OrderInfo> {
        self.authorize().await?;
        let bearer = self.bearer.as_ref().cloned().unwrap_or_default();
        let path = format!("{}/orders/{}", self.base_url(), order_id);
        let client = get_client!(self);
        let text = client
            .get(path)
            .bearer_auth(bearer)
            .send()
            .await?
            .text()
            .await?;
        log::trace!("Response: {}", text);
        dbg!(&text);
        let mut res: OrdersInfo = serde_json::from_str(&text).map_err(|e| {
            log::error!("{e:?}");
            dbg!(e);
            Error::OrderDetails
        })?;
        if !res.status.is_success() {
            let Status {
                status_code,
                status_desc,
                code,
                severity,
                code_literal,
            } = res.status;
            return Err(Error::OrderDetailsFailed {
                status_code,
                status_desc,
                code,
                severity,
                code_literal,
            });
        }
        Ok(res::OrderInfo {
            order: if res.orders.is_empty() {
                return Err(Error::NoOrderInDetails);
            } else {
                res.orders.remove(0)
            },
            status: res.status,
            properties: res.properties,
        })
    }

    /// The transaction retrieve request message enables you to retrieve the
    /// details of transactions created for an order.
    ///
    /// Using this endpoint is extremely useful if you would like to get bank
    /// account details or card details.
    ///
    /// > Please note that although card details are available right after
    /// > transaction has been processed, the bank details may be available
    /// > either after few minutes or on the next business day, depending on the
    /// > bank.
    ///
    /// # Examples
    ///
    /// ```
    /// # use pay_u::*;
    /// async fn order_transactions() {
    ///     let mut client = Client::new(ClientId::new("145227"), ClientSecret::new("12f071174cb7eb79d4aac5bc2f07563f"), MerchantPosId::new(300746))
    ///         .with_bearer("d9a4536e-62ba-4f60-8017-6053211d3f47", 2000)
    ///         .sandbox();
    ///     let res = client
    ///         .order_transactions(OrderId::new("H9LL64F37H160126GUEST000P01"))
    ///         .await;
    /// }
    /// ```
    pub async fn order_transactions(&mut self, order_id: OrderId) -> Result<res::Transactions> {
        self.authorize().await?;
        let bearer = self.bearer.as_ref().cloned().unwrap_or_default();
        let path = format!("{}/orders/{}/transactions", self.base_url(), order_id);
        let client = get_client!(self);
        let text = client
            .get(path)
            .bearer_auth(bearer)
            .send()
            .await?
            .text()
            .await?;
        log::trace!("Response: {}", text);
        dbg!(&text);
        serde_json::from_str(&text).map_err(|e| {
            log::error!("{e:?}");
            dbg!(e);
            Error::OrderTransactions
        })
    }

    /// The transaction retrieve request message enables you to retrieve the
    /// details of transactions created for an order.
    ///
    /// Using this endpoint is extremely useful if you would like to get bank
    /// account details or card details.
    ///
    /// > Please note that although card details are available right after
    /// > transaction has been processed, the bank details may be available
    /// > either after few minutes or on the next business day, depending on the
    /// > bank.
    ///
    /// # Examples
    ///
    /// ```
    /// # use pay_u::*;
    /// async fn order_transactions() {
    ///     let mut client = Client::new(ClientId::new("145227"), ClientSecret::new("12f071174cb7eb79d4aac5bc2f07563f"), MerchantPosId::new(300746))
    ///         .with_bearer("d9a4536e-62ba-4f60-8017-6053211d3f47", 2000)
    ///         .sandbox();
    ///     let res = client
    ///         .order_transactions(OrderId::new("H9LL64F37H160126GUEST000P01"))
    ///         .await;
    /// }
    /// ```
    pub async fn order_refunds(&mut self, order_id: OrderId) -> Result<Vec<Refund>> {
        self.authorize().await?;
        let bearer = self.bearer.as_ref().cloned().unwrap_or_default();
        let path = format!("{}/orders/{}/refunds", self.base_url(), order_id);
        let client = get_client!(self);
        let text = client
            .get(path)
            .bearer_auth(bearer)
            .send()
            .await?
            .text()
            .await?;
        log::trace!("Response: {}", text);
        let res::Refunds { refunds, .. } = serde_json::from_str(&text).map_err(|e| {
            log::error!("{e:?}");
            Error::OrderRefunds
        })?;
        Ok(refunds)
    }

    /// Get or refresh token
    pub async fn authorize(&mut self) -> Result<bool> {
        use chrono::{Duration, Utc};
        if Utc::now() - Duration::seconds(1) < self.bearer_expires_at {
            return Ok(true);
        }
        #[derive(Deserialize)]
        struct BearerResult {
            access_token: String,
            expires_in: i64,
        }

        let client = get_client!(self);
        let res = client.post(&format!(
            "https://secure.payu.com/pl/standard/user/oauth/authorize?grant_type=client_credentials&client_id={}&client_secret={}",
            self.client_id,
            self.client_secret
        ))
            .send()
            .await?;
        let res = res.json::<BearerResult>().await.map_err(|e| {
            log::error!("{e}");
            Error::Unauthorized
        })?;
        log::trace!("Bearer is {}", res.access_token);
        self.bearer_expires_at = Utc::now() + Duration::seconds(res.expires_in);
        self.bearer = Some(res.access_token);
        Ok(true)
    }

    fn base_url(&self) -> &str {
        if self.sandbox {
            "https://secure.snd.payu.com/api/v2_1"
        } else {
            "https://secure.payu.com/api/v2_1"
        }
    }

    fn build_client() -> reqwest::Client {
        reqwest::ClientBuilder::default()
            .user_agent("curl/7.82.0")
            .use_native_tls()
            // Do not follow redirect!
            .redirect(redirect::Policy::none())
            .connection_verbose(true)
            .build()
            .expect("Failed to create client")
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::res::CreateOrder;

    fn build_client() -> Client {
        dotenv::dotenv().ok();
        Client::new(
            ClientId::new("145227"),
            ClientSecret::new("12f071174cb7eb79d4aac5bc2f07563f"),
            MerchantPosId::new(300746),
        )
        .sandbox()
        .with_bearer("d9a4536e-62ba-4f60-8017-6053211d3f47", 999999)
    }

    async fn perform_create_order(client: &mut Client) -> Result<CreateOrder> {
        client
            .create_order(
                OrderCreateRequest::new(
                    Buyer::new("john.doe@example.com", "654111654", "John", "Doe", "pl"),
                    "127.0.0.1",
                    "PLN",
                )
                .with_notify_url("https://your.eshop.com/notify")
                .with_description("RTV market")
                .with_products(
                    [
                        Product::new("Wireless Mouse for Laptop", 15000, 1),
                        Product::new("HDMI cable", 6000, 1),
                    ]
                    .into_iter(),
                ),
            )
            .await
    }

    #[tokio::test]
    async fn create_order() {
        let mut client = build_client();
        let res = perform_create_order(&mut client).await;

        if res.is_err() {
            eprintln!("create_order res is {res:?}");
        }
        assert!(res.is_ok());
    }

    #[tokio::test]
    async fn partial_refund() {
        let mut client = build_client();
        let CreateOrder { order_id, .. } = perform_create_order(&mut client)
            .await
            .expect("Failed to create");
        let res = client
            .refund(order_id, RefundRequest::new("Refund", Some(10)))
            .await;

        if res.is_err() {
            eprintln!("partial_refund res is {res:?}");
        }
        assert!(matches!(res, Err(Error::RefundFailed { .. })));
    }

    #[tokio::test]
    async fn full_refund() {
        let mut client = build_client();
        let CreateOrder { order_id, .. } = perform_create_order(&mut client)
            .await
            .expect("Failed to create");
        let res = client
            .refund(order_id, RefundRequest::new("Refund", None))
            .await;

        if res.is_err() {
            eprintln!("full_refund res is {res:?}");
        }
        assert!(matches!(res, Err(Error::RefundFailed { .. })));
    }

    #[tokio::test]
    async fn order_details() {
        let mut client = build_client();
        let CreateOrder { order_id, .. } = perform_create_order(&mut client)
            .await
            .expect("Failed to create");
        let res = client.order_details(order_id).await;

        if res.is_err() {
            eprintln!("order_details res is {res:?}");
        }
        assert!(matches!(res, Ok(res::OrderInfo { .. })));
    }

    #[tokio::test]
    async fn order_transactions() {
        let mut client = build_client();
        let CreateOrder { order_id, .. } = perform_create_order(&mut client)
            .await
            .expect("Failed to create");
        let res = client.order_transactions(order_id).await;
        if res.is_err() {
            eprintln!("order_transactions res is {res:?}");
        }
        assert!(matches!(res, Err(Error::OrderTransactions)));
    }

    #[test]
    fn check_accepted_refund_json() {
        let res = serde_json::from_str::<res::RefundDetails>(include_str!(
            "../tests/responses/accepted_refund.json"
        ));
        assert!(res.is_ok());
    }
    #[test]
    fn check_cancel_json() {
        let res = serde_json::from_str::<notify::StatusUpdate>(include_str!(
            "../tests/responses/cancel.json"
        ));
        assert!(res.is_ok());
    }
    #[test]
    fn check_completed_cart_token_json() {
        let res = serde_json::from_str::<notify::StatusUpdate>(include_str!(
            "../tests/responses/completed_cart_token.json"
        ));
        assert!(res.is_ok());
    }
    #[test]
    fn check_completed_installments_json() {
        let res = serde_json::from_str::<notify::StatusUpdate>(include_str!(
            "../tests/responses/completed_installments.json"
        ));
        assert!(res.is_ok());
    }
    #[test]
    fn check_completed_pbl_json() {
        let res = serde_json::from_str::<notify::StatusUpdate>(include_str!(
            "../tests/responses/completed_pbl.json"
        ));
        assert!(res.is_ok());
    }
    #[test]
    fn check_refund_json() {
        let res = serde_json::from_str::<notify::RefundUpdate>(include_str!(
            "../tests/responses/refund.json"
        ));
        assert!(res.is_ok());
    }
    #[test]
    fn check_rejection_json() {
        let res = serde_json::from_str::<res::RefundDetails>(include_str!(
            "../tests/responses/rejection.json"
        ));
        assert!(res.is_ok());
    }
    #[test]
    fn check_custom_literal_json() {
        let res = serde_json::from_str::<res::RefundDetails>(include_str!(
            "../tests/responses/custom_code_literal.json"
        ));
        assert!(res.is_ok());
    }
}
