mod deserialize;
mod serialize;

use std::sync::Arc;

use reqwest::redirect;
use serde::{Deserialize, Serialize};

macro_rules! get_client {
    ($self:expr) => {{
        #[cfg(feature = "single-client")]
        {
            $self.client.clone()
        }
        #[cfg(not(feature = "single-client"))]
        {
            Client::build_client()
        }
    }};
}

#[derive(Debug, thiserror::Error)]
pub enum Error {
    #[error("Client is not authorized. No bearer token available")]
    NoToken,
    #[error("{0}")]
    Io(#[from] std::io::Error),
    #[error("Total value is not sum of products price")]
    IncorrectTotal,
    #[error("{0}")]
    Reqwest(#[from] reqwest::Error),
    #[error("Buyer is required to place an order")]
    NoBuyer,
    #[error("Description is required to place an order")]
    NoDescription,
    #[error("Client is not authorized")]
    Unauthorized,
    #[error("Refund returned invalid response")]
    Refund,
    #[error("Create order returned invalid response")]
    CreateOrder,
}

pub type Result<T> = std::result::Result<T, Error>;

/// PayU payment status.
///
/// Each payment is initially Pending and can change according to following
/// graph:
///
/// <img src="https://developers.payu.com/images/order_statusesV2-en.png">
#[derive(Serialize, Deserialize, Copy, Clone, Debug)]
#[serde(rename_all = "SCREAMING_SNAKE_CASE")]
pub enum PaymentStatus {
    /// Payment is currently being processed.
    Pending,
    /// PayU is currently waiting for the merchant system to receive (capture)
    /// the payment. This status is set if auto-receive is disabled on the
    /// merchant system.
    WaitingForConfirmation,
    /// Payment has been accepted. PayU will pay out the funds shortly.
    Completed,
    /// Payment has been cancelled and the buyer has not been charged (no money
    /// was taken from buyer's account).
    Canceled,
}

#[derive(Serialize, Deserialize, Copy, Clone, Debug)]
#[serde(rename_all = "SCREAMING_SNAKE_CASE")]
pub enum RefundStatus {
    /// refund was completed successfully
    Finalized,
    /// refund was cancelled
    Canceled,
    /// refund in progress
    Pending,
    /// PayU is currently waiting for the merchant system to receive (capture)
    /// the payment. This status is set if auto-receive is disabled on the
    /// merchant system.
    WaitingForConfirmation,
    /// Payment has been accepted. PayU will pay out the funds shortly.
    Completed,
}

#[derive(Serialize, Deserialize, Debug, Default)]
#[serde(rename_all = "camelCase")]
pub struct Buyer {
    /// Required customer e-mail
    email: String,
    /// Required customer phone number
    phone: String,
    /// Required customer first name
    first_name: String,
    /// Required customer last name
    last_name: String,
    /// Required customer language
    language: String,
}

impl Buyer {
    pub fn new<Email, Phone, FirstName, LastName, Language>(
        email: Email,
        phone: Phone,
        first_name: FirstName,
        last_name: LastName,
        lang: Language,
    ) -> Self
    where
        Email: Into<String>,
        Phone: Into<String>,
        FirstName: Into<String>,
        LastName: Into<String>,
        Language: Into<String>,
    {
        Self {
            email: email.into(),
            phone: phone.into(),
            first_name: first_name.into(),
            last_name: last_name.into(),
            language: lang.into(),
        }
    }

    pub fn email(&self) -> &str {
        &self.email
    }
    pub fn with_email<S>(mut self, email: S) -> Self
    where
        S: Into<String>,
    {
        self.email = email.into();
        self
    }
    pub fn phone(&self) -> &str {
        &self.phone
    }
    pub fn with_phone<S>(mut self, phone: S) -> Self
    where
        S: Into<String>,
    {
        self.phone = phone.into();
        self
    }
    pub fn first_name(&self) -> &str {
        &self.first_name
    }
    pub fn with_first_name<S>(mut self, first_name: S) -> Self
    where
        S: Into<String>,
    {
        self.first_name = first_name.into();
        self
    }
    pub fn last_name(&self) -> &str {
        &self.last_name
    }
    pub fn with_last_name<S>(mut self, last_name: S) -> Self
    where
        S: Into<String>,
    {
        self.last_name = last_name.into();
        self
    }
    pub fn language(&self) -> &str {
        &self.language
    }
    pub fn with_language<S>(mut self, language: S) -> Self
    where
        S: Into<String>,
    {
        self.language = language.into();
        self
    }
}

pub type Price = i32;
pub type Quantity = u32;
pub type MerchantPosId = i32;

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Product {
    pub name: String,
    #[serde(
        serialize_with = "serialize::serialize_i32",
        deserialize_with = "deserialize::deserialize_i32"
    )]
    pub unit_price: Price,
    #[serde(
        serialize_with = "serialize::serialize_u32",
        deserialize_with = "deserialize::deserialize_u32"
    )]
    pub quantity: Quantity,
}

impl Product {
    pub fn new<Name: Into<String>>(name: Name, unit_price: Price, quantity: Quantity) -> Self {
        Self {
            name: name.into(),
            unit_price,
            quantity,
        }
    }
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct OrderCreateRequest {
    /// URL to which web hook will be send. It's important to return 200 to all
    /// notifications.
    ///
    /// All notifications are send as POST with JSON payload
    ///
    /// Notifications are sent immediately after a payment status changes. If
    /// the notification is not received by the Shop application, it will be
    /// sent again in accordance with the table below:
    ///
    /// | Attempt | Time |
    /// |---------|------|
    /// | 1 | immediately |
    /// | 2 | 1 minute |
    /// | 3 | 2 minutes |
    /// | 4 | 5 minutes |
    /// | 5 | 10 minutes |
    /// | 6 | 30 minutes |
    /// | 7 | 1 hour |
    /// | 8 | 2 hours |
    /// | 9 | 3 hours |
    /// | 10| 6 hours |
    /// | 11| 9 hours |
    /// | 12| 12 hours |
    /// | 13| 15 hours |
    /// | 14| 18 hours |
    /// | 15| 21 hours |
    /// | 16| 24 hours |
    /// | 17| 36 hours |
    /// | 18| 48 hours |
    /// | 19| 60 hours |
    /// | 20| 72 hours |
    #[serde(skip_serializing_if = "Option::is_none")]
    notify_url: Option<String>,
    /// Customer client IP address
    customer_ip: String,
    /// Secret pos ip. This is connected to PayU account
    #[serde(
        serialize_with = "serialize::serialize_i32",
        deserialize_with = "deserialize::deserialize_i32"
    )]
    merchant_pos_id: MerchantPosId,
    /// Transaction description
    description: String,
    /// 3 characters currency identifier, ex. PLN
    currency_code: String,
    /// Total price of the order in pennies (e.g. 1000 is 10.00 EUR). Applies
    /// also to currencies without subunits (e.g. 1000 is 10 HUF).
    #[serde(
        serialize_with = "serialize::serialize_i32",
        deserialize_with = "deserialize::deserialize_i32"
    )]
    total_amount: Price,
    /// @see [Buyer]
    buyer: Option<Buyer>,
    /// List of products
    products: Vec<Product>,
    #[serde(skip_serializing)]
    order_create_date: Option<String>,
}

impl OrderCreateRequest {
    pub fn new<CustomerIp, Currency>(
        buyer: Buyer,
        customer_ip: CustomerIp,
        currency: Currency,
    ) -> Self
    where
        CustomerIp: Into<String>,
        Currency: Into<String>,
    {
        Self {
            notify_url: None,
            customer_ip: customer_ip.into(),
            merchant_pos_id: 0,
            description: String::from(""),
            currency_code: currency.into(),
            total_amount: 0,
            buyer: Some(buyer),
            products: Vec::new(),
            order_create_date: None,
        }
    }

    pub fn with_products<Products>(mut self, products: Products) -> Self
    where
        Products: Iterator<Item = Product>,
    {
        self.products.extend(products);
        self.total_amount = self
            .products
            .iter()
            .fold(0, |agg, p| agg + (p.quantity as i32 * p.unit_price as i32));
        self
    }

    pub fn with_product(mut self, product: Product) -> Self {
        self.products.push(product);
        self.total_amount = self
            .products
            .iter()
            .fold(0, |agg, p| agg + (p.quantity as i32 * p.unit_price as i32));
        self
    }

    pub fn with_description<Description>(mut self, desc: Description) -> Self
    where
        Description: Into<String>,
    {
        self.description = String::from(desc.into().trim());
        self
    }

    /// Add url to which PayU will be able to send http request with payment
    /// status updates
    ///
    /// All requests from PayU should receive 200 response!
    ///
    /// See more [Order::notify_url]
    pub fn with_notify_url<NotifyUrl>(mut self, notify_url: NotifyUrl) -> Self
    where
        NotifyUrl: Into<String>,
    {
        self.notify_url = Some(notify_url.into());
        self
    }

    /// URL to which web hook will be send. It's important to return 200 to all
    /// notifications.
    ///
    /// All notifications are send as POST with JSON payload
    ///
    /// Notifications are sent immediately after a payment status changes. If
    /// the notification is not received by the Shop application, it will be
    /// sent again in accordance with the table below:
    ///
    /// | Attempt | Time |
    /// |---------|------|
    /// | 1 | immediately |
    /// | 2 | 1 minute |
    /// | 3 | 2 minutes |
    /// | 4 | 5 minutes |
    /// | 5 | 10 minutes |
    /// | 6 | 30 minutes |
    /// | 7 | 1 hour |
    /// | 8 | 2 hours |
    /// | 9 | 3 hours |
    /// | 10| 6 hours |
    /// | 11| 9 hours |
    /// | 12| 12 hours |
    /// | 13| 15 hours |
    /// | 14| 18 hours |
    /// | 15| 21 hours |
    /// | 16| 24 hours |
    /// | 17| 36 hours |
    /// | 18| 48 hours |
    /// | 19| 60 hours |
    /// | 20| 72 hours |
    pub fn notify_url(&self) -> &Option<String> {
        &self.notify_url
    }

    /// Customer IP address from http request received from client
    pub fn customer_ip(&self) -> &String {
        &self.customer_ip
    }

    ///
    pub fn merchant_pos_id(&self) -> MerchantPosId {
        self.merchant_pos_id
    }
    pub fn description(&self) -> &String {
        &self.description
    }
    pub fn currency_code(&self) -> &String {
        &self.currency_code
    }
    pub fn total_amount(&self) -> &Price {
        &self.total_amount
    }
    pub fn buyer(&self) -> &Option<Buyer> {
        &self.buyer
    }
    pub fn products(&self) -> &[Product] {
        &self.products
    }
    pub fn order_create_date(&self) -> &Option<String> {
        &self.order_create_date
    }

    pub(crate) fn with_merchant_pos_id(mut self, merchant_pos_id: MerchantPosId) -> Self {
        self.merchant_pos_id = merchant_pos_id;
        self
    }
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "SCREAMING_SNAKE_CASE")]
pub enum PaymentType {
    Pbl,
    CardToken,
    Installments,
}

/// Wrapper around pay method. It's used only for deserializing notifications
///
/// # Examples
///
/// ```
/// # use pay_u::PayMethod;
/// let method: PayMethod = serde_json::from_str(r#"
///     {
///         "type": "INSTALLMENTS"
///     }
/// "#).unwrap();
/// ```
#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct PayMethod {
    #[serde(rename = "type")]
    pub payment_type: PaymentType,
}

#[derive(Deserialize, Serialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Status {
    status_code: String,
    status_desc: Option<String>,
    code: Option<String>,
    severity: Option<String>,
    code_literal: Option<CodeLiteral>,
}

#[derive(Serialize, Deserialize, Copy, Clone, Debug)]
#[serde(rename_all = "SCREAMING_SNAKE_CASE")]
pub enum StatusCode {
    ErrorValueMissing,
    OpenpayuBusinessError,
    OpenpayuErrorValueInvalid,
    OpenpayuErrorInternal,
}

#[derive(Serialize, Deserialize, Clone, Debug)]
#[serde(rename_all = "SCREAMING_SNAKE_CASE")]
pub enum CodeLiteral {
    /// Request lacks "refund" object.
    MissingRefundSection,

    /// Transaction has not been finalized
    TransNotEnded,

    /// Lack of funds in account
    NoBalance,

    /// Refund amount exceeds transaction amount
    AmountToBig,

    /// Refund value is too small
    AmountToSmall,

    /// Refunds have been disabled
    RefundDisabled,

    /// Too many refund attempts have been made
    RefundToOften,

    /// Refund was already created
    Paid,

    /// Unknown error
    UnknownError,

    /// extRefundId was re-used and other params do not match the values
    /// sent during the first call.
    RefundIdempotencyMismatch,

    /// Shop billing has not yet been completed
    TransBillingEntriesNotCompleted,

    /// The available time for refund has passed.
    TransTooOld,

    /// Transaction amount that remains after refund creation will be too
    /// small to make another refund.
    RemainingTransAmountTooSmall,

    #[serde(other)]
    /// Implementation changed
    Unknown,
}

impl Status {
    /// Check if http request was successful
    ///
    /// # Examples
    ///
    /// ```
    /// # use pay_u::Status;
    /// let status: Status = serde_json::from_str("{\"statusCode\":\"SUCCESS\"}").unwrap();
    /// assert_eq!(status.is_success(), true);
    /// ```
    pub fn is_success(&self) -> bool {
        self.status_code.as_str() == "SUCCESS"
    }

    /// Returns http status
    ///
    /// # Examples
    ///
    /// ```
    /// # use pay_u::Status;
    /// let status: Status = serde_json::from_str("{\"statusCode\":\"SUCCESS\"}").unwrap();
    /// assert_eq!(status.status_code(), "SUCCESS");
    /// ```
    pub fn status_code(&self) -> &str {
        &self.status_code
    }

    pub fn status_desc(&self) -> Option<&str> {
        self.status_desc.as_deref()
    }
}

#[derive(Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct CreateOrderResult {
    /// Http status as a text
    pub status: Status,
    /// Client should be redirected to this URI
    pub redirect_uri: String,
    /// This should be connected to your own order
    pub order_id: String,
    /// This is YOUR_EXT_ORDER_ID
    pub ext_order_id: Option<String>,
}

#[derive(Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct PartialRefundResult {
    pub order_id: Option<String>,
    pub refund: Option<Refund>,
    pub status: Status,
}

#[derive(Deserialize, Serialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct RefundRequest {
    description: String,
    amount: Price,
}

impl RefundRequest {
    pub fn new<Description>(description: Description, amount: Price) -> Self
    where
        Description: Into<String>,
    {
        Self {
            description: description.into(),
            amount,
        }
    }

    pub fn description(&self) -> &str {
        &self.description
    }
    pub fn amount(&self) -> Price {
        self.amount
    }
}

#[derive(Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Refund {
    pub refund_id: String,
    pub ext_refund_id: Option<String>,
    pub amount: String,
    pub currency_code: String,
    pub description: String,
    pub creation_date_time: String,
    pub status: String,
    pub status_date_time: String,
}

pub mod notify {
    use serde::Deserialize;

    /// Payment notification object received on [super::Order].[notify_url]
    #[derive(Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct StatusUpdate {
        pub order: Order,
        pub local_receipt_date_time: Option<String>,
        pub properties: Option<Vec<Prop>>,
        pub status: Option<super::Status>,
    }

    impl StatusUpdate {
        pub fn status(&self) -> super::PaymentStatus {
            self.order.status
        }
    }

    /// Refund notification object
    #[derive(Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct RefundUpdate {
        pub ext_order_id: String,
        pub order_id: String,
        pub refund: Refund,
    }

    #[derive(Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct Refund {
        pub refund_id: String,
        pub amount: String,
        pub currency_code: String,
        pub status: super::RefundStatus,
        pub status_date_time: String,
        pub reason: String,
        pub reason_description: String,
        pub refund_date: String,
    }

    #[derive(Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct Order {
        #[serde(flatten)]
        pub order: super::OrderCreateRequest,
        pub pay_method: Option<super::PayMethod>,
        pub status: super::PaymentStatus,
    }

    impl std::ops::Deref for Order {
        type Target = super::OrderCreateRequest;

        fn deref(&self) -> &Self::Target {
            &self.order
        }
    }

    #[derive(Deserialize, Debug)]
    #[serde(rename_all = "camelCase")]
    pub struct Prop {
        pub name: String,
        pub value: String,
    }
}

pub struct Client {
    sandbox: bool,
    merchant_pos_id: MerchantPosId,
    client_id: String,
    client_secret: String,
    bearer: Option<String>,
    bearer_expires_at: chrono::DateTime<chrono::Utc>,
    #[cfg(feature = "single-client")]
    client: Arc<reqwest::Client>,
}

impl Client {
    /// Create new PayU client
    pub fn new<ClientId, ClientSecret>(
        client_id: ClientId,
        client_secret: ClientSecret,
        merchant_pos_id: MerchantPosId,
    ) -> Self
    where
        ClientId: Into<String>,
        ClientSecret: Into<String>,
    {
        #[cfg(feature = "single-client")]
        {
            Self {
                bearer: None,
                sandbox: false,
                merchant_pos_id,
                client_id: client_id.into(),
                client_secret: client_secret.into(),
                bearer_expires_at: chrono::Utc::now(),
                client: Arc::new(Self::build_client()),
            }
        }
        #[cfg(not(feature = "single-client"))]
        {
            Self {
                bearer: None,
                sandbox: false,
                merchant_pos_id,
                client_id: client_id.into(),
                client_secret: client_secret.into(),
                bearer_expires_at: chrono::Utc::now(),
            }
        }
    }

    /// All operation will be performed in sandbox PayU environment
    pub fn sandbox(mut self) -> Self {
        self.sandbox = true;
        self
    }

    /// Set your own bearer key
    pub fn with_bearer<Bearer: Into<String>>(mut self, bearer: Bearer, expires_in: i64) -> Self {
        self.bearer = Some(bearer.into());
        self.bearer_expires_at = chrono::Utc::now() + chrono::Duration::seconds(expires_in);
        self
    }

    /// Create new order in PayU
    ///
    /// ### IMPORTANT:
    /// Do not follow redirect for any reason. Location points to
    /// payment page which is useless in this context
    ///
    /// ### IMPORTANT:
    /// Do not use rustls. It'll freeze application!
    ///
    /// # Examples
    ///
    /// ```rust
    /// # use pay_u::{Client, OrderCreateRequest, Product, Buyer};
    /// async fn pay() {
    ///     let mut client = Client::new("145227", "12f071174cb7eb79d4aac5bc2f07563f", 300746)
    ///         .sandbox()
    ///         .with_bearer("d9a4536e-62ba-4f60-8017-6053211d3f47", 2000);
    ///     let res = client
    ///         .create_order(
    ///                 OrderCreateRequest::new(
    ///                     Buyer::new("john.doe@example.com", "654111654", "John", "Doe", "pl"),
    ///                     "127.0.0.1",
    ///                     "PLN",
    ///                 )
    ///                 .with_notify_url("https://your.eshop.com/notify")
    ///                 .with_description("RTV market")
    ///                 .with_products([
    ///                     Product::new("Wireless Mouse for Laptop", 15000, 1),
    ///                     Product::new("HDMI cable", 6000, 1),
    ///                 ].into_iter()),
    ///             )
    ///             .await;
    /// }
    /// ```
    pub async fn create_order(&mut self, order: OrderCreateRequest) -> Result<CreateOrderResult> {
        self.authorize().await?;
        if order.total_amount
            != order
                .products
                .iter()
                .fold(0, |memo, p| memo + (p.unit_price * p.quantity as i32))
        {
            return Err(Error::IncorrectTotal);
        }

        if order.buyer().is_none() {
            return Err(Error::NoBuyer);
        }

        if order.description().trim().is_empty() {
            return Err(Error::NoDescription);
        }

        let bearer = self.bearer.as_ref().cloned().unwrap_or_default();
        let path = format!("{}/orders", self.base_url());
        let client = get_client!(self);
        let text = client
            .post(path)
            .bearer_auth(bearer)
            .json(&order.with_merchant_pos_id(self.merchant_pos_id))
            .send()
            .await?
            .text()
            .await?;
        log::trace!("Response: {}", text);
        serde_json::from_str(&text).map_err(|e| {
            log::error!("{e:?}");
            Error::CreateOrder
        })
    }

    /// The PayU system fully supports refunds for processed payments, the
    /// balance of which is transferred directly to the buyer’s account.
    ///
    /// > For „PayU | Pay later” payment method funds are transferred to a
    /// > credit provider.
    ///
    /// You can process refund requests as either full or partial. For partial
    /// refunds, always specify the amount in the lowest unit of a given
    /// currency, which must be the same currency as the initial order (np. for
    /// Poland lowest currency unit is “grosz” so 10 pln should be given as
    /// 1000).
    ///
    /// You can send several partial refund requests for each single order. The
    /// total value of the requests must not exceed the order value.
    ///
    /// The payu system allows multiple partial refunds to be executed at the
    /// same time. To do so, the extRefundId parameter must be sent in the
    /// request. In situations where partial refunds will not be executed more
    /// than once per second, the extRefundId parameter is not required.
    ///
    /// # Examples
    ///
    /// ```
    /// # use pay_u::*;
    /// async fn perform_refund() {
    ///     let mut client = Client::new("145227", "12f071174cb7eb79d4aac5bc2f07563f", 300746)
    ///         .with_bearer("d9a4536e-62ba-4f60-8017-6053211d3f47", 2000)
    ///         .sandbox();
    ///     let res = client
    ///         .partial_refund(
    ///             "H9LL64F37H160126GUEST000P01",
    ///             RefundRequest::new("Refund", 1000),
    ///         )
    ///         .await;
    /// }
    /// ```
    pub async fn partial_refund<OrderId>(
        &mut self,
        order_id: OrderId,
        refund: RefundRequest,
    ) -> Result<PartialRefundResult>
    where
        OrderId: std::fmt::Display,
    {
        self.authorize().await?;
        if refund.description().trim().is_empty() {
            return Err(Error::NoDescription);
        }

        let bearer = self.bearer.as_ref().cloned().unwrap_or_default();
        let path = format!("{}/orders/{}/refunds", self.base_url(), order_id);
        let client = get_client!(self);
        let text = client
            .post(path)
            .bearer_auth(bearer)
            .json(&refund)
            .send()
            .await?
            .text()
            .await?;
        log::trace!("Response: {}", text);
        serde_json::from_str::<'_, PartialRefundResult>(&text).map_err(|e| {
            log::error!("Invalid PayU response {e:?}");
            Error::Refund
        })
    }

    /// Get or refresh token
    pub(crate) async fn authorize(&mut self) -> Result<bool> {
        use chrono::{Duration, Utc};
        if Utc::now() - Duration::seconds(1) < self.bearer_expires_at {
            return Ok(true);
        }
        #[derive(Deserialize)]
        struct BearerResult {
            access_token: String,
            expires_in: i64,
        }

        let client = get_client!(self);
        let res = client.post(&format!(
            "https://secure.payu.com/pl/standard/user/oauth/authorize?grant_type=client_credentials&client_id={}&client_secret={}",
            self.client_id,
            self.client_secret
        ))
            .send()
            .await?;
        let res = res.json::<BearerResult>().await.map_err(|e| {
            log::error!("{e}");
            Error::Unauthorized
        })?;
        log::trace!("Bearer is {}", res.access_token);
        self.bearer_expires_at = Utc::now() + Duration::seconds(res.expires_in);
        self.bearer = Some(res.access_token);
        Ok(true)
    }

    fn base_url(&self) -> &str {
        if self.sandbox {
            "https://secure.snd.payu.com/api/v2_1"
        } else {
            "https://secure.payu.com/api/v2_1"
        }
    }

    fn build_client() -> reqwest::Client {
        reqwest::ClientBuilder::default()
            .user_agent("curl/7.82.0")
            .use_native_tls()
            // Do not follow redirect!
            .redirect(redirect::Policy::none())
            .connection_verbose(true)
            .build()
            .expect("Failed to create client")
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    fn build_client() -> Client {
        dotenv::dotenv().ok();
        Client::new("145227", "12f071174cb7eb79d4aac5bc2f07563f", 300746)
            .sandbox()
            .with_bearer("d9a4536e-62ba-4f60-8017-6053211d3f47", 999999)
    }

    #[tokio::test]
    async fn create_order() {
        let res = build_client()
            .create_order(
                OrderCreateRequest::new(
                    Buyer::new("john.doe@example.com", "654111654", "John", "Doe", "pl"),
                    "127.0.0.1",
                    "PLN",
                )
                .with_notify_url("https://your.eshop.com/notify")
                .with_description("RTV market")
                .with_products(
                    [
                        Product::new("Wireless Mouse for Laptop", 15000, 1),
                        Product::new("HDMI cable", 6000, 1),
                    ]
                    .into_iter(),
                ),
            )
            .await;
        assert!(res.is_ok());
    }

    #[tokio::test]
    async fn partial_refund() {
        let res = build_client()
            .partial_refund(
                "H9LL64F37H160126GUEST000P01",
                RefundRequest::new("Refund", 1000),
            )
            .await;
        eprintln!("{:?}", res);
        assert!(matches!(res, Ok(_)));
    }

    #[tokio::test]
    async fn check_refund() {}

    #[test]
    fn check_accepted_refund_json() {
        let res = serde_json::from_str::<PartialRefundResult>(include_str!(
            "../tests/responses/accepted_refund.json"
        ));
        assert!(res.is_ok());
    }
    #[test]
    fn check_cancel_json() {
        let res = serde_json::from_str::<notify::StatusUpdate>(include_str!(
            "../tests/responses/cancel.json"
        ));
        assert!(res.is_ok());
    }
    #[test]
    fn check_completed_cart_token_json() {
        let res = serde_json::from_str::<notify::StatusUpdate>(include_str!(
            "../tests/responses/completed_cart_token.json"
        ));
        assert!(res.is_ok());
    }
    #[test]
    fn check_completed_installments_json() {
        let res = serde_json::from_str::<notify::StatusUpdate>(include_str!(
            "../tests/responses/completed_installments.json"
        ));
        assert!(res.is_ok());
    }
    #[test]
    fn check_completed_pbl_json() {
        let res = serde_json::from_str::<notify::StatusUpdate>(include_str!(
            "../tests/responses/completed_pbl.json"
        ));
        assert!(res.is_ok());
    }
    #[test]
    fn check_refund_json() {
        let res = serde_json::from_str::<notify::RefundUpdate>(include_str!(
            "../tests/responses/refund.json"
        ));
        assert!(res.is_ok());
    }
    #[test]
    fn check_rejection_json() {
        let res = serde_json::from_str::<PartialRefundResult>(include_str!(
            "../tests/responses/rejection.json"
        ));
        assert!(res.is_ok());
    }
    #[test]
    fn check_custom_literal_json() {
        let res = serde_json::from_str::<PartialRefundResult>(include_str!(
            "../tests/responses/custom_code_literal.json"
        ));
        assert!(res.is_ok());
    }
}
