use crate::integration::popperjs;

use crate::integration::popperjs::{from_popper, Instance};
use std::fmt::Debug;
use std::marker::PhantomData;
use wasm_bindgen::closure::Closure;
use wasm_bindgen::JsValue;
use yew::prelude::*;

// popper

#[derive(Clone, Debug, PartialEq, Properties)]
pub struct Props<T>
where
    T: Clone + PartialEq + Debug,
{
    #[prop_or_default]
    pub children: Children,
    #[prop_or_default]
    pub active: bool,

    pub content: T,

    /// Close callback that will be emitted when the popper's component will emit the onclose callback.
    #[prop_or_default]
    pub onclose: Callback<()>,
}

pub struct Popper<C>
where
    C: PopperContent + 'static,
    C::Properties: PartialEq + Debug,
{
    target: NodeRef,
    content: NodeRef,
    popper: Option<popperjs::Instance>,
    _callback: Option<Closure<dyn Fn(&Instance)>>,

    state: Option<popperjs::State>,

    _marker: PhantomData<C>,
}

#[derive(Clone, Debug)]
pub enum Msg {
    Close,
    State(popperjs::State),
}

impl<C> Component for Popper<C>
where
    C: PopperContent + 'static,
    C::Properties: Clone + PartialEq + Debug,
{
    type Message = Msg;
    type Properties = Props<C::Properties>;

    fn create(_: &Context<Self>) -> Self {
        Self {
            target: NodeRef::default(),
            content: NodeRef::default(),
            popper: None,
            _callback: None,
            state: None,
            _marker: Default::default(),
        }
    }

    fn update(&mut self, ctx: &Context<Self>, msg: Self::Message) -> bool {
        match msg {
            Msg::State(state) => {
                let state = Some(state);
                let mut changed = false;
                if self.state != state {
                    self.state = state;
                    changed = true;
                }
                changed
            }
            Msg::Close => {
                ctx.props().onclose.emit(());
                false
            }
        }
    }

    fn changed(&mut self, ctx: &Context<Self>) -> bool {
        if ctx.props().active {
            self.show(ctx).ok();
        } else {
            self.hide();
        }
        true
    }

    fn view(&self, ctx: &Context<Self>) -> Html {
        self.check_update().ok();

        let onclose = ctx.link().callback(|_| Msg::Close);

        let content = <C as PopperContent>::view(
            &ctx.props().content,
            onclose,
            self.content.clone(),
            self.state.clone(),
        );

        return html! {
            <>
                <span ref={self.target.clone()}>
                    { for ctx.props().children.iter() }
                </span>
                { content }
            </>
        };
    }
}

impl<C> Popper<C>
where
    C: PopperContent,
    C::Properties: Clone + PartialEq + Debug,
{
    fn show(&mut self, ctx: &Context<Self>) -> Result<(), JsValue> {
        if self.popper.is_some() {
            return Ok(());
        }

        let target = self
            .target
            .get()
            .ok_or_else(|| JsValue::from("Missing target"))?;
        let content = self
            .content
            .get()
            .ok_or_else(|| JsValue::from("Missing content"))?;

        let update = ctx.link().callback(|state| Msg::State(state));
        let update = Closure::wrap(Box::new(move |this: &Instance| {
            // web_sys::console::debug_2(&JsValue::from("apply: "), this);
            let msg = from_popper(this).unwrap();
            // log::info!("Msg: {:?}", msg);

            update.emit(msg);
        }) as Box<dyn Fn(&Instance)>);

        let opts = popperjs::create_default_opts(&update)?;

        //web_sys::console::debug_1(&opts);

        let popper = popperjs::create_popper(target, content, &opts);

        // web_sys::console::debug_1(&popper);
        self.popper = Some(popper);
        self._callback = Some(update);

        Ok(())
    }

    fn hide(&mut self) {
        self.destroy();
        self.state = None;
    }

    fn check_update(&self) -> Result<(), JsValue> {
        if let Some(popper) = &self.popper {
            popper.update();
        }
        Ok(())
    }

    fn destroy(&mut self) {
        if let Some(popper) = self.popper.take() {
            popper.destroy();
        }
    }
}

pub trait PopperContent: Component {
    fn view(
        props: &Self::Properties,
        onclose: Callback<()>,
        r#ref: NodeRef,
        state: Option<popperjs::State>,
    ) -> Html;
}
