use std::collections::HashMap;

use serde::{Deserialize, Serialize, Serializer, Deserializer};
use std::env::VarError;
use std::rc::Rc;
use serde::ser::SerializeMap;
use serde::de::{Visitor, MapAccess};
use core::fmt;
use std::fmt::Display;
use std::str::FromStr;
use std::io;

use crate::config::{ConfigSource, PathFlags};

/// Single entry in the to be generated _$PATH_ variable.
///
/// This type includes the path, the flags that regulate if it should be added
/// and the source of the Path (configuration file, environment, ...)
#[derive(Default, Debug, Clone, Serialize, Deserialize, Eq, PartialEq)]
pub struct Path {
    path: String,
    flags: PathFlags,
    #[serde(skip)]
    source: Option<Rc<ConfigSource>>,
}

impl Path {
    pub fn new<S>(path: S, flags: PathFlags) -> Path
        where S: ToString {
        Path {
            path: path.to_string(),
            flags,
            source: None,
        }
    }

    pub fn with_source<S, C> (path: S, flags: PathFlags, source: C) -> Path
        where S: ToString, C: Into<Rc<ConfigSource>> {
        Path {
            path: path.to_string(),
            flags,
            source: Some(source.into()),
        }
    }

    pub fn resolve(&self, env: &HashMap<String, String>) -> Option<String> {
        let path: Option<Vec<String>> = self.path
            .split('/')
            .map(|folder| {
                if folder == "~" {
                    env.get("HOME").map(String::to_string)
                } else if let Some(env_name) = folder.strip_prefix('$') {
                    env.get(env_name).map(String::to_string)
                } else {
                    Some(folder.to_string())
                }
            })
            .collect();
        path.map(|path| path.join("/"))
    }

    /// Returns the contained path string
    pub fn path(&self) -> &str {
        &self.path
    }

    pub fn flags(&self) -> PathFlags {
        self.flags
    }

    /// Returns the source of where the path originates from
    pub fn source(&self) -> Option<&Rc<ConfigSource>> {
        self.source.as_ref()
    }

    /// Removes the source from the path
    ///
    /// # Examples
    ///
    /// ```
    /// use pathfix::config::{Path, PathFlags, ConfigSource};
    /// let path = Path::with_source("/foo/bar", "unix".parse().unwrap(), ConfigSource::PathVar);
    /// let wanted = Path::new("/foo/bar", "unix".parse().unwrap());
    ///
    /// assert_eq!(path.normalize(), wanted);
    /// ```
    pub fn normalize(self) -> Path {
        Path {
            path: self.path,
            flags: self.flags,
            source: None,
        }
    }
}

impl Display for Path {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if self.flags() != PathFlags::default() {
            write!(f, "{}|{}", self.path(), self.flags())
        } else {
            write!(f, "{}", self.path())
        }
    }
}

impl FromStr for Path {
    type Err = io::Error;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        let mut split = s.rsplitn(2, '|');
        let last = split.next().unwrap();
        Ok(if let Some(first) = split.next() {
            Path::new(first.trim(), last.parse()?)
        } else {
            Path::new(last.trim(), PathFlags::default())
        })
    }
}

impl<S> From<S> for Path
    where S: Into<String> {
    fn from(s: S) -> Self {
        Path {
            path: s.into(),
            ..Default::default()
        }
    }
}

/// List of Paths
///
/// This type is usually generated by deserializing it from configuration, parsing a path
/// or merging multiple of it's instances.
#[derive(Debug, Default, Clone, Eq, PartialEq)]
pub struct Paths(pub Vec<Path>);

impl Paths {
    pub fn new(v: Vec<Path>) -> Paths {
        Paths(v)
    }

    /// Reads PATH environment variable file and adds content to config.
    ///
    /// The PATH environment variable will be split on ':'
    pub fn from_env() -> Result<Paths, VarError> {
        Ok(Paths::from_path(
            &std::env::var("PATH")?
        ))
    }

    /// Parses a colon seperated path and sets that as included path.
    ///
    /// The parameter will be split on ':'
    pub fn from_path(path: &str) -> Paths {
        let config_source = Rc::new(ConfigSource::PathVar);
        Paths(
            path.split(':')
                .map(Path::from)
                .map(|mut path| {
                    path.source = Some(config_source.clone());
                    path
                })
                .collect()
        )
    }

    /// Merges two `Paths` structures.
    /// Values `other` Config will be inserted before `self.
    pub fn merge(self, other: Paths) -> Paths {
        Paths(other.0.iter().chain(self.0.iter()).map(ToOwned::to_owned).collect())
    }

    pub fn resolve(&self, system_flags: PathFlags, env: &HashMap<String, String>) -> Vec<String> {
        self.0.iter()
            .cloned()
            .filter(|p| p.flags.check(system_flags))
            .filter_map(|p| p.resolve(env))
            .collect()
    }

    /// Sets the source of all paths in the internal vector.
    pub fn set_source(&mut self, source: ConfigSource) {
        let rc = Rc::new(source);
        for path in self.0.iter_mut() {
            path.source = Some(rc.clone());
        }
    }

    /// Strips the `from` from all Paths
    pub fn normalize(self) -> Paths {
        Paths::new(self.0.into_iter().map(Path::normalize).collect())
    }
}

impl Serialize for Paths {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error> where
        S: Serializer,
    {
        let mut serialize_map = serializer.serialize_map(Some(self.0.len()))?;
        for path in &self.0 {
            serialize_map.serialize_entry(&path.path, &path.flags)?;
        }
        serialize_map.end()
    }
}

struct PathsVisitor;

impl<'de> Visitor<'de> for PathsVisitor {
    type Value = Paths;

    fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        formatter.write_str("an map that maps paths to their flags as strings")
    }

    fn visit_map<A>(self, mut map: A) -> Result<Paths, A::Error> where
        A: MapAccess<'de>, {
        let mut paths = Vec::new();
        while let Some((path, flags)) = map.next_entry()? {
            paths.push(Path {
                path, flags, ..Default::default()
            })
        }
        Ok(Paths(paths))
    }
}

impl<'de> Deserialize<'de> for Paths {
    fn deserialize<D>(deserializer: D) -> Result<Paths, D::Error> where
        D: Deserializer<'de> {
        deserializer.deserialize_map(PathsVisitor)
    }
}

impl<T> From<Vec<T>> for Paths
    where T: ToString {
    fn from(v: Vec<T>) -> Self {
        Paths(
            v.iter()
                .map(ToString::to_string)
                .map(Path::from)
                .collect()
        )
    }
}

#[cfg(test)]
mod tests {
    use std::collections::HashMap;
    use std::string::ToString;
    use std::rc::Rc;

    use crate::config::{Path, Paths, PathFlags, ConfigSource};
    use std::io;

    #[test]
    fn test_parse_path() {
        let cases = [
            ("/foo/bar", Path::new("/foo/bar", PathFlags::new())),
            ("/foo/bar|admin", Path::new("/foo/bar", "admin".parse().unwrap())),
            (
                "  /foo/bar|windows    |  admin  ",
                Path::new("/foo/bar|windows", "admin".parse().unwrap())
            ),
        ];
        for (s, wanted) in &cases {
            let path: Path = s.parse().unwrap();
            assert_eq!(&path, wanted);
        }

        let failures = [
            "/foo/bar|adsfhahdsf",
        ];
        for s in &failures {
            let path: Result<Path, io::Error> = s.parse();
            assert!(path.is_err());
        }
    }

    #[test]
    fn test_resolve() {
        let env: HashMap<String, String> = [("HOME", "/home/user"), ("FOO", "/foobar")]
            .iter()
            .map(|(k, v)| (k.to_string(), v.to_string()))
            .collect();
        let testvec = [
            (
                Path::from("/fnort/bar"),
                Some("/fnort/bar".to_string())
            ),
            (
                Path::from("$HOME/foo"),
                Some("/home/user/foo".to_string())
            ),
            (
                Path::from("~/foo"),
                Some("/home/user/foo".to_string())
            ),
            (
                Path::from("$UNKOWN/foo"),
                None
            ),
        ];

        for (path, wanted) in &testvec {
            assert_eq!(path.resolve(&env), *wanted);
        }
    }

    #[test]
    fn test_from_env() {
        use std::env;
        env::set_var("PATH", "/foo/bar:/fnorti/fnuff");
        let paths = Paths::from_env().unwrap();
        assert_eq!(paths, Paths::new(vec![
            Path::with_source("/foo/bar", PathFlags::default(), ConfigSource::PathVar),
            Path::with_source("/fnorti/fnuff", PathFlags::default(), ConfigSource::PathVar),
        ]));
    }

    #[test]
    fn test_from_path() {
        let source = Rc::new(ConfigSource::PathVar);
        let test_vec: Vec<(&'static str, Paths)> = vec![
            (
                "/foo/bar",
                Paths::new(vec![
                    Path::with_source("/foo/bar", PathFlags::default(), ConfigSource::PathVar)
                ])
            ),
            (
                "/foo/bar:~/bin/bazz:$HOME/fnort/bar:${VAR}/some/path",
                Paths::new(vec![
                    Path::with_source("/foo/bar", PathFlags::default(), source.clone()),
                    Path::with_source("~/bin/bazz", PathFlags::default(), source.clone()),
                    Path::with_source("$HOME/fnort/bar", PathFlags::default(), source.clone()),
                    Path::with_source("${VAR}/some/path", PathFlags::default(), source.clone())
                ])
            ),
        ];

        for (path, paths) in test_vec {
            assert_eq!(Paths::from_path(path), paths);
        }
    }

    #[test]
    fn test_merge() {
        let test_vec: Vec<(Paths, Paths, Paths)> = vec![
            (Paths::default(), Paths::default(), Paths::default()),
            (
                vec!["/foo/bar", "/bar/bazz"].into(),
                vec!["/fnort"].into(),
                vec!["/fnort", "/foo/bar", "/bar/bazz"].into(),
                )
        ];

        for (a, b, out) in test_vec {
            assert_eq!(a.merge(b), out)
        }
    }
}
