/* Copyright (C) 2020 Dylan Staatz - All Rights Reserved. */

use std::marker::PhantomData;

use nalgebra::storage::Storage;
use nalgebra::{Const, SVector, SimdRealField, Vector};
use serde::{de::DeserializeOwned, Deserialize, Serialize};

use super::Trajectory;

/// A trajectory where that represents a straight line between two points in euclidean space
///
/// Has zero size
#[derive(Debug, Clone, Copy, Serialize, Deserialize)]
#[serde(bound(
  serialize = "X: Serialize",
  deserialize = "X: DeserializeOwned"
))]
pub struct EuclideanTrajectory<X, const N: usize> {
  phantom_data: PhantomData<X>,
}

impl<X, const N: usize> EuclideanTrajectory<X, N> {
  pub fn new() -> Self {
    Self {
      phantom_data: PhantomData,
    }
  }
}

impl<X, const N: usize> Trajectory<X, N> for EuclideanTrajectory<X, N>
where
  X: SimdRealField,
{
  fn cost(&self) -> Option<X> {
    None
  }

  fn calc_cost<S1, S2>(
    &self,
    start: &Vector<X, Const<N>, S1>,
    end: &Vector<X, Const<N>, S2>,
  ) -> X
  where
    S1: Storage<X, Const<N>>,
    S2: Storage<X, Const<N>>,
  {
    start.metric_distance(end)
  }

  fn interpolate<S1, S2>(
    &self,
    start: &Vector<X, Const<N>, S1>,
    end: &Vector<X, Const<N>, S2>,
    t: X,
  ) -> SVector<X, N>
  where
    S1: Storage<X, Const<N>>,
    S2: Storage<X, Const<N>>,
  {
    start.lerp(end, t)
  }
}
