/* Copyright (C) 2020 Dylan Staatz - All Rights Reserved. */

use nalgebra::SVector;
use serde::{de::DeserializeOwned, Deserialize, Serialize};

use crate::scalar::Scalar;

use super::Behavior;

#[derive(Clone, Debug, PartialEq, Serialize, Deserialize)]
#[serde(bound(
  serialize = "X: Serialize",
  deserialize = "X: DeserializeOwned"
))]
pub struct Obstacle2dParams<X: Scalar> {
  pub shape: Shape2dParams<X>,
  #[serde(default)]
  pub offset: Offset2dParams<X>,
  #[serde(default)]
  pub behavior: Behavior,
}

#[derive(Copy, Clone, Debug, PartialEq, Serialize, Deserialize)]
#[serde(bound(
  serialize = "X: Serialize",
  deserialize = "X: DeserializeOwned"
))]
pub struct Offset2dParams<X: Scalar> {
  pub translation: SVector<X, 2>,
  pub angle: X,
}

impl<X: Scalar> Default for Offset2dParams<X> {
  fn default() -> Self {
    Self {
      translation: SVector::zeros(),
      angle: X::zero(),
    }
  }
}

#[derive(Clone, Debug, PartialEq, Serialize, Deserialize)]
#[serde(bound(
  serialize = "X: Serialize",
  deserialize = "X: DeserializeOwned"
))]
pub enum Shape2dParams<X: Scalar> {
  Ball {
    radius: X,
  },
  Capsule {
    a: SVector<X, 2>,
    b: SVector<X, 2>,
    radius: X,
  },
  Compound {
    shapes: Vec<(Offset2dParams<X>, Shape2dParams<X>)>,
  },
  RegularPolygon {
    sides: u8,
    scale: X,
  },
  ConvexPolygon {
    points: Vec<SVector<X, 2>>,
  },
  Cuboid {
    half_extents: SVector<X, 2>,
  },
  HalfSpace {
    normal: SVector<X, 2>,
  },
  RoundRegularPolygon {
    sides: u8,
    scale: X,
    border_radius: X,
  },
  RoundConvexPolygon {
    points: Vec<SVector<X, 2>>,
    border_radius: X,
  },
  RoundCuboid {
    half_extents: SVector<X, 2>,
    border_radius: X,
  },
  RoundTriangle {
    a: SVector<X, 2>,
    b: SVector<X, 2>,
    c: SVector<X, 2>,
    border_radius: X,
  },
  Segment {
    a: SVector<X, 2>,
    b: SVector<X, 2>,
  },
  Triangle {
    a: SVector<X, 2>,
    b: SVector<X, 2>,
    c: SVector<X, 2>,
  },
}

#[derive(Clone, Debug, PartialEq, Serialize, Deserialize)]
#[serde(bound(
  serialize = "X: Serialize",
  deserialize = "X: DeserializeOwned"
))]
pub struct Obstacle3dParams<X: Scalar> {
  pub shape: Shape3dParams<X>,
  #[serde(default)]
  pub offset: Offset3dParams<X>,
  #[serde(default)]
  pub behavior: Behavior,
}

#[derive(Copy, Clone, Debug, PartialEq, Serialize, Deserialize)]
#[serde(bound(
  serialize = "X: Serialize",
  deserialize = "X: DeserializeOwned"
))]
pub struct Offset3dParams<X: Scalar> {
  pub translation: SVector<X, 3>,
  pub axis_angle: SVector<X, 3>,
}

impl<X: Scalar> Default for Offset3dParams<X> {
  fn default() -> Self {
    Self {
      translation: SVector::zeros(),
      axis_angle: SVector::zeros(),
    }
  }
}

#[derive(Clone, Debug, PartialEq, Serialize, Deserialize)]
#[serde(bound(
  serialize = "X: Serialize",
  deserialize = "X: DeserializeOwned"
))]
pub enum Shape3dParams<X: Scalar> {
  Ball {
    radius: X,
  },
  Capsule {
    a: SVector<X, 3>,
    b: SVector<X, 3>,
    radius: X,
  },
  Compound {
    shapes: Vec<(Offset3dParams<X>, Shape3dParams<X>)>,
  },
  ConvexPolyhedron {
    points: Vec<SVector<X, 3>>,
  },
  Cone {
    half_height: X,
    radius: X,
  },
  Cuboid {
    half_extents: SVector<X, 3>,
  },
  Cylinder {
    half_height: X,
    radius: X,
  },
  HalfSpace {
    normal: SVector<X, 3>,
  },
  RoundCone {
    half_height: X,
    radius: X,
    border_radius: X,
  },
  RoundConvexPolyhedron {
    points: Vec<SVector<X, 3>>,
    border_radius: X,
  },
  RoundCuboid {
    half_extents: SVector<X, 3>,
    border_radius: X,
  },
  RoundCylinder {
    half_height: X,
    radius: X,
    border_radius: X,
  },
  RoundTriangle {
    a: SVector<X, 3>,
    b: SVector<X, 3>,
    c: SVector<X, 3>,
    border_radius: X,
  },
  Segment {
    a: SVector<X, 3>,
    b: SVector<X, 3>,
  },
  Triangle {
    a: SVector<X, 3>,
    b: SVector<X, 3>,
    c: SVector<X, 3>,
  },
}
