/* Copyright (C) 2020 Dylan Staatz - All Rights Reserved. */

use std::error::Error as StdError;
use std::path::PathBuf;

use clap::{
  app_from_crate, crate_authors, crate_description, crate_name, crate_version,
  value_t, Arg,
};

use path_planning::algorithms::{run_and_save, Algo};
use path_planning::run::Settings;

fn main() -> Result<(), Box<dyn StdError>> {
  let matches = app_from_crate!()
    .arg(
      Arg::with_name("verbose")
        .short("v")
        .help("Print out status information to stdout"),
    )
    .arg(
      Arg::with_name("algorithm")
        .required(true)
        .value_name("ALGORITHM")
        .long_help(&Algo::long_help()),
    )
    .arg(
      Arg::with_name("name")
        .short("n")
        .long("name")
        .takes_value(true)
        .value_name("NAME")
        .help("Name to associated with the run")
    )
    .arg(
      Arg::with_name("parameter_file")
        .required(true)
        .value_name("PARAMETER_FILE")
        .help("Path to parameters file"),
    )
    .arg(
      Arg::with_name("max_iterations")
        .short("i")
        .long("iterations")
        .takes_value(true)
        .value_name("MAX_ITERATIONS")
        .help("Limit the run to this many iterations")
    )
    .arg(
      Arg::with_name("keep_in_memory")
        .short("m")
        .help("Set to keep a copy of the states in memory")
    )
    .arg(
      Arg::with_name("double")
        .short("d")
        .short("double")
        .help("Set to use double precision floating point numbers (f64)")
    )
    .arg(
      Arg::with_name("output_dir")
        .required(true)
        .value_name("OUTPUT_DIR")
        .help("Directory to record output to"),
    )
    .arg(
      Arg::with_name("filename_prefix")
        .long("prefix")
        .value_name("FILENAME_PREFIX")
        .takes_value(true)
        .default_value("frame_")
        .long_help("The prefix to append to data filenames when saving to the data directory. Has not effect if OUTPUT_DIR is not set.")
    )
    .arg(
      Arg::with_name("filename_suffix")
        .long("suffix")
        .value_name("FILENAME_SUFFIX")
        .takes_value(true)
        .default_value(".bincode")
        .long_help("The suffix to append to data filenames when saving to the data directory. Has not effect if OUTPUT_DIR is not set.")
    )
    .get_matches();

  let verbose = matches.is_present("verbose");

  let algorithm =
    value_t!(matches, "algorithm", String).unwrap_or_else(|e| e.exit());

  let name = matches.value_of("name").map(|s| s.to_string());

  let parameter_file =
    value_t!(matches, "parameter_file", PathBuf).unwrap_or_else(|e| e.exit());

  let max_iterations = if matches.is_present("max_iterations") {
    Some(
      value_t!(matches, "max_iterations", usize).unwrap_or_else(|e| e.exit()),
    )
  } else {
    None
  };

  let keep_in_memory = matches.is_present("keep_in_memory");
  let double = matches.is_present("double");

  let output_dir =
    Some(value_t!(matches, "output_dir", PathBuf).unwrap_or_else(|e| e.exit()));

  let filename_prefix = Some(
    value_t!(matches, "filename_prefix", String).unwrap_or_else(|e| e.exit()),
  );

  let filename_suffix = Some(
    value_t!(matches, "filename_suffix", String).unwrap_or_else(|e| e.exit()),
  );

  let settings = Settings {
    verbose,
    algorithm,
    name,
    parameter_file,
    max_iterations,
    keep_in_memory,
    double,
    output_dir,
    filename_prefix,
    filename_suffix,
  };

  run_and_save(settings)?;

  Ok(())
}
