// Copyright 2021 Ian Jackson and contributors
// SPDX-License-Identifier: GPL-3.0-or-later
// There is NO WARRANTY.

use super::*;

#[proc_macro_error(allow_not_macro)]
pub fn derive(input: proc_macro::TokenStream) -> proc_macro::TokenStream {
  let input = parse_macro_input!(input as DeriveInput);
  let vis = &input.vis;
  let name = &input.ident;

  let us = format_ident!("partial_borrow");

  let fields = match &input.data {
    syn::Data::Struct(syn::DataStruct {
      fields: syn::Fields::Named(n), ..
    }) => &n.named,
    _ => abort_call_site!("PartialBorrow can only be derived for structs"),
  };

  let name_parts = format_ident!("{}__Partial", &name);
  let name_mod = format_ident!("{}__", &name);

  let mut in_lifetimes = vec![]; let mut in_lifetimes_act = vec![];
  let mut in_generics  = vec![]; let mut in_generics_act  = vec![];
  for g in input.generics.params {
    use syn::GenericParam as GP;
    match g {
      GP::Lifetime(t) => {
        let p = &t.lifetime;
        let p = quote_spanned!{p.span()=> #p };
        in_lifetimes    .push(p.clone());
        in_lifetimes_act.push(p);
      },
      GP::Type(t) => {
        let p = &t.ident;
        let p = quote_spanned!{p.span()=> #p };
        in_generics    .push(p.clone());
        in_generics_act.push(p);
      },
      GP::Const(t) => {
        let p = &t.ident;
        let ty = &t.ty;
        in_generics    .push(quote_spanned!{p.span()=> const #p: #ty });
        in_generics_act.push(quote_spanned!{p.span()=> #p});
      },
    }
  }

  let some_all = |l,g| {
    let a = chain!(l,g).collect_vec();
    let a = quote!{ #(#a,)* };
    a
  };

  let all_in_generics_act = some_all( &in_lifetimes_act, &in_generics_act );

  let ilts  = quote!{ #(#in_lifetimes    ,)* };
  let ilta  = quote!{ #(#in_lifetimes_act,)* };
  let igens = quote!{ #(#in_generics     ,)* };
  let igena = quote!{ #(#in_generics_act ,)* };
  let whole = quote!{ #name< #all_in_generics_act > };

  use format_ident as fi;

  #[allow(non_snake_case)]
  struct MutOrConst {
    Deref_   : syn::Ident,
    IsRef_   : syn::Ident,
    as_ref_  : syn::Ident,
    AsRef_   : syn::Ident,
    const_   : TokenStream2,
    mut_sfx  : &'static str,
    mut_     : Option<TokenStream2>,
  }

  macro_rules! mut_or_const { { $($local:ident)* ;
    $($x:tt)*
  } => {
    for m in [false,true] {
      // This business with the MutOrConst struct arranges to get the
      // members in scope as local variables, while ensuring that they
      // are each bound to the right thing.  Ideally this would be
      // unhygienic but that is impossible without a proc macro...
      #[allow(non_snake_case)]
      #[allow(unused_variables)]
      let MutOrConst { $( $local, )* .. } = MutOrConst {
        Deref_  : if m { fi!("DerefMut"   )} else { fi!("Deref")  },
        mut_sfx : if m {     "_mut"        } else {     ""        },
        IsRef_  : if m { fi!("IsMut")      } else { fi!("IsRef")  },
        as_ref_ : if m { fi!("as_mut")     } else { fi!("as_ref") },
        AsRef_  : if m { fi!("AsMut")      } else { fi!("AsRef")  },
        const_  : if m { quote!(mut)       } else { quote!(const) },
        mut_    : if m { Some(quote!(mut)) } else { None          },
      };
      $($x)*
    }
  } }

  let mut parts_fields = vec![];
  let mut mod_fields = vec![];
  let mut p_generics = vec![];
  let mut r_generics = vec![];
  let mut s_generics = vec![];
  for f in fields.iter() {
    let fty = &f.ty;

    let fname = f.ident.as_ref().unwrap();
    let permit_generic = format_ident!("_P_{}", fname);
    let fref = format_ident!("F_{}", fname);

    let fref_generics = quote!{ <#ilts _P, _T, #igens> };
    let fref_generics_act = quote!{ <#ilta _P, _T, #igena> };

    mod_fields.push(quote!{
      #[repr(C)]
      pub struct #fref #fref_generics {
        p: _P,
        t: PhantomData<_T>,
        s: PhantomData< *const super::#whole >,
      }
    });

    mut_or_const!{
      Deref_ IsRef_ as_ref_ AsRef_ const_ mut_sfx mut_;

      let target_= if mut_.is_some() { None }
                   else { Some(quote!{ type Target = _T; }) }.into_iter();

      let mc_deref_f = format_ident!("deref{}", mut_sfx);

      mod_fields.push(quote!{
        impl #fref_generics #Deref_ for #fref #fref_generics_act
        where _P : #IsRef_
        {
          #(#target_)*
          fn #mc_deref_f(& #mut_ self) -> &#mut_ _T where _T: Sized {
            let p: * #const_ Self = self;
            let offset = offset_of!(
              super::#whole,
              #fname
            );
            let p: * #const_ u8 = p as _;
            let p = unsafe { p.add(offset) };
            let p: * #const_ _T = p as _;
            let p: & #mut_ _T = unsafe { p.#as_ref_().unwrap() };
            p
          }
        }
      });
    }

    parts_fields.push(syn::Field {
      attrs:       default(),
      vis:         f.vis.clone(),
      ident:       f.ident.clone(),
      colon_token: f.colon_token.clone(),
      ty: {
        let mut ty = default();
        f.ty.to_tokens(&mut ty);
        let fref = quote!{
          #name_mod::#fref<
            #ilta
            #permit_generic,
            #fty,
            #igena
          > };
//        eprintln!("{}", &fref);
        syn::parse2(fref).expect("internal error in PartialBorrow derive")
      },
    });

    p_generics.push(permit_generic);
    r_generics.push(format_ident!("_R_{}", fname));
    s_generics.push(format_ident!("_S_{}", fname));
  }

  let parts_p = quote!{ #name_parts<#ilta #(#p_generics,)* #igena> };
  let parts_r = quote!{ #name_parts<#ilta #(#r_generics,)* #igena> };
  let parts_s = quote!{ #name_parts<#ilta #(#s_generics,)* #igena> };

  let def_defs = PERMITS.iter().map(|def| {
    let def_p = format_ident!("All_{}", def);
    let def_def = format_ident!("{}", def);
    let defs = fields.iter().map(
      |_| quote!{ #us::perms::#def_def }
    );
    let q = quote!{
      type #def_p = #name_parts< #ilta #( #defs, )* #igena >;
    };
    q
  });

  let fields_fields = {
    let idents = fields.iter().map(|f| f.ident.as_ref().unwrap());
    let numbers = 0..fields.len();
    let idents2 = idents.clone();
    quote!{
      pub struct                 Fields { #( pub #idents : usize,    )* }
      pub const FIELDS: Fields = Fields { #(     #idents2: #numbers, )* };
    }
  };

  let mut field_adj_defs = vec![];
  for (i,_f) in fields.iter().enumerate() {
    let n_gen = format_ident!("_N");

    let new_perms = p_generics.iter().enumerate().map(|(j,p)| {
      if i == j { &n_gen } else { &p }
    });
    field_adj_defs.push(quote!{
      impl <#ilts #(#p_generics,)* #n_gen, #igens>
        Adjust<_N, #i> for super::#parts_p {
          type Adjusted = super::#name_parts< #ilta #(#new_perms,)* #igena >;
        }
    });
/*
    for permit in &PERMITS {
      let now = format_ident!("{}", permit);
      let aty = format_ident!("_{}_{}", permit, &fname);
      ftrait_defs.push(quote!{ type #aty; });
      ftrait_impls.push(quote!{
        type #aty = 
      });
    }*/
  }

  let mut mod_conversions = vec![];
  {
    for input_parts in [true,false] {
      let p_mut = format_ident!("Mut");
      let ci_ps_none = vec![];
      let (ci_ps, ci_pa) = if input_parts {
        (&p_generics,
         p_generics.iter().collect_vec())
      } else {
        (&ci_ps_none,
         iter::repeat(&p_mut).take(p_generics.len()).collect_vec())
      };
      let ci_ty = if input_parts { &parts_p } else { &whole };

      // DOWNGRADE
      {
        let mut fns = vec![];
        mut_or_const!{
          Deref_ IsRef_ as_ref_ AsRef_ const_ mut_sfx mut_;

          let mc_downgrade = format_ident!("downgrade{}", mut_sfx);

          // impl AsRef / AsMut
          mod_conversions.push(quote!{
            impl <#ilts #(#r_generics,)* #(#ci_ps,)* #igens>
              #AsRef_<super::#parts_r>
              for super::#ci_ty
            where #( #r_generics: IsDowngradeFrom<#ci_pa> ),*
            {
              fn #as_ref_(&#mut_ self) -> &#mut_ super::#parts_r {
                Downgrade::#mc_downgrade(self)
              }
            }
          });
          // fn downgrade() **UNSAFE**
          fns.push(quote!{
            fn #mc_downgrade(input: &#mut_ Self) -> &#mut_ super::#parts_r
            { unsafe { mem::transmute(input) } }
          });
        }
        // impl Downgrade **UNSAFE**
        mod_conversions.push(quote!{
          impl <#ilts #(#r_generics,)* #(#ci_ps,)* #igens>
            Downgrade<super::#parts_r>
            for super::#ci_ty
          where #( #r_generics: IsDowngradeFrom<#ci_pa> ),*
          { #(#fns)* }
        });
      }

      // SPLIT OFF
      {
        let mut fns = vec![];
        mut_or_const!{
          Deref_ IsRef_ as_ref_ AsRef_ const_ mut_sfx mut_;
          let mc_split_off = format_ident!("split_off{}", mut_sfx);

          // fn split_off() **UNSAFE**
          fns.push(quote!{
            fn #mc_split_off<'__r>(input: &'__r #mut_ super::#ci_ty)
                              -> (&'__r #mut_ super::#parts_r,
                                  &'__r #mut_ Self::Remaining)
            {
              unsafe {
                let input = input as *#const_ _;
                (mem::transmute(input), mem::transmute(input))
              }
            }
          });
        }
        // impl SplitOff
        mod_conversions.push(quote!{
          impl <#ilts #(#r_generics,)* #(#ci_ps,)* #igens>
            SplitOff<super::#parts_r>
            for super::#ci_ty
          where #( #r_generics: IsDowngradeFrom<#ci_pa> ),* {
            type Remaining = super::#name_parts <
              #ilta
              #( <#r_generics as IsDowngradeFrom<#ci_pa>>::Remaining ,)*
              #igena
            >;
            #(#fns)*
          }
        });
      }

      // SPLIT INTO
      {
        let mut fns = vec![];
        mut_or_const!{
          Deref_ IsRef_ as_ref_ AsRef_ const_ mut_sfx mut_;

          let mc_split_into = format_ident!("split_into{}", mut_sfx);

          // impl From<R,S> for P
          mod_conversions.push(quote!{
            impl <#ilts '__r, #(#r_generics,)* #(#s_generics,)* #(#ci_ps,)* #igens>
              From<&'__r #mut_ super::#ci_ty>
              for (&'__r #mut_ super::#parts_r, &'__r#mut_ super::#parts_s)
            where #( #ci_pa: CanSplitInto<#r_generics, #s_generics> ),*
            {
              fn from(input: &'__r #mut_ super::#ci_ty) -> Self {
                SplitInto::#mc_split_into(input)
              }
            }
          });

          // fn split_into() **UNSAFE**
          fns.push(quote!{
            fn #mc_split_into<'__r>(input: &'__r #mut_ super::#ci_ty)
                              -> (&'__r #mut_ super::#parts_r,
                                  &'__r #mut_ super::#parts_s)
            {
              unsafe {
                let input = input as *#const_ _;
                (mem::transmute(input), mem::transmute(input))
              }
            }
          });
        }
        // impl SplitInto
        mod_conversions.push(quote!{
          impl <#ilts #(#r_generics,)* #(#s_generics,)* #(#ci_ps,)* #igens>
            SplitInto<super::#parts_r, super::#parts_s>
            for super::#ci_ty
          where #( #ci_pa: CanSplitInto<#r_generics, #s_generics> ),*
          { #(#fns)* }
        });
      }
    }
  }
  //eprintln!("{}", &mod_conversions);

/*
    };
  }).collect_vec();
*/

  let output = quote!{
    #[allow(dead_code)] #[allow(non_camel_case_types)]
    #[repr(C)]
    #vis struct #name_parts <#ilts #(#p_generics,)* #igens> {
      #( #parts_fields ),*
    }
    impl <#ilts #igens> #us::PartialBorrow for #whole {
      #( #def_defs )*
      type Fields = #name_mod::Fields;
      const FIELDS: Self::Fields = #name_mod::FIELDS;
    }

    #[allow(dead_code)] #[allow(non_camel_case_types)] #[allow(non_snake_case)]
    #vis mod #name_mod {
      use ::std::marker::PhantomData;
      use ::std::mem;
      use ::std::ops::{Deref, DerefMut};

      use super::#us::memoffset::offset_of;
      use super::#us::perms::*;
      use super::#us::{Downgrade, SplitInto, SplitOff};

      #( #mod_fields )*
      #( #mod_conversions )*
      #fields_fields
      #( #field_adj_defs )*
    }
  };

  let ev = env::var_os(DEBUG_ENVVAR);
  let ev = ev.as_ref().map(|ev| ev.to_str().expect(DEBUG_ENVVAR));

  if let Some(how) = {
    if let Some(ev) = ev {
      if ev == "0" { None }
      else if let Some(rest) = ev.strip_prefix("<") { Some(Left(rest)) }
      else if let Some(rest) = ev.strip_prefix(">") { Some(Right(rest)) }
      else if ev == "1" { Some(Right(STDERR)) }
      else { panic!("did not understand {}", DEBUG_ENVVAR) }
    } else { None }
  } {

    let mut tf = tempfile::tempfile().unwrap();
    write!(tf, "{}", &output).unwrap();
    tf.rewind().unwrap();
    let mut tf2 = tempfile::tempfile().unwrap();
    Command::new("rustfmt")
      .args(&["--config","tab_spaces=2,max_width=80,fn_call_width=80"])
      .stdin(tf)
      .stdout(tf2.try_clone().unwrap())
      .status().unwrap()
      .exit_ok().unwrap();
    tf2.rewind().unwrap();
    let mut nl_done = true;

    let output = match how {
      Left(_) => tempfile::tempfile().unwrap(),
      Right(from) => File::create(from).expect(from),
    };
    let mut output = BufWriter::new(output);

    let divvy = if how == Right(STDERR) { "\n------\n\n" } else { "" };
    write!(output, "{}", divvy).unwrap();

    for l in BufReader::new(tf2).lines() {
      let l = l.unwrap();
      let want_nl = {
        let l = l.trim_start();
        l.strip_prefix("pub ").unwrap_or(l);
        "# impl struct fn".split(' ').any(|s| l.starts_with(s))
      };
      if want_nl && !nl_done { writeln!(output,"").unwrap(); nl_done=true; }
      else if !want_nl { nl_done=false; }
      if l.trim_end().ends_with("{") { nl_done=true; }

      writeln!(output, "{}", l).unwrap();
    }
    write!(output, "{}", divvy).unwrap();
    let mut output = output.into_inner().unwrap();

    if let Left(reference) = how {
      output.rewind().unwrap();
      eprintln!("{}: checking output against {}", DEBUG_ENVVAR, reference);
      Command::new("diff")
        .args(&["-u","--",reference,"-"])
        .stdin(output)
        .stdout(File::create(STDERR).expect(STDERR))
        .status().unwrap()
        .exit_ok().expect("macro ouptut changed!");
    }
  }

  proc_macro::TokenStream::from(output)
}
