/*
Copyright (C) 2021 Kunal Mehta <legoktm@debian.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

use kuchiki::{Attribute, ExpandedName, NodeRef};
use serde::{Deserialize, Serialize};
use std::fmt;

#[derive(Deserialize, Serialize)]
pub(crate) struct IncludeOnlyDataMw {
    pub(crate) src: String,
}

#[derive(Clone, Debug)]
pub struct NoInclude {
    start: NodeRef,
    end: NodeRef,
    siblings: Vec<NodeRef>,
}

impl NoInclude {
    const TYPEOF_START: &'static str = "mw:Includes/NoInclude";
    pub(crate) const TYPEOF_END: &'static str = "mw:Includes/NoInclude/End";
    pub(crate) const SELECTOR: &'static str =
        "[typeof=\"mw:Includes/NoInclude\"]";

    pub fn new(contents: &NodeRef) -> Self {
        Self {
            start: NodeRef::new_element(
                crate::build_qual_name(local_name!("meta")),
                vec![(
                    ExpandedName::new(ns!(), "typeof"),
                    Attribute {
                        prefix: None,
                        value: Self::TYPEOF_START.to_string(),
                    },
                )],
            ),
            end: NodeRef::new_element(
                crate::build_qual_name(local_name!("meta")),
                vec![(
                    ExpandedName::new(ns!(), "typeof"),
                    Attribute {
                        prefix: None,
                        value: Self::TYPEOF_END.to_string(),
                    },
                )],
            ),
            siblings: vec![contents.clone()],
        }
    }

    pub(crate) fn new_from_node(
        start: &NodeRef,
        end: &NodeRef,
        siblings: &[NodeRef],
    ) -> Self {
        if start.as_element().is_none() {
            unreachable!("Non-element start node passed");
        } else if end.as_element().is_none() {
            unreachable!("Non-element end node passed");
        }
        Self {
            start: start.clone(),
            end: end.clone(),
            siblings: siblings.to_vec(),
        }
    }

    pub fn as_nodes(&self) -> Vec<NodeRef> {
        let mut nodes = vec![self.start.clone()];
        for node in &self.siblings {
            nodes.push(node.clone());
        }
        nodes.push(self.end.clone());
        nodes
    }

    /// Remove this template from the document
    pub fn detach(&self) {
        for node in self.as_nodes() {
            node.detach();
        }
    }

    /// Prepend this template into a node.
    /// Effectively calling `node.prepend(template)`
    pub fn prepend_on(&self, code: &NodeRef) {
        for node in self.as_nodes().iter().rev() {
            code.prepend(node.clone());
        }
    }

    /// Append this template into a node.
    /// Effectively calling `node.append(template)`
    pub fn append_on(&self, code: &NodeRef) {
        for node in self.as_nodes() {
            code.append(node);
        }
    }

    /// Insert this template after the node
    /// Effectively calling `node.insert_after(template)`
    pub fn insert_after_on(&self, code: &NodeRef) {
        // First add the siblings in reverse order
        for node in self.as_nodes().iter().rev() {
            code.insert_after(node.clone());
        }
    }

    /// Insert this template before the node
    /// Effectively calling `node.insert_before(template)`
    pub fn insert_before_on(&self, code: &NodeRef) {
        for node in self.as_nodes() {
            code.insert_before(node);
        }
    }
}

impl fmt::Display for NoInclude {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        let nodes: Vec<_> = self
            .as_nodes()
            .iter()
            .map(|node| node.to_string())
            .collect();
        write!(f, "{}", nodes.join(""))
    }
}
