#-------------------------------------------------------------------------------
# Copyright (c) 2020, Arm Limited and Contributors. All rights reserved.
#
# SPDX-License-Identifier: BSD-3-Clause
#
#-------------------------------------------------------------------------------

#[===[.rst:
  Abstraction to interface to SP DEV KIT.
  ---------------------------------------

  SP DEV KIT is a component generated by the OP-TEE OS build and defines the
  FF-A SP interface for the OS to SPs.

  This file provides the following functionality:
	- creates a static library to easy building the SP side component of the
	  SP DEV KIT. This library will be properly configured with transitive
	  dependencies to apply the needed compiler configurations to the
	  target linking against it.
	- defines a function to allow configuration of the linking phase of the
	  executable target. (See: :command:`sp_dev_kit_configure_linking`)

  Inputs:
  ``SP_DEV_KIT_DIR``
  A cmake or environment variable specifying the location of the SP DEV KIT.
  This is mandatory.

  ``SP_DEV_KIT_INC_DIR``
  The SP DEV KIT depends on some header files living in the SP. This variable
  must be set to an include path giving access to these headers.

  Outputs:
  :command:sp_dev_kit_configure_linking()

  :variable:`SP_DEV_KIT_LIBRARIES`

#]===]

# Store SP DEV KIT location to cache.
# If a cmake variable exist, use it as is.
# If not, try to copy over the value from the environment.
if(NOT DEFINED SP_DEV_KIT_DIR AND NOT DEFINED ENV{SP_DEV_KIT_DIR})
	message(FATAL_ERROR "'SP_DEV_KIT_DIR' is not defined.")
endif()
set(SP_DEV_KIT_DIR $ENV{SP_DEV_KIT_DIR} CACHE STRING "SP dev kit from original OP-TEE build system")

if (DEFINED ENV{SP_DEV_KIT_DIR} AND NOT SP_DEV_KIT_DIR STREQUAL "$ENV{SP_DEV_KIT_DIR}")
	message(WARNING "Suspicious settings: the value of SP_DEV_KIT_DIR in the environment is not matching cmakes settings!")
endif()

# Find the directories inside SP DEV KIT. This gives more flexibility than when using static settings.
find_path (SP_DEV_KIT_SRC_DIR
	NAMES
		sp_header.c
	PATHS
		${SP_DEV_KIT_DIR}
	PATH_SUFFIXES
		"src"
	NO_DEFAULT_PATH
	REQUIRED
	DOC
		"SP DEV KIT source directory"
	)

find_path (SP_DEV_KIT_INCLUDE_DIR
	NAMES
		ffa.h atomic.h compiler.h
	PATHS
		${SP_DEV_KIT_DIR}
	PATH_SUFFIXES
		"include"
	NO_DEFAULT_PATH
	REQUIRED
	DOC
		"SP DEV KIT include directory"
	)

find_path (SP_DEV_KIT_LIB_DIR
	NAMES
		${CMAKE_STATIC_LIBRARY_PREFIX}utils${CMAKE_STATIC_LIBRARY_SUFFIX} libutils.a utils.a utils.lib libutils.link_libraries
	PATHS
		${SP_DEV_KIT_DIR}
	PATH_SUFFIXES
		"lib"
	NO_DEFAULT_PATH
	REQUIRED
	DOC
		"SP DEV KIT library directory"
	)

#[===[.rst:
.. cmake:variable:: SP_DEV_KIT_LIBRARIES

  List of libraries forming the SP DEV KIT interface.

#]===]

# Create an imported target for libutils.a
add_library(spdevkit::libutils STATIC IMPORTED)
set_target_properties(spdevkit::libutils PROPERTIES
	INTERFACE_INCLUDE_DIRECTORIES "${SP_DEV_KIT_INCLUDE_DIR}"
	INTERFACE_LINK_LIBRARIES "${SP_DEV_KIT_LIB_DIR}/libutils.a"
	IMPORTED_LOCATION "${SP_DEV_KIT_LIB_DIR}/libutils.a"
)
list(APPEND SP_DEV_KIT_LIBRARIES spdevkit::libutils)

if (NOT Spdevkit_FIND_COMPONENTS OR "SP_HEADER" IN_LIST Spdevkit_FIND_COMPONENTS)
	if (NOT DEFINED SP_DEV_KIT_INC_DIR)
		message(FATAL_ERROR "Mandatory input variable 'SP_DEV_KIT_INC_DIR' is not defined.")
	endif()

	# Define a static library to compile the SP side source files and to
	# capture dependencies (settings).
	add_library(sp_header STATIC
		${SP_DEV_KIT_SRC_DIR}/sp_assert.c
		${SP_DEV_KIT_SRC_DIR}/sp_entry.c
		${SP_DEV_KIT_SRC_DIR}/sp_header.c
		${SP_DEV_KIT_SRC_DIR}/sp_trace.c
		)

	target_include_directories(sp_header
							PRIVATE
								${SP_DEV_KIT_INC_DIR}
							)

	target_link_libraries(sp_header PRIVATE spdevkit::libutils)
	add_library(spdevkit::sp_header ALIAS sp_header)
	list(APPEND SP_DEV_KIT_LIBRARIES spdevkit::sp_header)
endif()

#[===[.rst:
.. cmake:command:: sp_dev_kit_configure_linking

  .. code-block:: cmake

    sp_dev_kit_configure_linking(TARGET <executable target> DEFINES <list of pre-processor macros>)

  Connect an executable target to SP DEV KIT link requirements:
	- add rules to run the linker command file trough the pre-processor if needed
	- configure the target to use the linker command file
	- link SP DEV KIT libraries to the target

  Inputs:

  ``TARGET``
  Mandatory. The name of an already defined executable target (add_executable())

  ``DEFINES``
  Optional. Macro definitions for the pre-processing step.

#]===]
function(sp_dev_kit_configure_linking)
	set(options  )
	set(oneValueArgs TARGET)
	set(multiValueArgs DEFINES)
	cmake_parse_arguments(MY_PARAMS "${options}" "${oneValueArgs}"
						"${multiValueArgs}" ${ARGN} )

	if(NOT DEFINED MY_PARAMS_TARGET)
		message(FATAL_ERROR "sp_dev_kit_configure_linking: mandatory parameter TARGET not defined!")
	endif()

	if(NOT DEFINED MY_PARAMS_DEFINES)
		set(MY_PARAMS_DEFINES "")
	endif()

	compiler_preprocess_file(
		SRC ${SP_DEV_KIT_DIR}/src/sp.ld.S
		DST ${CMAKE_BINARY_DIR}/sp.ld
		DEFINES ${MY_PARAMS_DEFINES}
	)

	add_custom_target(${MY_PARAMS_TARGET}-pplscript DEPENDS ${CMAKE_BINARY_DIR}/sp.ld)
	add_dependencies(${MY_PARAMS_TARGET} ${MY_PARAMS_TARGET}-pplscript)

	target_link_options(${MY_PARAMS_TARGET} PRIVATE
		-T${CMAKE_BINARY_DIR}/sp.ld
	)
endfunction()

