use crate::math::Real;
use crate::transformation::voxelization::FillMode;

/// Parameters controlling the VHACD convex decomposition.
///
/// See https://github.com/Unity-Technologies/VHACD#parameters for details.
#[derive(Debug, Clone, PartialEq)]
pub struct VHACDParameters {
    /// Maximum concavity.
    ///
    /// Default: 0.1 (in 2D), 0.01 (in 3D).
    /// Valid range `[0.0, 1.0]`.
    pub concavity: Real,
    /// Controls the bias toward clipping along symmetry planes.
    ///
    /// Default: 0.05.
    /// Valid Range: `[0.0, 1.0]`.
    pub alpha: Real,
    /// Controls the bias toward clipping along revolution planes.
    ///
    /// Default: 0.05.
    /// Valid Range: `[0.0, 1.0]`.
    pub beta: Real,
    /// Resolution used during the voxelization stage.
    ///
    /// Default: 256 (in 2D), 64 (in 3D).
    pub resolution: u32,
    /// Controls the granularity of the search for the best
    /// clipping plane during the decomposition.
    ///
    /// Default: 4
    pub plane_downsampling: u32,
    /// Controls the precision of the convex-hull generation
    /// process during the clipping plane selection stage.
    ///
    /// Default: 4
    pub convex_hull_downsampling: u32,
    /// Controls the way the input mesh or polyline is being
    /// voxelized.
    ///
    /// Default: `FillMode::FloodFill { detect_cavities: false, detect_self_intersections: false }`
    pub fill_mode: FillMode,
    /// Controls whether the convex-hull should be approximated during the decomposition stage.
    /// Setting this to `true` increases performances with a slight degradation of the decomposition
    /// quality.
    ///
    /// Default: true
    pub convex_hull_approximation: bool,
    /// Controls the max number of convex-hull generated by the convex decomposition.
    ///
    /// Default: 1024
    pub max_convex_hulls: u32,
}

impl Default for VHACDParameters {
    fn default() -> Self {
        Self {
            #[cfg(feature = "dim3")]
            resolution: 64,
            #[cfg(feature = "dim3")]
            concavity: 0.01,
            #[cfg(feature = "dim2")]
            resolution: 256,
            #[cfg(feature = "dim2")]
            concavity: 0.1,
            plane_downsampling: 4,
            convex_hull_downsampling: 4,
            alpha: 0.05,
            beta: 0.05,
            convex_hull_approximation: true,
            max_convex_hulls: 1024,
            fill_mode: FillMode::FloodFill {
                detect_cavities: false,
                #[cfg(feature = "dim2")]
                detect_self_intersections: false,
            },
        }
    }
}
