use std::sync::Arc;

/// List of features whose non-activation may cause a runtime error.
/// Used to indicate which lack of feature caused [`ParquetError::FeatureNotActive`].
#[derive(Debug, Clone, Copy, PartialEq, Eq, Hash)]
#[non_exhaustive]
pub enum Feature {
    Snappy,
    Brotli,
    Gzip,
    Lz4,
    Zstd,
}

/// Errors generated by this crate
#[derive(Debug, Clone)]
#[non_exhaustive]
pub enum ParquetError {
    /// General Parquet error.
    General(String),
    /// Error presented when trying to use a code branch that requires a feature.
    FeatureNotActive(Feature, String),
    /// When the parquet file is known to be out of spec.
    OutOfSpec(String),
    // An error originating from a consumer or dependency
    External(String, Arc<dyn std::error::Error + Send + Sync>),
}

impl std::error::Error for ParquetError {}

impl std::fmt::Display for ParquetError {
    fn fmt(&self, fmt: &mut std::fmt::Formatter) -> std::fmt::Result {
        match self {
            ParquetError::General(message) => {
                write!(fmt, "{}", message)
            }
            ParquetError::FeatureNotActive(feature, reason) => {
                write!(
                    fmt,
                    "The feature \"{:?}\" needs to be active to {}",
                    feature, reason
                )
            }
            ParquetError::OutOfSpec(message) => {
                write!(fmt, "{}", message)
            }
            ParquetError::External(message, err) => {
                write!(fmt, "{}: {}", message, err)
            }
        }
    }
}

#[cfg(feature = "snappy")]
impl From<snap::Error> for ParquetError {
    fn from(e: snap::Error) -> ParquetError {
        ParquetError::General(format!("underlying snap error: {}", e))
    }
}

impl From<parquet_format_async_temp::thrift::Error> for ParquetError {
    fn from(e: parquet_format_async_temp::thrift::Error) -> ParquetError {
        ParquetError::General(format!("underlying thrift error: {}", e))
    }
}

impl From<std::io::Error> for ParquetError {
    fn from(e: std::io::Error) -> ParquetError {
        ParquetError::General(format!("underlying IO error: {}", e))
    }
}

/// A specialized `Result` for Parquet errors.
pub type Result<T> = std::result::Result<T, ParquetError>;

macro_rules! general_err {
    ($fmt:expr) => (ParquetError::General($fmt.to_owned()));
    ($fmt:expr, $($args:expr),*) => (ParquetError::General(format!($fmt, $($args),*)));
    ($e:expr, $fmt:expr) => (ParquetError::General($fmt.to_owned(), $e));
    ($e:ident, $fmt:expr, $($args:tt),*) => (
        ParquetError::General(&format!($fmt, $($args),*), $e));
}
