use crate::parch::enc::{io::DecryptorStream, io::EncryptorStream, StreamableAead};
use brotli::{enc::BrotliEncoderParams, CompressorWriter, Decompressor};
use rand::{rngs::StdRng, RngCore, SeedableRng};
use std::{
    ffi::OsStr,
    fmt::Display,
    fs::{File, OpenOptions},
    io::{Error, Read, Seek, SeekFrom, Write},
    path::{Path, PathBuf},
};
use zeroize::Zeroizing;

const MEDIA_ROOT: &str = "./media/";

fn get_media_path<S: AsRef<OsStr> + ?Sized + Display>(file_name: &S) -> PathBuf {
    Path::new(MEDIA_ROOT).join(Path::new(file_name))
}

fn open_media_file<S: AsRef<OsStr> + ?Sized + Display>(
    file_name: &S,
    write: bool,
) -> Result<(File, PathBuf), Error> {
    let path = get_media_path(file_name);
    if !path.exists() {
        let _f = OpenOptions::new()
            .write(true)
            .create_new(true)
            .open(path.clone())?;
    }
    let mut oo = OpenOptions::new();
    oo.read(true);
    if write {
        oo.write(true).truncate(true);
    }
    let path_clone = path.clone();
    let f = oo.open(path)?;
    Ok((f, path_clone))
}

macro_rules! stream_test {
    (
        $data_bytes: literal,
        $write_block: literal,
        $read_block: literal,
        $seed: literal,
        $func_name: tt,
        $aead: ty,
        $initiate_encryption_func_name: tt,
        $initiate_decryption_func_name: tt,
        $brotli_params: expr$(,)?
    ) => {
        #[test]
        fn $func_name() {
            const DATA_BYTES: usize = $data_bytes;
            const WRITE_BLOCK: usize = $write_block;
            const READ_BLOCK: usize = $read_block;
            const LAST_WRITE_BLOCK: usize = DATA_BYTES % WRITE_BLOCK;
            const LAST_READ_BLOCK: usize = DATA_BYTES % READ_BLOCK;
            const NUM_WRITE_BLOCKS: usize =
                DATA_BYTES / WRITE_BLOCK + (LAST_WRITE_BLOCK > 0) as usize;
            const NUM_READ_BLOCKS: usize = DATA_BYTES / READ_BLOCK + (LAST_READ_BLOCK > 0) as usize;

            let mut data = [0u8; DATA_BYTES];
            let mut rng = StdRng::seed_from_u64($seed);
            rng.fill_bytes(&mut data);
            let data_clone = data.clone();

            let (mut f, _) = open_media_file("test", true).unwrap();
            let mut key = Zeroizing::new(Vec::new());
            let mut nonce = Zeroizing::new(Vec::new());

            let encryptor = <$aead>::$initiate_encryption_func_name(&mut rng, &mut key, &mut nonce);
            let (blocks_to_read, last_block_len) = {
                let mut es = EncryptorStream::new(encryptor, &mut f);
                let mut comp = CompressorWriter::with_params(es, WRITE_BLOCK, &$brotli_params);
                for i in 0..(NUM_WRITE_BLOCKS - 1) {
                    let d = &mut data[WRITE_BLOCK * i..WRITE_BLOCK * (i + 1)];
                    comp.write_all(d).unwrap();
                }
                let d = &mut data[WRITE_BLOCK * (NUM_WRITE_BLOCKS - 1)..];
                comp.write_all(d).unwrap();
                es = comp.into_inner();
                es.flush().unwrap();
                (es.get_total_blocks_written(), es.get_last_block_len())
            };

            f.seek(SeekFrom::Start(0)).unwrap();
            data.fill(0u8);

            let decryptor = <$aead>::$initiate_decryption_func_name(&key, &nonce);
            {
                let ds = DecryptorStream::new(decryptor, &mut f, blocks_to_read, last_block_len);
                let mut decomp = Decompressor::new(ds, READ_BLOCK);
                for i in 0..(NUM_READ_BLOCKS - 1) {
                    let d = &mut data[READ_BLOCK * i..READ_BLOCK * (i + 1)];
                    decomp.read_exact(d).unwrap();
                }
                let d = &mut data[READ_BLOCK * (NUM_READ_BLOCKS - 1)..];
                decomp.read_exact(d).unwrap();
            }

            assert_eq!(data, data_clone);
        }
    };
}

macro_rules! be_and_le_stream_test {
    (
        {
            $data_bytes: literal,
            $write_block: literal,
            $read_block: literal,
            $seed: literal,
            $brotli_params: expr$(,)?
        },
        $({ $be_func_name: tt, $le_func_name: tt, $aead: ty$(,)? }$(,)?)*
    ) => {$(
        stream_test!(
            $data_bytes,
            $write_block,
            $read_block,
            $seed,
            $be_func_name,
            $aead,
            initiate_be32_encryption_stream,
            initiate_be32_decryption_stream,
            $brotli_params,
        );
        stream_test!(
            $data_bytes,
            $write_block,
            $read_block,
            $seed,
            $le_func_name,
            $aead,
            initiate_le31_encryption_stream,
            initiate_le31_decryption_stream,
            $brotli_params,
        );
    )*};
}

be_and_le_stream_test!(
    { 84631, 2711, 2659, 0x22D154802315F2CD, BrotliEncoderParams::default() },
    { aes256gcmsiv_stream_be32_test, aes256gcmsiv_stream_le31_test, aes_gcm_siv::Aes256GcmSiv },
    { aes128gcmsiv_stream_be32_test, aes128gcmsiv_stream_le31_test, aes_gcm_siv::Aes128GcmSiv },
    { chacha20poly1305_stream_be32_test, chacha20poly1305_stream_le31_test, chacha20poly1305::ChaCha20Poly1305 },
    { chacha12poly1305_stream_be32_test, chacha12poly1305_stream_le31_test, chacha20poly1305::ChaCha12Poly1305 },
    { xchacha20poly1305_stream_be32_test, xchacha20poly1305_stream_le31_test, chacha20poly1305::XChaCha20Poly1305 },
    { xchacha12poly1305_stream_be32_test, xchacha12poly1305_stream_le31_test, chacha20poly1305::XChaCha12Poly1305 },
    { deoxysii256_stream_be32_test, deoxysii256_stream_le31_test, deoxys::DeoxysII256 },
    { deoxysii128_stream_be32_test, deoxysii128_stream_le31_test, deoxys::DeoxysII128 },
    { deoxysi256_stream_be32_test, deoxysi256_stream_le31_test, deoxys::DeoxysI256 },
    { deoxysi128_stream_be32_test, deoxysi128_stream_le31_test, deoxys::DeoxysI128 },
);
