use core::ops::Sub;
use crypto::aead::{
    consts::{U4, U5},
    generic_array::{typenum::Unsigned, ArrayLength, GenericArray},
    stream::{
        DecryptorBE32, DecryptorLE31, EncryptorBE32, EncryptorLE31, NewStream, Nonce, NonceSize,
        StreamBE32, StreamLE31, StreamPrimitive,
    },
    AeadCore, AeadInPlace, Key, NewAead,
};
use rand::{CryptoRng, RngCore};
use zeroize::Zeroizing;

// derived from https://stackoverflow.com/a/57578431/1526155.
macro_rules! enum_from_u16 {
    ($(#[$meta:meta])* $vis:vis enum $name:ident {
        $($(#[$vmeta:meta])* $vname:ident $(= $val:expr)?,)*
    }) => {
        $(#[$meta])*
        $vis enum $name {
            $($(#[$vmeta])* $vname $(= $val)?,)*
        }

        impl std::convert::TryFrom<u16> for $name {
            type Error = ();

            fn try_from(v: u16) -> Result<Self, Self::Error> {
                match v {
                    $(x if x == $name::$vname as u16 => Ok($name::$vname),)*
                    _ => Err(()),
                }
            }
        }
    }
}

enum_from_u16! {
    pub enum SupportedAeads {
        Aes256GcmSiv = 0,
        Aes128GcmSiv = 1,
        XChaCha20Poly1305 = 2,
        ChaCha20Poly1305 = 3,
        XChaCha12Poly1305 = 4,
        ChaCha12Poly1305 = 5,
        DeoxysII256 = 6,
        DeoxysII128 = 7,
        DeoxysI256 = 8,
        DeoxysI128 = 9,
    }
}

/// Wraps a key and nonce into `GenericArray`s
macro_rules! wrap_key_nonce {
    ($key: expr, $nonce: expr) => {
        (
            GenericArray::from_slice($key),
            GenericArray::from_slice($nonce),
        )
    };
}

/// Generates a random key and nonce using the specified rng and aead and stream types.
macro_rules! rand_key_nonce {
    ($rng: expr, $aead_ty: ty, $stream_ty: ty, $key_out: expr, $nonce_out: expr$(,)?) => {{
        type KeyLen = <$aead_ty as NewAead>::KeySize;
        let mut key_out = $key_out;
        key_out.resize(KeyLen::USIZE, 0u8);
        $rng.fill_bytes(&mut key_out);
        type NonceLen = NonceSize<$aead_ty, $stream_ty>;
        let mut nonce_out = $nonce_out;
        nonce_out.resize(NonceLen::USIZE, 0u8);
        $rng.fill_bytes(&mut nonce_out);
        let (key_ga, nonce_ga) = wrap_key_nonce!(key_out, nonce_out);
        (key_ga, nonce_ga)
    }};
}

/// Generic trait which adds support for creating an encryption stream and decryption stream to a given aead
/// implementation.
pub trait StreamableAead<A = Self> {
    /// Create a BE32 encryption stream for this `Aead`, generating a random and unique key and nonce. Fills `key_out`
    /// and `nonce_out` with the generated key and nonce respectively.
    fn initiate_be32_encryption_stream<R>(
        rng: &mut R,
        key_out: &mut Zeroizing<Vec<u8>>,
        nonce_out: &mut Zeroizing<Vec<u8>>,
    ) -> EncryptorBE32<A>
    where
        R: CryptoRng + RngCore,
        A: AeadInPlace,
        A::NonceSize: Sub<U5>,
        <<A as AeadCore>::NonceSize as Sub<U5>>::Output: ArrayLength<u8>;
    /// Creates a BE32 decryption stream for this `Aead`, using the given key and nonce for decryption.
    fn initiate_be32_decryption_stream(
        key: &Zeroizing<Vec<u8>>,
        nonce: &Zeroizing<Vec<u8>>,
    ) -> DecryptorBE32<A>
    where
        A: AeadInPlace,
        A::NonceSize: Sub<U5>,
        <<A as AeadCore>::NonceSize as Sub<U5>>::Output: ArrayLength<u8>;
    /// Create a LE31 encryption stream for this `Aead`, generating a random and unique key and nonce. Fills `key_out`
    /// and `nonce_out` with the generated key and nonce respectively.
    fn initiate_le31_encryption_stream<R>(
        rng: &mut R,
        key_out: &mut Zeroizing<Vec<u8>>,
        nonce_out: &mut Zeroizing<Vec<u8>>,
    ) -> EncryptorLE31<A>
    where
        R: CryptoRng + RngCore,
        A: AeadInPlace,
        A::NonceSize: Sub<U4>,
        <<A as AeadCore>::NonceSize as Sub<U4>>::Output: ArrayLength<u8>;
    /// Creates a LE31 decryption stream for this `Aead`, using the given key and nonce for decryption.
    fn initiate_le31_decryption_stream(
        key: &Zeroizing<Vec<u8>>,
        nonce: &Zeroizing<Vec<u8>>,
    ) -> DecryptorLE31<A>
    where
        A: AeadInPlace,
        A::NonceSize: Sub<U4>,
        <<A as AeadCore>::NonceSize as Sub<U4>>::Output: ArrayLength<u8>;
}

/// Implements `EncAlg` for a given aead implementation.
macro_rules! impl_encalg {
    ($($aead: ty$(,)?)+) => {
        $(
            impl StreamableAead for $aead {
                fn initiate_be32_encryption_stream<R: CryptoRng + RngCore>(
                    rng: &mut R,
                    key_out: &mut Zeroizing<Vec<u8>>,
                    nonce_out: &mut Zeroizing<Vec<u8>>,
                ) -> EncryptorBE32<$aead> {
                    let (key, nonce) = rand_key_nonce!(
                        rng,
                        $aead,
                        StreamBE32<$aead>,
                        key_out,
                        nonce_out,
                    );
                    let stream: StreamBE32<$aead> = create_stream(&key, &nonce);
                    stream.encryptor()
                }
                fn initiate_be32_decryption_stream(
                    key: &Zeroizing<Vec<u8>>,
                    nonce: &Zeroizing<Vec<u8>>
                ) -> DecryptorBE32<$aead> {
                    let (key, nonce) = wrap_key_nonce!(key, nonce);
                    let stream: StreamBE32<$aead> = create_stream(&key, &nonce);
                    stream.decryptor()
                }
                fn initiate_le31_encryption_stream<R: CryptoRng + RngCore>(
                    rng: &mut R,
                    key_out: &mut Zeroizing<Vec<u8>>,
                    nonce_out: &mut Zeroizing<Vec<u8>>,
                ) -> EncryptorLE31<$aead> {
                    let (key, nonce) = rand_key_nonce!(
                        rng,
                        $aead,
                        StreamLE31<$aead>,
                        key_out,
                        nonce_out,
                    );
                    let stream: StreamLE31<$aead> = create_stream(&key, &nonce);
                    stream.encryptor()
                }
                fn initiate_le31_decryption_stream(
                    key: &Zeroizing<Vec<u8>>,
                    nonce: &Zeroizing<Vec<u8>>
                ) -> DecryptorLE31<$aead> {
                    let (key, nonce) = wrap_key_nonce!(key, nonce);
                    let stream: StreamLE31<$aead> = create_stream(&key, &nonce);
                    stream.decryptor()
                }
            }
        )*
    };
}

// impl EncAlg
impl_encalg!(
    aes_gcm_siv::Aes256GcmSiv,
    aes_gcm_siv::Aes128GcmSiv,
    chacha20poly1305::XChaCha20Poly1305,
    chacha20poly1305::ChaCha20Poly1305,
    chacha20poly1305::XChaCha12Poly1305,
    chacha20poly1305::ChaCha12Poly1305,
    deoxys::DeoxysII256,
    deoxys::DeoxysII128,
    deoxys::DeoxysI256,
    deoxys::DeoxysI128,
);

/// Generic function to create an `aead::stream` Stream struct using the given key and nonce.
fn create_stream<A, S>(key: &Key<A>, nonce: &Nonce<A, S>) -> S
where
    A: AeadInPlace + NewAead,
    <A as AeadCore>::NonceSize: Sub<<S as StreamPrimitive<A>>::NonceOverhead>,
    <<A as AeadCore>::NonceSize as Sub<<S as StreamPrimitive<A>>::NonceOverhead>>::Output:
        ArrayLength<u8>,
    S: NewStream<A> + StreamPrimitive<A>,
{
    let aead = A::new(key);
    S::from_aead(aead, nonce)
}

pub mod io {
    use crypto::aead::{
        generic_array::{typenum::Unsigned, ArrayLength},
        stream::{Decryptor, Encryptor, StreamPrimitive},
        AeadCore, AeadInPlace,
    };
    use std::{
        io::{Error, ErrorKind, Read, Write},
        ops::Sub,
    };
    /// Number of plaintext bytes contained in a single block
    pub const BLOCK_SIZE: usize = 8192;
    /// # EncryptorStream
    /// `EncryptorStream` is a generic struct which implements
    /// [`Write`](https://doc.rust-lang.org/std/io/trait.Write.html). When written to, it encrypts the data using the
    /// provided [`Encryptor`](https://docs.rs/aead/latest/aead/stream/struct.Encryptor.html) and writes the encrypted
    /// bytes to the provided `Write`.
    /// ### Blocks
    /// The most important concept about an `EncryptorStream` is blocks. Due to the nature of AEADs, data is written in
    /// blocks. The size of each block is `8192` plaintext bytes by default. A smaller block can be written by writing
    /// any number of bytes to this stream and forcing the write by calling `flush`. After calling `flush`, the number
    /// of plaintext bytes which were encrypted and written can be obtained by calling `get_last_block_len`. *Note that
    /// `get_last_block_len` only updates after calling `flush`.* The total number of blocks written can be obtained by
    /// calling `get_total_blocks_written`. This value is updated after each write to the underlying stream. If some
    /// number of bytes is written to this stream without the internal buffer filling up, then the stream is dropped
    /// without calling `flush`, **those bytes will not be encrypted or written to the underlying stream!** `flush` must
    /// be called to ensure that all written bytes are encrypted and written to the underlying stream.
    pub struct EncryptorStream<'a, A, S, W>
    where
        A: AeadInPlace,
        <A as AeadCore>::NonceSize: Sub<<S as StreamPrimitive<A>>::NonceOverhead>,
        <<A as AeadCore>::NonceSize as Sub<<S as StreamPrimitive<A>>::NonceOverhead>>::Output:
            ArrayLength<u8>,
        S: StreamPrimitive<A>,
        W: Write,
    {
        /// the encryptor stream responsible for handling the encryption of data as it flows out
        enc: Encryptor<A, S>,
        /// the writer to which we write data
        out: &'a mut W,
        /// a buffer where we store the last block of data which has been written to this stream. since the encryption
        /// and decryption process works in chunks and the owner of this stream might not fill the entire block at once,
        /// we use this buffer to store the bytes which have been written to this stream, but not yet encrypted and
        /// written to the underlying stream. Bytes which have been written to the underlying stream are called "stale".
        buf: Vec<u8>,
        /// the index of the first "non-stale" byte in `buf`
        buf_cursor: usize,
        ///the number of blocks written to the underlying stream
        num_blocks: u32,
        ///the number of plaintext bytes which were encrypted and written to the underlying stream during the most
        /// recent flush
        last_block_len: u16,
    }
    impl<'a, A, S, W> EncryptorStream<'a, A, S, W>
    where
        A: AeadInPlace,
        <A as AeadCore>::NonceSize: Sub<<S as StreamPrimitive<A>>::NonceOverhead>,
        <<A as AeadCore>::NonceSize as Sub<<S as StreamPrimitive<A>>::NonceOverhead>>::Output:
            ArrayLength<u8>,
        S: StreamPrimitive<A>,
        W: Write,
    {
        /// size of the [tag (aka MAC)](https://en.wikipedia.org/wiki/Message_authentication_code) for the associated
        /// `AEAD` algorithm
        const TAG_SIZE: usize = <A as AeadCore>::TagSize::USIZE;
        /// Creates a new `EncryptorStream` with the given
        /// [`Encryptor`](https://docs.rs/aead/latest/aead/stream/struct.Encryptor.html) and
        /// [`Write`](https://doc.rust-lang.org/std/io/trait.Write.html)
        pub fn new(enc: Encryptor<A, S>, out: &'a mut W) -> EncryptorStream<'a, A, S, W> {
            EncryptorStream {
                enc: enc,
                out,
                buf: {
                    let mut v = Vec::with_capacity(BLOCK_SIZE + Self::TAG_SIZE);
                    // SAFETY: `v.len()` is currently 0, so nothing needs to be dropped. Great care is taken to ensure
                    // that nothing is read from `v` unless it has previously been written by us.
                    unsafe { v.set_len(BLOCK_SIZE) };
                    v
                },
                buf_cursor: 0,
                num_blocks: 0,
                last_block_len: 0,
            }
        }
        /// Returns the total number of blocks written to the underlying stream. This value is updated after every write
        /// to the underlying stream, and thus should properly reflect the total number of blocks actually written to
        /// the underlying stream at any given time.
        ///
        /// *Note that `get_last_block_len` does not share this property. Also note that in complex pipelines, any
        /// amount of buffering, compression, etc., might affect the timing and number of bytes reaching the end of the
        /// pipeline.*
        pub fn get_total_blocks_written(&self) -> u32 {
            self.num_blocks
        }
        /// Returns the number of plaintext bytes contained in the last block written.
        ///
        /// *Note that unlike `get_total_blocks_written`, this number is only calculated when calling `flush`. It is not
        /// updated after every call to `write`.*
        pub fn get_last_block_len(&self) -> u16 {
            self.last_block_len
        }
        // Consumes self and returns the internal `Encryptor` stream
        pub fn into_inner(self) -> Encryptor<A, S> {
            self.enc
        }
    }
    impl<'a, A, S, W> Write for EncryptorStream<'a, A, S, W>
    where
        A: AeadInPlace,
        <A as AeadCore>::NonceSize: Sub<<S as StreamPrimitive<A>>::NonceOverhead>,
        <<A as AeadCore>::NonceSize as Sub<<S as StreamPrimitive<A>>::NonceOverhead>>::Output:
            ArrayLength<u8>,
        S: StreamPrimitive<A>,
        W: Write,
    {
        fn write(&mut self, data: &[u8]) -> Result<usize, Error> {
            let data_len = data.len();
            // tracks how much of `data` has been "read".
            let mut read = 0;
            // loop until we've written or buffered all the bytes in `data`
            loop {
                // number of bytes remaining in `data`
                let remaining = data_len - read;
                // number of bytes we will read from `data`. We need to read all of the bytes from `data`, but if they
                // won't all fit in `self.buf`, we only read what will fit (for now).
                let will_read = remaining.min(BLOCK_SIZE - self.buf_cursor);
                // offset into `data` where we will begin reading
                let data_start = read;
                // offset into `data` where reading will stop (exclusive)
                let data_end = read + will_read;
                // offset into `self.buf` where we will start writing
                let buf_start = self.buf_cursor;
                // offset into `self.buf` where writing will stop (exclusive)
                let buf_end = buf_start + will_read;
                // copy as much data from `data` into `self.buf` as possible
                self.buf[buf_start..buf_end].copy_from_slice(&data[data_start..data_end]);
                // track that we read some bytes from `data`
                read += will_read;
                // track the number of "stale" bytes in `self.buf`
                self.buf_cursor += will_read;
                // if we've read everything from `data`, we're done
                if read == data_len {
                    break;
                }
                // encrypt the block of data contained in `self.buf`, mapping any `aead::Error` we encounter into an
                // `std::io::Error`
                self.enc
                    .encrypt_next_in_place(&[0u8; 0], &mut self.buf)
                    .map_err(|e| Error::new(ErrorKind::Other, e))?;
                // write all the newly encrypted data
                self.out.write_all(&mut self.buf)?;
                // set self.buf back to the appropriate length
                self.buf.truncate(BLOCK_SIZE);
                // track that we wrote a block
                self.num_blocks += 1;
                // track that `self.buf` was written, thus there all the bytes in `self.buf` are "stale"
                self.buf_cursor = 0;
            }
            Ok(data_len)
        }
        fn flush(&mut self) -> Result<(), Error> {
            if self.buf_cursor > 0 {
                // there's still data left in the buffer
                // the plaintext length of this last block
                let last_block_len = self.buf_cursor as u16;
                // truncate the buffer so that encryption and writing will take the correct number of bytes
                self.buf.truncate(self.buf_cursor);
                // encrypt and write
                self.enc
                    .encrypt_next_in_place(&[0u8; 0], &mut self.buf)
                    .map_err(|e| Error::new(ErrorKind::Other, e))?;
                self.out.write_all(&mut self.buf)?;
                // reset the cursor
                self.buf_cursor = 0;
                // resize the buffer to prepare for the next write. it's possible that the buffer is larger than
                // `BLOCK_SIZE` bytes (since the tag must be appended to the buffer). alternatively, since the buffer
                // might be less than `BLOCK_SIZE`, we also need to resize it back to that size. we check and handle
                // that here.
                if self.buf.len() > BLOCK_SIZE {
                    // `self.buf` is too big, so drop the unneeded bytes
                    self.buf.truncate(BLOCK_SIZE);
                } else {
                    // `self.buf` is too small, so set the length to be the correct size
                    // SAFETY: `self.buf_cursor` is 0, enforcing that nothing will be read from the buffer prior to
                    // being written. additionally, nothing needs to be dropped since the buffer is guaranteed to be
                    // smaller than `BLOCK_SIZE`.
                    unsafe { self.buf.set_len(BLOCK_SIZE) };
                }
                // increment num_blocks one last time
                self.num_blocks += 1;
                // now that the write has finished successfully, set the last block len
                self.last_block_len = last_block_len;
            } else {
                // nothing extra to write. the last block was BLOCK_SIZE.
                self.last_block_len = BLOCK_SIZE as u16;
            }
            // flush out writer
            self.out.flush()
        }
    }
    /// # DecryptorStream
    /// `DecryptorStream` is a generic struct which implements
    /// [`Read`](https://doc.rust-lang.org/std/io/trait.Read.html). It reads from any other `Read`, decrypts the read
    /// data using any [`Decryptor`](https://docs.rs/aead/latest/aead/stream/struct.Decryptor.html), and returns the
    /// plaintext.
    /// ### Blocks
    /// The most important concept about a `DecryptorStream` is blocks. Due to the nature of AEADs, data is read
    /// internally in blocks. The size of each block must be known in order to decrypt the ciphertext. For this reason,
    /// prior to any read, the number of blocks (`blocks_to_read`) and the length (in bytes) of the last block
    /// (`last_block_len`) must be provided. Data which has been read and decrypted is buffered internally. Once the
    /// buffer runs out of data, the next block in the underlying `Read` is read. The plaintext size of every block
    /// except the last block is assumed to be `8192` bytes. The last block is expected to be `last_block_len` bytes. A
    /// call to `read` will not necessarily read a block from the underlying stream. Depending on the size of the buffer
    /// expected to be filled when calling `read`, any number of blocks could be read, including zero blocks. If a block
    /// is any size other than what's expected according to the assumed `8192` bytes or `last_block_len`, decryption
    /// will fail. If the data originated from an `EncryptorStream`, the values for `blocks_to_read` and
    /// `last_block_len` can be obtained through `EncryptorStream::get_total_blocks_written` and
    /// `EncryptorStream::get_last_block_len` respectively.
    pub struct DecryptorStream<'a, A, S, R>
    where
        A: AeadInPlace,
        <A as AeadCore>::NonceSize: Sub<<S as StreamPrimitive<A>>::NonceOverhead>,
        <<A as AeadCore>::NonceSize as Sub<<S as StreamPrimitive<A>>::NonceOverhead>>::Output:
            ArrayLength<u8>,
        S: StreamPrimitive<A>,
        R: Read,
    {
        /// the decryptor stream responsible for handling the decryption of data as it flows in
        dec: Decryptor<A, S>,
        /// the reader from which we read data
        in_: &'a mut R,
        /// a buffer where we store the last block of data which has been read and decrypted. since the encryption and
        /// decryption process works in chunks and the owner of this stream might not want the entire block at once, we
        /// use this buffer to store the bytes which have been read and decrypted, but not yet passed onto the owner.
        /// bytes which have been passed are called "stale".
        buf: Vec<u8>,
        /// the index of the first "non-stale" byte in `buf`. `buf_cursor` works such that a value of `0` refers to the
        /// end of `buf` (index `buf.len()`) and a value of `buf.len()` points to the beginning of `buf` (index 0).
        buf_cursor: usize,
        /// the number of blocks remaining which need to be read and decrypted. this is primarily used to keep track of
        /// when we've reached the last block, which, for large streams of data, is usually the only block of an unusual
        /// size. blocks are expected to be of size `BLOCK_SIZE`, unless `blocks_to_read` is 1, in which case the next
        /// block will be the last block and is expected to be of size `last_block_len`.
        blocks_to_read: u32,
        /// the number of plaintext bytes which the last block read from `in_` will contain
        last_block_len: u16,
    }
    impl<'a, A, S, R> DecryptorStream<'a, A, S, R>
    where
        A: AeadInPlace,
        <A as AeadCore>::NonceSize: Sub<<S as StreamPrimitive<A>>::NonceOverhead>,
        <<A as AeadCore>::NonceSize as Sub<<S as StreamPrimitive<A>>::NonceOverhead>>::Output:
            ArrayLength<u8>,
        S: StreamPrimitive<A>,
        R: Read,
    {
        /// size of the [tag (aka MAC)](https://en.wikipedia.org/wiki/Message_authentication_code) for the associated
        /// `AEAD` algorithm
        const TAG_SIZE: usize = <A as AeadCore>::TagSize::USIZE;
        /// BLOCK_SIZE + TAG_SIZE, i.e. the number of bytes required to read one block
        const BUF_LEN: usize = BLOCK_SIZE + <A as AeadCore>::TagSize::USIZE;
        /// Creates a new `DecryptorStream` with the given
        /// [`Decryptor`](https://docs.rs/aead/latest/aead/stream/struct.Decryptor.html),
        /// [`Read`](https://doc.rust-lang.org/std/io/trait.Read.html), expected number of blocks to read, and the
        /// plaintext length of the last block in the sequence.
        pub fn new(
            dec: Decryptor<A, S>,
            in_: &'a mut R,
            blocks_to_read: u32,
            last_block_len: u16,
        ) -> DecryptorStream<'a, A, S, R> {
            DecryptorStream {
                dec: dec,
                in_,
                buf: Vec::with_capacity(Self::BUF_LEN),
                buf_cursor: 0,
                blocks_to_read,
                last_block_len,
            }
        }
        /// Sets the expected number of blocks to read and the plaintext length of the last block in the sequence.
        pub fn set_blocks_remaining(&mut self, blocks_to_read: u32, last_block_len: u16) {
            self.blocks_to_read = blocks_to_read;
            self.last_block_len = last_block_len;
        }
        // Consumes self and returns the internal `Decryptor` stream
        pub fn into_inner(self) -> Decryptor<A, S> {
            self.dec
        }
    }
    impl<'a, A, S, R> Read for DecryptorStream<'a, A, S, R>
    where
        A: AeadInPlace,
        <A as AeadCore>::NonceSize: Sub<<S as StreamPrimitive<A>>::NonceOverhead>,
        <<A as AeadCore>::NonceSize as Sub<<S as StreamPrimitive<A>>::NonceOverhead>>::Output:
            ArrayLength<u8>,
        S: StreamPrimitive<A>,
        R: Read,
    {
        fn read(&mut self, data: &mut [u8]) -> Result<usize, Error> {
            let data_len = data.len();
            // tracks how many bytes have been "written" to `data`
            let mut written = 0;
            // loop until `data` is full or we've read all the blocks
            loop {
                // number of bytes still needing to be written
                let remaining = data_len - written;
                // number of bytes we will write to `data`. We need to fill `data`, but if there's not enough in
                // `self.buf`, we only write what will fit (for now).
                let will_write = remaining.min(self.buf_cursor);
                // offset into `data` where we will begin writing
                let data_start = written;
                // offset into `data` where writing will stop (exclusive)
                let data_end = written + will_write;
                // offset into `self.buf` where we will start reading
                let buf_start = self.buf.len() - self.buf_cursor;
                // offset into `self.buf` where reading will stop (exclusive)
                let buf_end = buf_start + will_write;
                // copy as much data from `self.buf` into `data` as possible
                data[data_start..data_end].copy_from_slice(&self.buf[buf_start..buf_end]);
                // track the number of bytes "written" to `data`
                written += will_write;
                // track the number of "stale" bytes in `self.buf`
                self.buf_cursor -= will_write;
                // if data is full, exit
                if written == data_len {
                    break;
                }
                // check for potential edge cases of `self.blocks_to_read`
                match self.blocks_to_read {
                    0 => {
                        // if there's no more blocks to read, return the number of blocks we've read
                        return Ok(written);
                    }
                    1 => {
                        // if there's only one more block to read, only read the amount specified by
                        // `self.last_block_len`
                        self.buf
                            .resize(self.last_block_len as usize + Self::TAG_SIZE, 0);
                    }
                    _ => {
                        // otherwise, read `BLOCK_SIZE` bytes plus the tag
                        unsafe { self.buf.set_len(Self::BUF_LEN) };
                    }
                }
                // read and decrypt
                self.in_.read_exact(&mut self.buf[..])?;
                self.dec
                    .decrypt_next_in_place(&[0u8; 0], &mut self.buf)
                    .map_err(|e| Error::new(ErrorKind::Other, e))?;
                // track that we've read a block
                self.blocks_to_read -= 1;
                // track that there are no "stale" bytes in `self.buf`
                self.buf_cursor = self.buf.len();
            }
            Ok(data_len)
        }
    }
}
