use anyhow::Result;
use nom::{
    error::{ErrorKind, ParseError, VerboseError},
    CompareResult, Err, IResult, InputLength, Needed, Parser, ToUsize,
};

use super::bytes::{ShitCompare, ShitNeededForParsing};

/// A specific form of `nom::IResult` that uses `nom::error::VerboseError`.
pub type VResult<I, O> = IResult<I, O, VerboseError<I>>;

/// `sep_list!(t, d)` represents `t *(d t)` and automatically collapses it into
/// `Vec<T>`.
///
/// Also `sep_list!(?t, d)` represents `[t *(d t)]` and will default to an empty
/// vec.
///
/// If `d` is not passed then it defaults to `SP`.
#[macro_export]
macro_rules! sep_list {
    ($t:expr) => {
        map(
            pair($t, many0(preceded(crate::proto::rfc2234::SP, $t))),
            |(hd, mut tl)| {
                tl.insert(0, hd);
                tl
            },
        )
    };
    ($t:expr, $d:expr) => {
        map(pair($t, many0(preceded($d, $t))), |(hd, mut tl)| {
            tl.insert(0, hd);
            tl
        })
    };
    (? $t:expr) => {
        map(
            opt(pair($t, many0(preceded(crate::proto::rfc2234::SP, $t)))),
            |opt| {
                opt.map(|(hd, mut tl)| {
                    tl.insert(0, hd);
                    tl
                })
                .unwrap_or_else(Vec::new)
            },
        )
    };
    (? $t:expr, $d:expr) => {
        map(opt(pair($t, many0(preceded($d, $t)))), |opt| {
            opt.map(|(hd, mut tl)| {
                tl.insert(0, hd);
                tl
            })
            .unwrap_or_else(Vec::new)
        })
    };
}

#[macro_export]
macro_rules! opt_nil {
    ($t:expr) => {
        alt((map($t, Some), map(crate::proto::rfc3501::nil, |_| None)))
    };
}

#[macro_export]
macro_rules! paren {
    ($t:expr) => {
        delimited(byte(b'('), $t, byte(b')'))
    };
}

/// Parse from a [u8] into a u32 without first decoding it to UTF-8.
pub fn parse_u32(s: impl AsRef<[u8]>) -> Result<u32> {
    let mut total = 0u32;
    let s = s.as_ref();
    for digit in s.iter() {
        let digit = *digit;
        total *= 10;
        if !(digit >= b'0' && digit <= b'9') {
            bail!("invalid digit {}", digit)
        }
        total += (digit - b'\x30') as u32;
    }
    Ok(total)
}

/// Always fails, used as a no-op.
pub fn never<I, O, E>(i: I) -> IResult<I, O, E>
where
    E: ParseError<I>,
{
    Err(Err::Error(E::from_error_kind(i, ErrorKind::Not)))
}

/// Skip the part of the input matched by the given parser.
pub fn skip<E, F, I, O>(mut f: F) -> impl FnMut(I) -> IResult<I, (), E>
where
    I: Clone,
    F: Parser<I, O, E>,
{
    move |i: I| match f.parse(i.clone()) {
        Ok(_) => Ok((i, ())),
        Err(err) => Err(err),
    }
}

/// Same as nom's streaming take, but operates on Bytes
pub fn take<C, I, E>(count: C) -> impl Fn(I) -> IResult<I, I, E>
where
    I: ShitNeededForParsing + InputLength,
    E: ParseError<I>,
    C: ToUsize,
{
    let c = count.to_usize();
    move |i: I| match i.slice_index(c) {
        Err(i) => Err(Err::Incomplete(i)),
        Ok(index) => Ok(i.take_split(index)),
    }
}

/// Same as nom's streaming take_while1, but operates on Bytes
pub fn take_while1<F, I, E, T>(cond: F) -> impl Fn(I) -> IResult<I, I, E>
where
    I: ShitNeededForParsing<Item = T>,
    E: ParseError<I>,
    F: Fn(T) -> bool,
{
    move |i: I| {
        let e: ErrorKind = ErrorKind::TakeWhile1;
        i.split_at_position1(|c| !cond(c), e)
    }
}

/// Tag (case-)Insensitive: Same as nom's tag_no_case, but operates on Bytes
pub fn tagi<T, I, E>(tag: T) -> impl Fn(I) -> IResult<I, I, E>
where
    I: ShitNeededForParsing<Sliced = I> + InputLength + ShitCompare<T>,
    T: InputLength + Clone,
    E: ParseError<I>,
{
    let tag_len = tag.input_len();
    move |i: I| match i.compare_no_case(tag.clone()) {
        CompareResult::Ok => {
            let fst = i.slice(0..tag_len);
            let snd = i.slice(tag_len..);
            Ok((snd, fst))
        }
        CompareResult::Incomplete => Err(Err::Incomplete(Needed::new(tag_len - i.input_len()))),
        CompareResult::Error => {
            let e: ErrorKind = ErrorKind::Tag;
            Err(Err::Error(E::from_error_kind(i, e)))
        }
    }
}

/// Same as nom's satisfy, but operates on `Bytes` instead of `&str`.
pub fn satisfy<F, I, E, T>(f: F) -> impl Fn(I) -> IResult<I, T, E>
where
    I: ShitNeededForParsing<Item = T, Sliced = I>,
    F: Fn(T) -> bool,
    E: ParseError<I>,
    T: Copy,
{
    move |i: I| match i.first().map(|t| (f(t), t)) {
        Some((true, ft)) => Ok((i.slice(1..), ft)),
        Some((false, _)) => Err(Err::Error(E::from_error_kind(i, ErrorKind::Satisfy))),
        None => Err(Err::Incomplete(Needed::Unknown)),
    }
}

/// Match a single byte exactly.
pub fn byte<I, E: ParseError<I>>(b: u8) -> impl Fn(I) -> IResult<I, u8, E>
where
    I: ShitNeededForParsing<Item = u8, Sliced = I>,
{
    satisfy(move |c| c == b)
}
