//! IMAP Specification
//! ---
//!
//! Grammar from <https://datatracker.ietf.org/doc/html/rfc3501#section-9>

use nom::{
    branch::alt,
    combinator::{map, map_res, opt, verify},
    multi::{many0, many1},
    sequence::{delimited, pair, preceded, separated_pair, terminated, tuple},
};

use super::bytes::Bytes;
use super::parsers::{byte, never, parse_u32, satisfy, tagi, take, take_while1, VResult};
use super::response::{
    Address, Atom, Capability, Condition, Envelope, Flag, Mailbox, MailboxData, MailboxList,
    MailboxListFlag, MessageAttribute, Response, ResponseCode, ResponseText, Status, Tag,
};
use super::rfc2234::{is_char, is_cr, is_ctl, is_digit, is_dquote, is_lf, is_sp, CRLF, DQUOTE, SP};

rule!(pub address : Address => map(paren!(tuple((
    terminated(addr_name, SP),
    terminated(addr_adl, SP),
    terminated(addr_mailbox, SP),
    addr_host
))), |(name, adl, mailbox, host)| Address { name, adl, mailbox, host }));

rule!(pub addr_adl : Option<Bytes> => nstring);

rule!(pub addr_host : Option<Bytes> => nstring);

rule!(pub addr_mailbox : Option<Bytes> => nstring);

rule!(pub addr_name : Option<Bytes> => nstring);

rule!(pub astring : Bytes => alt((take_while1(is_astring_char), string)));

pub(crate) fn is_astring_char(c: u8) -> bool { is_atom_char(c) || is_resp_specials(c) }
rule!(pub ASTRING_CHAR : u8 => alt((ATOM_CHAR, resp_specials)));

rule!(pub atom : Bytes => take_while1(is_atom_char));

pub(crate) fn is_atom_char(c: u8) -> bool { is_char(c) && !is_atom_specials(c) }
rule!(pub ATOM_CHAR : u8 => satisfy(is_atom_char));

pub(crate) fn is_atom_specials(c: u8) -> bool {
    c == b'('
        || c == b')'
        || c == b'{'
        || is_sp(c)
        || is_ctl(c)
        || is_list_wildcards(c)
        || is_quoted_specials(c)
        || is_resp_specials(c)
}
rule!(pub atom_specials : u8 => satisfy(is_atom_specials));

rule!(pub auth_type : Atom => atom);

rule!(pub capability : Capability => alt((
    map(preceded(tagi(b"AUTH="), auth_type), Capability::Auth),
    map(atom, Capability::Atom),
)));

rule!(pub capability_data : Vec<Capability> => preceded(tagi(b"CAPABILITY"), {
    map(separated_pair(
        many0(preceded(SP, capability)),
        pair(SP, tagi(b"IMAP4rev1")),
        many0(preceded(SP, capability))
    ), |(mut a, b)| { a.extend(b); a })
}));

pub(crate) fn is_char8(c: u8) -> bool { c != b'\0' }

rule!(pub continue_req : Response => delimited(pair(byte(b'+'), SP),
    // TODO: handle base64 case?
    map(resp_text, Response::Continue),
CRLF));

rule!(pub envelope : Envelope => map(paren!(tuple((
    terminated(env_date, SP),
    terminated(env_subject, SP),
    terminated(env_from, SP),
    terminated(env_sender, SP),
    terminated(env_reply_to, SP),
    terminated(env_to, SP),
    terminated(env_cc, SP),
    terminated(env_bcc, SP),
    terminated(env_in_reply_to, SP),
    env_message_id,
))), |(date, subject, from, sender, reply_to, to, cc, bcc, in_reply_to, message_id)|
    Envelope { date, subject, from, sender, reply_to, to, cc, bcc, in_reply_to, message_id }));

rule!(pub env_bcc : Option<Vec<Address>> => opt_nil!(paren!(many1(address))));

rule!(pub env_cc : Option<Vec<Address>> => opt_nil!(paren!(many1(address))));

rule!(pub env_date : Option<Bytes> => nstring);

rule!(pub env_from : Option<Vec<Address>> => opt_nil!(paren!(many1(address))));

rule!(pub env_in_reply_to : Option<Bytes> => nstring);

rule!(pub env_message_id : Option<Bytes> => nstring);

rule!(pub env_reply_to : Option<Vec<Address>> => opt_nil!(paren!(many1(address))));

rule!(pub env_sender : Option<Vec<Address>> => opt_nil!(paren!(many1(address))));

rule!(pub env_subject : Option<Bytes> => nstring);

rule!(pub env_to : Option<Vec<Address>> => opt_nil!(paren!(many1(address))));

rule!(pub flag : Flag => alt((
    map(tagi(b"\\Answered"), |_| Flag::Answered),
    map(tagi(b"\\Flagged"), |_| Flag::Flagged),
    map(tagi(b"\\Deleted"), |_| Flag::Deleted),
    map(tagi(b"\\Seen"), |_| Flag::Seen),
    map(tagi(b"\\Draft"), |_| Flag::Draft),
    map(flag_keyword, Flag::Keyword),
    map(flag_extension, Flag::Extension),
)));

rule!(pub flag_extension : Atom => preceded(byte(b'\\'), atom));

rule!(pub flag_fetch : Flag => alt((flag, map(tagi(b"\\Recent"), |_| Flag::Recent))));

rule!(pub flag_keyword : Atom => atom);

rule!(pub flag_list : Vec<Flag> => paren!(sep_list!(?flag)));

pub(crate) fn is_list_wildcards(c: u8) -> bool { c == b'%' || c == b'*' }
rule!(pub list_wildcards : u8 => satisfy(is_list_wildcards));

// literal = "{" number "}" CRLF *CHAR8
//         ; Number represents the number of CHAR8s
// TODO: Future work, could possibly initialize writing to file if the length is
// determined to exceed a certain threshold so we don't have insane amounts of
// data in memory
pub fn literal(i: Bytes) -> VResult<Bytes, Bytes> {
    let mut length_of = terminated(delimited(byte(b'{'), number, byte(b'}')), CRLF);
    let (i, length) = length_of(i)?;
    debug!("length is: {:?}", length);
    map(take(length), Bytes::from)(i)
}

rule!(pub mailbox : Mailbox => alt((
    map(tagi(b"INBOX"), |_| Mailbox::Inbox),
    map(astring, Mailbox::Name),
)));

rule!(pub mailbox_data : MailboxData => alt((
    map(preceded(pair(tagi(b"FLAGS"), SP), flag_list), MailboxData::Flags),
    map(preceded(pair(tagi(b"LIST"), SP), mailbox_list), MailboxData::List),
    map(preceded(pair(tagi(b"LSUB"), SP), mailbox_list), MailboxData::Lsub),
    map(terminated(number, pair(SP, tagi(b"EXISTS"))), MailboxData::Exists),
    map(terminated(number, pair(SP, tagi(b"RECENT"))), MailboxData::Recent),
)));

rule!(pub mailbox_list : MailboxList => map(separated_pair(
    paren!(map(opt(mbx_list_flags), |opt| opt.unwrap_or_else(Vec::new))),
    SP, separated_pair(
        alt((
            map(delimited(DQUOTE, QUOTED_CHAR, DQUOTE), Some),
            map(nil, |_| None),
        )),
        SP, mailbox,
    ),
), |(flags, (delimiter, mailbox))| MailboxList { flags, delimiter, mailbox }));

rule!(pub mbx_list_flags : Vec<MailboxListFlag> => alt((
    map(tuple((
        many0(terminated(mbx_list_oflag, SP)),
        mbx_list_sflag,
        many0(preceded(SP, mbx_list_oflag)),
    )), |(mut a, b, c)| { a.push(b); a.extend(c); a }),
    sep_list!(mbx_list_oflag),
)));

rule!(pub mbx_list_oflag : MailboxListFlag => alt((
    map(tagi(b"\\Noinferiors"), |_| MailboxListFlag::NoInferiors),
    map(flag_extension, MailboxListFlag::Extension),
)));

rule!(pub mbx_list_sflag : MailboxListFlag => alt((
    map(tagi(b"\\NoSelect"), |_| MailboxListFlag::NoSelect),
    map(tagi(b"\\Marked"), |_| MailboxListFlag::Marked),
    map(tagi(b"\\Unmarked"), |_| MailboxListFlag::Unmarked),
)));

rule!(pub message_data : Response => alt((
    map(terminated(nz_number, pair(SP, tagi(b"EXPUNGE"))), Response::Expunge),
    map(separated_pair(nz_number, SP, preceded(pair(tagi(b"FETCH"), SP), msg_att)),
        |(n, attrs)| Response::Fetch(n, attrs)),
)));

rule!(pub msg_att : Vec<MessageAttribute> => paren!(sep_list!(alt((msg_att_dynamic, msg_att_static)))));

rule!(pub msg_att_dynamic : MessageAttribute => alt((
    map(preceded(pair(tagi(b"FLAGS"), SP),
        paren!(sep_list!(?flag_fetch))), MessageAttribute::Flags),
    never,
)));

rule!(pub msg_att_static : MessageAttribute => alt((
    map(preceded(pair(tagi(b"ENVELOPE"), SP), envelope), MessageAttribute::Envelope),
    map(preceded(pair(tagi(b"ENVELOPE"), SP), envelope), MessageAttribute::Envelope),
)));

rule!(pub nil : Bytes => tagi(b"NIL"));

rule!(pub nstring : Option<Bytes> => opt_nil!(string));

pub(crate) fn number(i: Bytes) -> VResult<Bytes, u32> {
    map_res(take_while1(is_digit), parse_u32)(i)
}

rule!(pub nz_number : u32 => verify(number, |n| *n != 0));

rule!(pub quoted : Bytes => delimited(DQUOTE, take_while1(is_quoted_char), DQUOTE));

fn is_quoted_char(c: u8) -> bool { is_char(c) && !is_quoted_specials(c) }
rule!(pub QUOTED_CHAR : u8 => alt((satisfy(is_quoted_char), preceded(byte(b'\\'), quoted_specials))));

pub(crate) fn is_quoted_specials(c: u8) -> bool { is_dquote(c) || c == b'\\' }
rule!(pub quoted_specials : u8 => satisfy(is_quoted_specials));

// TODO: technically, this is supposed to be
rule!(pub response : Response => alt((continue_req, response_data, response_done)));

rule!(pub response_data : Response => delimited(pair(byte(b'*'), SP), alt((
    map(resp_cond_state, Response::Condition),
    map(resp_cond_bye, Response::Condition),
    map(mailbox_data, Response::MailboxData),
    message_data,
    map(capability_data, Response::Capabilities),
)), CRLF));

rule!(pub response_done : Response => alt((response_tagged, response_fatal)));

rule!(pub response_fatal : Response => delimited(pair(byte(b'*'), SP),
    map(resp_cond_bye, Response::Fatal), CRLF));

rule!(pub response_tagged : Response => map(terminated(separated_pair(tag, SP, resp_cond_state), CRLF),
    |(tag, cond)| Response::Tagged(tag, cond)));

rule!(pub resp_cond_bye : Condition => preceded(pair(tagi(b"BYE"), SP),
    map(resp_text, |ResponseText { code, info }| Condition { status: Status::Bye, code, info })));

rule!(pub resp_cond_state : Condition => map(
    separated_pair(
        alt((
            map(tagi(b"OK"), |_| Status::Ok),
            map(tagi(b"NO"), |_| Status::No),
            map(tagi(b"BAD"), |_| Status::Bad),
        )),
        SP,
        resp_text,
    ),
    |(status, ResponseText { code, info })| Condition { status, code, info }
));

pub(crate) fn is_resp_specials(c: u8) -> bool { c == b']' }
rule!(pub resp_specials : u8 => satisfy(is_resp_specials));

rule!(pub resp_text : ResponseText => map(pair(
    opt(terminated(delimited(byte(b'['), resp_text_code, byte(b']')), SP)),
    text,
), |(code, info)| ResponseText { code, info }));

rule!(pub resp_text_code : ResponseCode => alt((
    map(tagi(b"ALERT"), |_| ResponseCode::Alert),
    map(capability_data, ResponseCode::Capabilities),
)));

rule!(pub string : Bytes => alt((quoted, literal)));

pub(crate) fn is_tag_char(c: u8) -> bool { is_astring_char(c) && c != b'+' }
rule!(pub tag : Tag => map(take_while1(is_tag_char), Tag));

rule!(pub text : Bytes => map(take_while1(is_text_char), Bytes::from));

pub(crate) fn is_text_char(c: u8) -> bool { is_char(c) && !is_cr(c) && !is_lf(c) }
rule!(pub TEXT_CHAR : u8 => satisfy(is_text_char));
