//! ABNF Core Rules
//! ---
//!
//! Grammar from <https://tools.ietf.org/html/rfc2234#section-6.1>

use nom::{
    branch::alt,
    multi::many0,
    sequence::{pair, preceded},
};

use super::parsers::{byte, satisfy, skip};

rule!(pub ALPHA : u8 => satisfy(|c| (c >= b'a' && c <= b'z') || (c >= b'A' && c <= b'Z')));

rule!(pub BIT : u8 => satisfy(|c| c == b'0' || c == b'1'));

pub(crate) fn is_char(c: u8) -> bool { c != b'\0' }
rule!(pub CHAR : u8 => satisfy(is_char));

pub(crate) fn is_cr(c: u8) -> bool { c == b'\x0d' }
rule!(pub CR : u8 => satisfy(is_cr));

rule!(pub CRLF : (u8, u8) => pair(CR, LF));

pub(crate) fn is_ctl(c: u8) -> bool { c <= b'\x1f' || c == b'\x7f' }
rule!(pub CTL : u8 => satisfy(is_ctl));

pub(crate) fn is_digit(c: u8) -> bool { c >= b'\x30' && c <= b'\x39' }
rule!(pub DIGIT : u8 => satisfy(is_digit));

pub(crate) fn is_dquote(c: u8) -> bool { c == b'\x22' }
rule!(pub DQUOTE : u8 => satisfy(is_dquote));

rule!(pub HEXDIG : u8 => alt((DIGIT, satisfy(|c| c >= b'A' && c <= b'F'))));

rule!(pub HTAB : u8 => byte(b'\x09'));

pub(crate) fn is_lf(c: u8) -> bool { c == b'\x0a' }
rule!(pub LF : u8 => satisfy(is_lf));

rule!(pub LWSP : () => skip(many0(alt((WSP, preceded(CRLF, WSP))))));

// rule!(pub OCTET : char => anychar);

pub(crate) fn is_sp(c: u8) -> bool { c == b'\x20' }
rule!(pub SP : u8 => satisfy(is_sp));

rule!(pub VCHAR : u8 => satisfy(|c| c >= b'\x21' && c <= b'\x7e'));

rule!(pub WSP : u8 => alt((SP, HTAB)));
