use std::io::{self, Write};

use format_bytes::DisplayBytes;

use crate::proto::{bytes::Bytes, formatter::quote_string as q};

#[derive(Clone, Debug)]
pub enum Command {
    // Any state
    Capability,
    Noop,
    Logout,

    // Not authenticated
    Login(CommandLogin),
    Starttls,
    Authenticate,

    // Authenticated
    Select(CommandSelect),
    Examine,
    Create,
    Delete,
    Rename,
    Subscribe,
    Unsubscribe,
    List(CommandList),
    Lsub,
    Status,
    Append,

    // Selected
    Check,
    Close,
    Expunge,
    Search,
    Copy,
    Fetch(CommandFetch),
    Store,
    UidCopy,
    UidFetch(CommandFetch),
    UidStore,
    UidSearch(CommandSearch),

    // Extensions
    #[cfg(feature = "rfc2177")]
    Idle,
    #[cfg(feature = "rfc2177")]
    Done,
}

impl DisplayBytes for Command {
    fn display_bytes(&self, w: &mut dyn Write) -> io::Result<()> {
        match self {
            // command-any
            Command::Capability => write_bytes!(w, b"CAPABILITY"),
            Command::Logout => write_bytes!(w, b"LOGOUT"),
            Command::Noop => write_bytes!(w, b"NOOP"),

            // command-nonauth
            Command::Login(login) => {
                write_bytes!(w, b"LOGIN {} {}", q(&login.userid), q(&login.password))
            }
            Command::Starttls => write_bytes!(w, b"STARTTLS"),

            // command-auth
            Command::List(list) => {
                write_bytes!(w, b"LIST {} {}", q(&list.reference), q(&list.mailbox))
            }
            Command::Select(select) => write_bytes!(w, b"SELECT {}", q(&select.mailbox)),

            #[cfg(feature = "rfc2177")]
            Command::Idle => write_bytes!(w, b"IDLE"),

            _ => Ok(()),
        }
    }
}

#[derive(Clone, Debug)]
pub struct CommandFetch {
    pub ids: Vec<u32>,
    pub items: FetchItems,
}

#[derive(Clone, Debug)]
pub struct CommandList {
    pub reference: Bytes,
    pub mailbox: Bytes,
}

#[derive(Clone, Debug)]
pub struct CommandLogin {
    pub userid: Bytes,
    pub password: Bytes,
}

#[derive(Clone, Debug)]
pub struct CommandSearch {
    pub criteria: SearchCriteria,
}

#[derive(Clone, Debug)]
pub struct CommandSelect {
    pub mailbox: Bytes,
}

//

#[derive(Clone, Debug)]
pub enum FetchItems {
    All,
    Fast,
    Full,
    BodyPeek,
    Items(Vec<FetchAttr>),
    /// item set that panorama uses, TODO: remove when FetchItems has a builder
    PanoramaAll,
}

#[derive(Clone, Debug)]
pub enum FetchAttr {}

#[derive(Clone, Debug)]
pub enum SearchCriteria {
    All,
}
