use std::io::{self, Write};
use std::ops::{Deref, RangeBounds};

use format_bytes::DisplayBytes;
use nom::{
    error::{ErrorKind, ParseError},
    CompareResult, Err, IResult, InputLength, Needed,
};

/// Glue code between nom and Bytes so they work together
#[derive(Clone, Debug, Default, PartialEq, Eq)]
pub struct Bytes(bytes::Bytes);

impl Bytes {
    pub fn len(&self) -> usize { self.0.len() }
}

impl DisplayBytes for Bytes {
    fn display_bytes(&self, w: &mut dyn Write) -> io::Result<()> { w.write(&*self.0).map(|_| ()) }
}

impl From<bytes::Bytes> for Bytes {
    fn from(b: bytes::Bytes) -> Self { Bytes(b) }
}

impl From<&'static [u8]> for Bytes {
    fn from(slice: &'static [u8]) -> Self { Bytes(bytes::Bytes::from(slice)) }
}

impl From<&'static str> for Bytes {
    fn from(s: &'static str) -> Self { Bytes(bytes::Bytes::from(s.as_bytes())) }
}

impl From<String> for Bytes {
    fn from(slice: String) -> Self { Bytes(bytes::Bytes::from(slice)) }
}

pub trait ShitNeededForParsing: Sized {
    type Item;
    type Sliced;
    fn slice<R: RangeBounds<usize>>(&self, range: R) -> Self::Sliced;

    fn first(&self) -> Option<Self::Item>;
    fn slice_index(&self, count: usize) -> Result<usize, Needed>;

    // InputTake
    fn take(&self, count: usize) -> Self;
    fn take_split(&self, count: usize) -> (Self, Self);

    // InputTakeAtPosition
    fn split_at_position<P, E: ParseError<Self>>(&self, predicate: P) -> IResult<Self, Self, E>
    where
        P: Fn(Self::Item) -> bool;
    fn split_at_position1<P, E: ParseError<Self>>(
        &self,
        predicate: P,
        e: ErrorKind,
    ) -> IResult<Self, Self, E>
    where
        P: Fn(Self::Item) -> bool;
    fn split_at_position_complete<P, E: ParseError<Self>>(
        &self,
        predicate: P,
    ) -> IResult<Self, Self, E>
    where
        P: Fn(Self::Item) -> bool;
    fn split_at_position1_complete<P, E: ParseError<Self>>(
        &self,
        predicate: P,
        e: ErrorKind,
    ) -> IResult<Self, Self, E>
    where
        P: Fn(Self::Item) -> bool;
}

impl ShitNeededForParsing for Bytes {
    type Item = u8;

    type Sliced = Bytes;
    fn slice<R: RangeBounds<usize>>(&self, range: R) -> Self::Sliced { Self(self.0.slice(range)) }

    fn first(&self) -> Option<Self::Item> { self.0.first().copied() }
    fn slice_index(&self, count: usize) -> Result<usize, Needed> {
        if self.len() >= count {
            Ok(count)
        } else {
            Err(Needed::new(count - self.len()))
        }
    }

    // InputTake
    fn take(&self, count: usize) -> Self { self.slice(..count) }
    fn take_split(&self, count: usize) -> (Self, Self) {
        let mut prefix = self.clone();
        let suffix = Self(prefix.0.split_off(count));
        (suffix, prefix)
    }

    // InputTakeAtPosition
    fn split_at_position<P, E: ParseError<Self>>(&self, predicate: P) -> IResult<Self, Self, E>
    where
        P: Fn(Self::Item) -> bool,
    {
        match (0..self.len()).find(|b| predicate(self[*b])) {
            Some(i) => Ok((self.slice(i..), self.slice(..i))),
            None => Err(Err::Incomplete(Needed::new(1))),
        }
    }

    fn split_at_position1<P, E: ParseError<Self>>(
        &self,
        predicate: P,
        e: ErrorKind,
    ) -> IResult<Self, Self, E>
    where
        P: Fn(Self::Item) -> bool,
    {
        match (0..self.len()).find(|b| predicate(self[*b])) {
            Some(0) => Err(Err::Error(E::from_error_kind(self.clone(), e))),
            Some(i) => Ok((self.slice(i..), self.slice(..i))),
            None => Err(Err::Incomplete(Needed::new(1))),
        }
    }

    fn split_at_position_complete<P, E: ParseError<Self>>(
        &self,
        predicate: P,
    ) -> IResult<Self, Self, E>
    where
        P: Fn(Self::Item) -> bool,
    {
        match (0..self.len()).find(|b| predicate(self[*b])) {
            Some(i) => Ok((self.slice(i..), self.slice(..i))),
            None => Ok(self.take_split(self.input_len())),
        }
    }

    fn split_at_position1_complete<P, E: ParseError<Self>>(
        &self,
        predicate: P,
        e: ErrorKind,
    ) -> IResult<Self, Self, E>
    where
        P: Fn(Self::Item) -> bool,
    {
        match (0..self.len()).find(|b| predicate(self[*b])) {
            Some(0) => Err(Err::Error(E::from_error_kind(self.clone(), e))),
            Some(i) => Ok((self.slice(i..), self.slice(..i))),
            None => {
                if self.is_empty() {
                    Err(Err::Error(E::from_error_kind(self.clone(), e)))
                } else {
                    Ok(self.take_split(self.input_len()))
                }
            }
        }
    }
}

pub trait ShitCompare<T> {
    fn compare(&self, t: T) -> CompareResult;
    fn compare_no_case(&self, t: T) -> CompareResult;
}

impl ShitCompare<&[u8]> for Bytes {
    fn compare(&self, other: &[u8]) -> CompareResult {
        match self.iter().zip(other.iter()).any(|(a, b)| a != b) {
            true => CompareResult::Error,
            false if self.len() < other.len() => CompareResult::Incomplete,
            false => CompareResult::Ok,
        }
    }
    fn compare_no_case(&self, other: &[u8]) -> CompareResult {
        match self
            .iter()
            .zip(other.iter())
            .any(|(a, b)| (a | 0x20) != (b | 0x20))
        {
            true => CompareResult::Error,
            false if self.len() < other.len() => CompareResult::Incomplete,
            false => CompareResult::Ok,
        }
    }
}

macro_rules! array_impls {
    ($($N:expr)+) => { $(
        impl ShitCompare<[u8; $N]> for Bytes {
            #[inline(always)]
            fn compare(&self, t: [u8; $N]) -> CompareResult {
              self.compare(&t[..])
            }

            #[inline(always)]
            fn compare_no_case(&self, t: [u8;$N]) -> CompareResult {
              self.compare_no_case(&t[..])
            }
        }

        impl ShitCompare<&[u8; $N]> for Bytes {
            #[inline(always)]
            fn compare(&self, t: &[u8; $N]) -> CompareResult {
              self.compare(&t[..])
            }

            #[inline(always)]
            fn compare_no_case(&self, t: &[u8;$N]) -> CompareResult {
              self.compare_no_case(&t[..])
            }
        }

        impl From<&'static [u8; $N]> for Bytes {
            fn from(slice: &'static [u8; $N]) -> Self { Bytes(bytes::Bytes::from(&slice[..])) }
        }
    )* }
}

array_impls! {
     0  1  2  3  4  5  6  7  8  9
    10 11 12 13 14 15 16 17 18 19
    20 21 22 23 24 25 26 27 28 29
    30 31 32 33 34 35 36 37 38 39
    40 41 42 43 44 45 46 47 48 49
    50 51 52 53 54 55 56 57 58 59
}

impl Bytes {
    pub fn inner(self) -> bytes::Bytes { self.0 }
}

impl Deref for Bytes {
    type Target = [u8];
    fn deref(&self) -> &Self::Target { &*self.0 }
}

impl AsRef<[u8]> for Bytes {
    fn as_ref(&self) -> &[u8] { &*self.0 }
}

impl InputLength for Bytes {
    fn input_len(&self) -> usize { self.0.len() }
}
