use std::sync::atomic::{AtomicU32, Ordering};

use anyhow::Result;
use futures::{
    future::{self, FutureExt, TryFutureExt},
    stream::StreamExt,
};
use tokio::{
    io::{split, AsyncRead, AsyncWrite, AsyncWriteExt, BufWriter, ReadHalf, WriteHalf},
    sync::{mpsc, oneshot},
    task::JoinHandle,
};
use tokio_rustls::client::TlsStream;
use tokio_util::codec::FramedRead;

use crate::proto::{
    bytes::Bytes,
    command::Command,
    response::{Condition, Response, Status, Tag},
    rfc3501::capability as parse_capability,
};

use super::client::Config;
use super::codec::{ImapCodec, TaggedCommand};
use super::response_stream::ResponseStream;
use super::upgrade::upgrade;

const TAG_PREFIX: &str = "panotag";

type ExitSender = oneshot::Sender<()>;
type ExitListener = oneshot::Receiver<()>;
type GreetingSender = oneshot::Sender<()>;
type GreetingWaiter = oneshot::Receiver<()>;

/// Low-level client, can directly read from and write to the stream
/// without the additional type-safety of the higher-level state machine.
pub struct Inner<C> {
    config: Config,
    tag_number: AtomicU32,
    command_tx: mpsc::UnboundedSender<CommandContainer>,

    read_exit: ExitSender,
    read_handle: JoinHandle<ReadHalf<C>>,

    write_exit: ExitSender,
    write_handle: JoinHandle<WriteHalf<C>>,
    _write_tx: mpsc::UnboundedSender<TaggedCommand>,

    greeting_rx: Option<GreetingWaiter>,
}

#[derive(Debug)]
struct CommandContainer {
    tag: Tag,
    command: Command,
    channel: mpsc::UnboundedSender<Response>,
}

impl<C> Inner<C>
where
    C: AsyncRead + AsyncWrite + Unpin + Send + 'static,
{
    pub async fn new(c: C, config: Config) -> Result<Self> {
        let (command_tx, command_rx) = mpsc::unbounded_channel();

        // break the stream of bytes into a reader and a writer
        // the read_half represents the server->client connection
        // the write_half represents the client->server connection
        let (read_half, write_half) = split(c);

        // this channel is used to inform clients when we receive the
        // initial greeting from the server
        let (greeting_tx, greeting_rx) = oneshot::channel();

        // spawn the server->client loop
        let (read_exit, exit_rx) = oneshot::channel();
        let (write_tx, write_rx) = mpsc::unbounded_channel(); // TODO: maybe an arbitrary/configurable limit here would be better?
        let read_handle = tokio::spawn(read_loop(
            read_half,
            exit_rx,
            greeting_tx,
            write_tx.clone(),
            command_rx,
        ));

        // spawn the client->server loop
        let (write_exit, exit_rx) = oneshot::channel();
        let write_handle = tokio::spawn(write_loop(write_half, exit_rx, write_rx));

        let tag_number = AtomicU32::new(0);
        Ok(Inner {
            config,
            tag_number,
            command_tx,
            read_exit,
            read_handle,
            write_exit,
            write_handle,
            _write_tx: write_tx,
            greeting_rx: Some(greeting_rx),
        })
    }

    pub async fn execute(&mut self, command: Command) -> Result<ResponseStream> {
        let id = self.tag_number.fetch_add(1, Ordering::SeqCst);
        let tag = Tag(Bytes::from(format!("{}{}", TAG_PREFIX, id)));

        let (channel, rx) = mpsc::unbounded_channel();
        self.command_tx.send(CommandContainer {
            tag,
            command,
            channel,
        })?;

        let stream = ResponseStream { inner: rx };
        Ok(stream)
    }

    pub async fn has_capability(&mut self, cap: impl AsRef<str>) -> Result<bool> {
        // TODO: cache capabilities if needed?
        let cap = cap.as_ref().to_owned();
        let (_, cap) = parse_capability(Bytes::from(cap))?;

        let resp = self.execute(Command::Capability).await?;
        let (_, data) = resp.wait().await?;

        for resp in data {
            if let Response::Capabilities(caps) = resp {
                return Ok(caps.contains(&cap));
            }
            // debug!("cap: {:?}", resp);
        }

        Ok(false)
    }

    pub async fn upgrade(mut self) -> Result<Inner<TlsStream<C>>> {
        debug!("preparing to upgrade using STARTTLS");
        // TODO: check that this capability exists??
        // TODO: issue the STARTTLS command to the server
        let resp = self.execute(Command::Starttls).await?;
        dbg!(resp.wait().await?);
        debug!("received OK from server");

        // issue exit to the read loop and retrieve the read half
        let _ = self.read_exit.send(());
        let read_half = self.read_handle.await?;

        // issue exit to the write loop and retrieve the write half
        let _ = self.write_exit.send(());
        let write_half = self.write_handle.await?;

        // put the read half and write half back together
        let stream = read_half.unsplit(write_half);
        let tls_stream = upgrade(stream, &self.config.hostname).await?;

        Inner::new(tls_stream, self.config).await
    }

    pub async fn wait_for_greeting(&mut self) -> Result<()> {
        if let Some(greeting_rx) = self.greeting_rx.take() {
            greeting_rx.await?;
        }
        Ok(())
    }
}

// exit is a channel that will notify this loop when some external
// even requires this loop to stop (for example, TLS upgrade).
//
// when the loop exits, the read half of the stream will be returned
async fn read_loop<C>(
    stream: ReadHalf<C>,
    exit: ExitListener,
    greeting_tx: GreetingSender,
    write_tx: mpsc::UnboundedSender<TaggedCommand>,
    mut command_rx: mpsc::UnboundedReceiver<CommandContainer>,
) -> ReadHalf<C>
where
    C: AsyncRead,
{
    // this lets us "use up" the greeting sender
    let mut greeting_tx = Some(greeting_tx);

    let mut curr_cmd: Option<CommandContainer> = None;

    // set up framed communication
    let codec = ImapCodec::default();
    let mut framed = FramedRead::new(stream, codec);

    let exit = exit.fuse();
    pin_mut!(exit);
    loop {
        debug!("READ LOOP ITER");
        let next = framed.next().fuse();
        pin_mut!(next);

        // only listen for a new command if there isn't one already
        let mut cmd_fut = if let Some(ref cmd) = curr_cmd {
            debug!("current command: {:?}", cmd);
            // if there is one, just make a future that never resolves so it'll always pick
            // the other options in the select.
            future::pending().boxed().fuse()
        } else {
            command_rx.recv().boxed().fuse()
        };

        select! {
            // read a command from the command list
            command = cmd_fut => {
                if curr_cmd.is_none() {
                    if let Some(CommandContainer { ref tag, ref command, .. }) = command {
                        let _ = write_tx.send(TaggedCommand(tag.clone(), command.clone()));
                        // let cmd_str = format!("{} {:?}\r\n", tag, cmd);
                        // write_tx.send(cmd_str);
                    }
                    curr_cmd = command;
                    debug!("new command: {:?}", curr_cmd);
                }
            }

            // new message from the server
            resp = next => {
                let resp = match resp {
                    Some(Ok(v)) => v,
                    a => { error!("failed: {:?}", a); todo!("fuck"); },
                };
                trace!("S>>>C: {:?}", resp);

                // if this is the very first response, then it's a greeting
                if let Some(greeting_tx) = greeting_tx.take() {
                    greeting_tx.send(()).unwrap();
                }

                if let Response::Done(_) = resp {
                    // since this is the DONE message, clear curr_cmd so another one can be sent
                    if let Some(CommandContainer { channel, .. }) = curr_cmd.take() {
                        let _ = channel.send(resp);
                        // debug!("res0: {:?}", res);
                    }
                } else if let Response::Tagged(_, Condition { status: Status::Ok, ..}) = resp {
                    // clear curr_cmd so another one can be sent
                    if let Some(CommandContainer { channel, .. }) = curr_cmd.take() {
                        let _ = channel.send(resp);
                        // debug!("res0: {:?}", res);
                    }
                } else if let Some(CommandContainer { channel, .. }) = curr_cmd.as_mut() {
                    // we got a response from the server for this command, so send it over the
                    // channel

                    // debug!("sending {:?} to tag {}", resp, tag);
                    let _res = channel.send(resp);
                    // debug!("res1: {:?}", res);
                }
            }

            _ = exit => break,
        }
    }

    framed.into_inner()
}

async fn write_loop<C>(
    stream: WriteHalf<C>,
    exit_rx: ExitListener,
    mut command_rx: mpsc::UnboundedReceiver<TaggedCommand>,
) -> WriteHalf<C>
where
    C: AsyncWrite,
{
    // set up framed communication
    // let codec = ImapCodec::default();
    let mut stream = BufWriter::new(stream);
    // let mut framed = FramedWrite::new(stream, codec);

    let mut exit_rx = exit_rx.map_err(|_| ()).shared();
    loop {
        let command_fut = command_rx.recv().fuse();
        pin_mut!(command_fut);

        select! {
            command = command_fut => {
                // TODO: handle errors here
                if let Some(command) = command {
                    let cmd = format_bytes!(b"{} {}\r\n", &*command.0.0, command.1);
                    debug!("sending command: {:?}", String::from_utf8_lossy(&cmd));
                    let _ = stream.write_all(&cmd).await;
                    let _ = stream.flush().await;
                    // let _ = framed.send(&command).await;
                    // let _ = framed.flush().await;
                }
                // let _ = stream.write_all(line.as_bytes()).await;
                // let _ = stream.flush().await;
            }
            _ = exit_rx => break,
        }
    }

    // framed.into_inner()
    stream.into_inner()
}
