use std::io::{self};

use bytes::{BufMut, BytesMut};
use nom::Needed;
use tokio_util::codec::{Decoder, Encoder};

use crate::proto::{
    bytes::Bytes,
    command::Command,
    convert_error::convert_error,
    response::{Response, Tag},
    rfc3501::response as parse_response,
};

/// A codec that can be used for decoding `Response`s and encoding `Command`s.
#[derive(Default)]
pub struct ImapCodec {
    decode_need_message_bytes: usize,
}

impl<'a> Decoder for ImapCodec {
    type Item = Response;
    type Error = io::Error;

    fn decode(&mut self, buf: &mut BytesMut) -> Result<Option<Self::Item>, io::Error> {
        use nom::Err;

        if self.decode_need_message_bytes > buf.len() {
            return Ok(None);
        }

        let buf2 = buf.split();
        let buf3 = buf2.clone().freeze();
        debug!("going to parse a response since buffer len: {}", buf3.len());
        // trace!("buf: {:?}", buf3);
        let buf4: Bytes = buf3.clone().into();
        let buf4_len = buf4.len();
        let (response, len) = match parse_response(buf4) {
            Ok((remaining, response)) => (response, buf4_len - remaining.len()),
            Err(nom::Err::Incomplete(Needed::Size(min))) => {
                self.decode_need_message_bytes = min.get();
                return Ok(None);
            }
            Err(nom::Err::Incomplete(_)) => {
                return Ok(None);
            }
            Err(Err::Error(err)) | Err(Err::Failure(err)) => {
                let buf4 = buf3.clone().into();
                error!("failed to parse: {:?}", buf4);
                error!("code: {}", convert_error(buf4, err));
                return Err(io::Error::new(
                    io::ErrorKind::Other,
                    format!("error during parsing of {:?}", buf),
                ));
            }
        };

        info!("success, parsed as {:?}", response);
        buf.unsplit(buf2);
        let _ = buf.split_to(len);
        debug!("buf: {:?}", buf);

        self.decode_need_message_bytes = 0;
        Ok(Some(response))
    }
}

/// A command with its accompanying tag.
#[derive(Debug)]
pub struct TaggedCommand(pub Tag, pub Command);

impl<'a> Encoder<&'a TaggedCommand> for ImapCodec {
    type Error = io::Error;

    fn encode(&mut self, tagged_cmd: &TaggedCommand, dst: &mut BytesMut) -> Result<(), io::Error> {
        let tag = &*tagged_cmd.0 .0;
        let command = &tagged_cmd.1;

        dst.put(tag);
        dst.put_u8(b' ');

        // TODO: don't allocate here! use a stream writer
        let cmd_bytes = format_bytes!(b"{}", command);
        dst.extend_from_slice(cmd_bytes.as_slice());

        debug!("C>>>S: {:?}", dst);
        Ok(())
    }
}
