use std::ops::RangeInclusive;

use chrono::{DateTime, FixedOffset};

use super::bytes::Bytes;

pub type Atom = Bytes;

#[derive(Clone, Debug)]
pub struct Tag(pub Bytes);

#[derive(Debug)]
#[non_exhaustive]
pub enum Response {
    Capabilities(Vec<Capability>),
    Continue(ResponseText),
    Condition(Condition),
    Done(ResponseDone),
    MailboxData(MailboxData),
    Fetch(u32, Vec<MessageAttribute>),
    Expunge(u32),
    Fatal(Condition),
    Tagged(Tag, Condition),
}

#[derive(Debug)]
pub struct ResponseText {
    pub code: Option<ResponseCode>,
    pub info: Bytes,
}

#[derive(Debug)]
pub enum MessageAttribute {
    BodySection,
    BodyStructure,
    Envelope(Envelope),
    Flags(Vec<Flag>),
    InternalDate(DateTime<FixedOffset>),
    ModSeq(u64), // RFC 4551, section 3.3.2
    Rfc822(Option<String>),
    Rfc822Header(Option<String>),
    Rfc822Size(u32),
    Rfc822Text(Option<String>),
    Uid(u32),
}

#[derive(Debug)]
pub struct Envelope {
    pub date: Option<Bytes>,
    pub subject: Option<Bytes>,
    pub from: Option<Vec<Address>>,
    pub sender: Option<Vec<Address>>,
    pub reply_to: Option<Vec<Address>>,
    pub to: Option<Vec<Address>>,
    pub cc: Option<Vec<Address>>,
    pub bcc: Option<Vec<Address>>,
    pub in_reply_to: Option<Bytes>,
    pub message_id: Option<Bytes>,
}

#[derive(Debug)]
pub struct Address {
    pub name: Option<Bytes>,
    pub adl: Option<Bytes>,
    pub mailbox: Option<Bytes>,
    pub host: Option<Bytes>,
}

#[derive(Debug)]
pub struct ResponseDone {
    pub tag: Tag,
    pub status: Status,
    pub code: Option<ResponseCode>,
    pub info: Option<Bytes>,
}

#[derive(Debug)]
pub struct Condition {
    pub status: Status,
    pub code: Option<ResponseCode>,
    pub info: Bytes,
}

#[derive(Debug)]
pub enum Status {
    Ok,
    No,
    Bad,
    PreAuth,
    Bye,
}

#[derive(Debug)]
#[non_exhaustive]
pub enum ResponseCode {
    Alert,
    BadCharset(Option<Vec<Bytes>>),
    Capabilities(Vec<Capability>),
    HighestModSeq(u64), // RFC 4551, section 3.1.1
    Parse,
    PermanentFlags(Vec<Bytes>),
    ReadOnly,
    ReadWrite,
    TryCreate,
    UidNext(u32),
    UidValidity(u32),
    Unseen(u32),
    AppendUid(u32, Vec<UidSetMember>),
    CopyUid(u32, Vec<UidSetMember>, Vec<UidSetMember>),
    UidNotSticky,
    Other(Bytes, Option<Bytes>),
}

#[derive(Debug)]
pub enum UidSetMember {
    UidRange(RangeInclusive<u32>),
    Uid(u32),
}

#[derive(Debug, PartialEq, Eq)]
pub enum Capability {
    Imap4rev1,
    Auth(Atom),
    Atom(Atom),
}

#[derive(Debug)]
pub enum MailboxData {
    Flags(Vec<Flag>),
    List(MailboxList),
    Lsub,
    Search(Vec<u32>),
    Status,
    Exists(u32),
    Recent(u32),
}

#[derive(Debug)]
pub enum Mailbox {
    Inbox,
    Name(Bytes),
}

#[derive(Debug)]
pub enum Flag {
    Answered,
    Flagged,
    Deleted,
    Seen,
    Draft,
    Recent,
    Keyword(Atom),
    Extension(Atom),
}

#[derive(Debug)]
pub struct MailboxList {
    pub flags: Vec<MailboxListFlag>,
    pub delimiter: Option<u8>,
    pub mailbox: Mailbox,
}

#[derive(Debug)]
pub enum MailboxListFlag {
    NoInferiors,
    NoSelect,
    Marked,
    Unmarked,
    Extension(Atom),
}
