use std::pin::Pin;
use std::task::{Context, Poll};

use anyhow::Result;
use futures::stream::Stream;
use tokio::sync::mpsc;

use crate::proto::response::{Response, ResponseDone};

/// A series of responses that follow an
pub struct ResponseStream {
    pub(crate) inner: mpsc::UnboundedReceiver<Response>,
}

impl ResponseStream {
    /// Retrieves just the DONE item in the stream, discarding the rest
    pub async fn done(mut self) -> Result<Option<ResponseDone>> {
        while let Some(resp) = self.inner.recv().await {
            if let Response::Done(done) = resp {
                return Ok(Some(done));
            }
        }
        Ok(None)
    }

    /// Waits for the entire stream to finish, returning the DONE status and the
    /// stream
    pub async fn wait(mut self) -> Result<(Option<ResponseDone>, Vec<Response>)> {
        let mut done = None;
        let mut vec = Vec::new();
        while let Some(resp) = self.inner.recv().await {
            if let Response::Done(d) = resp {
                done = Some(d);
                break;
            } else {
                vec.push(resp);
            }
        }
        Ok((done, vec))
    }
}

impl Stream for ResponseStream {
    type Item = Response;
    fn poll_next(mut self: Pin<&mut Self>, cx: &mut Context) -> Poll<Option<Self::Item>> {
        self.inner.poll_recv(cx)
    }
}
