use std::io;

use bytes::{Buf, BufMut, BytesMut};
use nom::Needed;
use tokio_util::codec::{Decoder, Encoder};

use crate::proto::{
    command::Command,
    response::{Response, Tag},
    rfc3501::response as parse_response,
};

/// A codec that can be used for decoding `Response`s and encoding `Command`s.
#[derive(Default)]
pub struct ImapCodec {
    decode_need_message_bytes: usize,
}

impl<'a> Decoder for ImapCodec {
    type Item = Response;
    type Error = io::Error;
    fn decode(&mut self, buf: &mut BytesMut) -> Result<Option<Self::Item>, io::Error> {
        if self.decode_need_message_bytes > buf.len() {
            return Ok(None);
        }

        let buf2 = buf.split();
        let buf3 = buf2.clone().freeze();
        let (response, len) = match parse_response(buf3.clone().into()) {
            Ok((remaining, response)) => (response, buf.len() - remaining.len()),
            Err(nom::Err::Incomplete(Needed::Size(min))) => {
                self.decode_need_message_bytes = min.get();
                return Ok(None);
            }
            Err(nom::Err::Incomplete(_)) => {
                return Ok(None);
            }
            Err(nom::Err::Error(nom::error::Error { code, .. }))
            | Err(nom::Err::Failure(nom::error::Error { code, .. })) => {
                return Err(io::Error::new(
                    io::ErrorKind::Other,
                    format!("{:?} during parsing of {:?}", code, buf),
                ));
            }
        };

        buf.unsplit(buf2);
        buf.advance(len);

        self.decode_need_message_bytes = 0;
        Ok(Some(response))
    }
}

/// A command with its accompanying tag.
pub struct TaggedCommand(pub Tag, pub Command);

impl<'a> Encoder<&'a TaggedCommand> for ImapCodec {
    type Error = io::Error;
    fn encode(&mut self, tagged_cmd: &TaggedCommand, dst: &mut BytesMut) -> Result<(), io::Error> {
        let tag = &tagged_cmd.0;
        let _command = &tagged_cmd.1;

        dst.put(&*tag.0);
        dst.put_u8(b' ');
        // TODO: write command
        dst.put_slice(b"\r\n");
        Ok(())
    }
}
